%%%  src/INet/Utils/List.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{List Utilities}

\begin{code}
{-# LANGUAGE FlexibleContexts #-}
module INet.Utils.List where

import Control.Monad.State.Class
import Data.Function (on)
import Data.List (groupBy)
\end{code}

Since I couldn't find an appropriate queue interface,
here is an auxiliary function that traverses a list only once,
combining |Eq|-based index lookup with append for new elements.
New elements need to be appended at the end in order to keep the index
of the other elements stable.
(An experiment with explicit length enabling reverse indices might be interesting.)
\edcomm{WK}{In monadic context, this could be implemented using destructive list update.
How to get a compiler to do this?}

\begin{code}
-- |Left (gb i)| for ``found |y| satisfying |f x| at |i|'';
-- |Right (gc x i)| for ``inserted (appended) |a| at |i|''.
checkElem :: MonadState [a] m => (x -> Int -> c) -> (x -> a -> Maybe (Int -> b)) -> x -> a -> m (Either b c)
checkElem gc f x a = state (h 0 id)
  where
    h i acc [] = (Right (gc x i), acc [a])
    h i acc ys@(y : ys') = case f x y of
      Just gb -> (Left (gb i), acc ys)
      Nothing -> h (succ i) (acc . (y :)) ys'
\end{code}

\begin{code}
groupByOnFst :: (a -> a -> Bool) -> [(a,b)] -> [(a, [b])]
groupByOnFst rel = map h . groupBy (rel `on` fst)
  where
    h ((a, b) : ps) =  (a, b : map snd ps)
    h [] = error $ "IMPOSSIBLE: groupByOnFst.h []"
\end{code}

|removeDuplicates| removes \emph{both} occurrences of all elements occurring twice.
(Actually it removes all occurring of all elements occurring an even number of times.)
\begin{code}
removeDuplicates :: (Eq a) => [a] -> [a]
removeDuplicates [] = []
removeDuplicates (x : xs) = if x `elem` xs  then removeDuplicates $ removeOnce x xs
                                            else x : removeDuplicates xs
removeOnce :: (Eq a) => a -> [a] -> [a]
removeOnce x [] = []
removeOnce x (y : ys) = if x == y then ys else y : removeOnce x ys
\end{code}
