%%%  src/INet/Description/Dot.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{GraphViz Representation of Net Descriptions}

\begin{code}
{-# LANGUAGE PatternGuards, FlexibleInstances #-}
module INet.Description.Dot where

import INet.Description
import INet.Description.Check (ptdRel)

import INet.Utils.Dot

import qualified INet.Utils.Vector as V
\end{code}


\begin{code}
srcName = "Source"
trgName = "Target"
nodeName ni = 'N' : show ni
portName1 i = 'p' : show i
portName2 i = 'p' : show i
portName3 i = 'p' : show i
\end{code}


\begin{code}
ptdString :: Char -> (NI -> String) -> PortTargetDescription -> String
ptdString sep nodeName ptd = case ptd of
  SourcePort pi1 -> srcName ++ sep : portName1 pi1
  TargetPort pi2 -> trgName ++ sep : portName2 pi2
  InternalPort ni pi3 -> nodeName ni ++ sep : portName3 pi3
\end{code}

\begin{code}
ptdName :: PortTargetDescription -> String
ptdName = ptdString ':' (show . nodeName)
\end{code}


\begin{code}

shapeRecord = ("shape","record")
shapeConn = ("shape","circle")

showsPortId portName i s = '<' : portName i ++ '>' :  shows i (' ' : s)

showsPortIds portName [] s = s
showsPortIds portName [i] s = showsPortId portName i s
showsPortIds portName (i : is) s = showsPortId portName i $ foldr (\ j r -> '|' :  showsPortId portName j r) s is

showsPortIds' portName [i] = showsPortId portName i
showsPortIds' portName is = brace $ showsPortIds portName is

brace ss = ('{' :) . ss . ('}' :)

defaultDotLabel portName label (i,j) = defaultDotLabel1 portName label [i .. j]

defaultDotLabel1 portName label (i : is) = defaultDotLabel2 portName label [i] is
defaultDotLabel1 portName label [] = error $ "defaultDotLabel1 " ++ label ++ " []"

defaultDotLabel2 portName label is1 is2 = brace
  ( (if null is1 then id else showsPortIds' portName is1 . ('|' :))
  . (label ++) .
    (if null is2 then id else ('|' :) . showsPortIds' portName is2)
  ) ""

dotNode :: String -> String -> Attrs -> Stmt
dotNode name label attrs
  = Node name
  $ shapeRecord
  : ("height","0.1")
  : ("label", label)
  : attrs

dotConnection :: (PortTargetDescription, PortTargetDescription) -> [Stmt] -> [Stmt]
dotConnection (ptd1, ptd2) s
  = Node connNode [("shape","circle"),("height","0.1"),("width","0.1"),("label",""),("fixedsize","true")]
  : UndirEdge (ptdName ptd1) connNode []
  : UndirEdge connNode (ptdName ptd2) []
  : s
  where connNode = (ptdString '_' nodeName ptd1 ++ "__" ++ ptdString '_' nodeName ptd2)
\end{code}


\begin{code}
dotNetDescription
  :: (nLab -> String)
  -> (nLab -> Maybe String)
  -> String
  -> NetDescription nLab
  -> DotGraph
dotNetDescription showNLab dotNLab name nd = DotGraph Graph name
  $ dotNode srcName (defaultDotLabel2 portName1 srcName [] [0 .. pred (V.length $ source nd)]) []
  : dotNode trgName (defaultDotLabel2 portName2 trgName [0 .. pred (V.length $ target nd)] []) []
  : flip (foldr (\ (ni, d) -> (:) $ dotNode (nodeName ni)
              (case dotNLab $ nLab d of
                 Nothing -> defaultDotLabel portName3 (showNLab $ nLab d) (V.bounds $ portDescriptions d)
                 Just s -> s
              ) [])) (V.assocs $ nodes nd)
  (foldr dotConnection [] $ ptdRel nd)
\end{code}

