%%%  src/INet/Description/JSON.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{JSON Representation of Net Descriptions}

\begin{code}
{-# LANGUAGE PatternGuards #-}
module INet.Description.JSON where

import INet.Description
import INet.Rule

import INet.Utils.Vector (Vector)
import qualified INet.Utils.Vector as V

import Text.JSON
\end{code}


\begin{code}
showVector :: (JSON a) => Vector a -> JSValue
showVector = showJSON . V.toList

readVector :: (JSON a) => JSValue -> Result (Vector a)
readVector = fmap V.fromList . readJSON
\end{code}

\begin{code}
instance (JSON a) => JSON (Vector a) where
  readJSON = readVector
  showJSON = showVector
\end{code}


\begin{code}
showPortTargetDescription :: PortTargetDescription -> JSValue
showPortTargetDescription ptd = JSObject . toJSObject $ case ptd of
  SourcePort pi1 -> [("SourcePort", showJSON pi1)]
  TargetPort pi2 -> [("TargetPort", showJSON pi2)]
  InternalPort ni pi3 -> [("InternalPort", JSArray [showJSON ni, showJSON pi3])]

readPortTargetDescription :: JSValue -> Result PortTargetDescription
readPortTargetDescription (JSObject obj) = case fromJSObject obj of
  [(s, v)]
    | s `elem` ["SourcePort", "src"] -> case readJSON v of
       Ok pi1 -> Ok (SourcePort pi1)
       Error e -> Error $ "PortTargetDescription: SourcePort: " ++ e
    | s `elem` ["TargetPort", "trg"] -> case readJSON v of
       Ok pi2 -> Ok (TargetPort pi2)
       Error e -> Error $ "PortTargetDescription: TargetPort: " ++ e
    | s `elem` ["InternalPort", "int"] -> case v of
       JSArray [vni, vpi3] -> case readJSON vni of
         Ok ni -> case  readJSON vpi3 of
           Ok pi3 -> Ok (InternalPort ni pi3)
           Error e -> Error $ "PortTargetDescription: InternalPort2: " ++ e
         Error e -> Error $ "PortTargetDescription: InternalPort1: " ++ e
       _ -> Error "PortTargetDescription: InternalPort: 2-element array expected"
  _ -> Error "PortTargetDescription: one-element object expected"
readPortTargetDescription _ = Error "PortTargetDescription expected"
\end{code}


\begin{code}
instance JSON PortTargetDescription where
  readJSON = readPortTargetDescription
  showJSON = showPortTargetDescription
\end{code}


\begin{code}
showNetDescription :: (JSON nLab) => NetDescription nLab -> JSValue
showNetDescription nd = JSObject $ toJSObject
  [("source", showJSON $ source nd)
  ,("target", showJSON $ target nd)
  ,("nodes", showJSON $ nodes nd)
  ]

readNetDescription :: (JSON nLab) => JSValue -> Result (NetDescription nLab)
readNetDescription (JSObject nd)
  | [("source", vsi),("target",vti),("nodes",vns)] <- fromJSObject nd
  , Ok si <- readJSON vsi
  , Ok ti <- readJSON vti
  , Ok ns <- readJSON vns
  = Ok $ NetDescription
      { source = si
      , target = ti
      , nodes = ns
      }
readNetDescription _ = Error "readNetDescription"
\end{code}

\begin{code}
instance  (JSON nLab) => JSON (NetDescription nLab) where
  readJSON = readNetDescription
  showJSON = showNetDescription
\end{code}


\begin{code}
showNodeDescription :: (JSON nLab) => NodeDescription nLab -> JSValue
showNodeDescription nd = JSArray
  [ showJSON (nLab nd)
  , showJSON $ portDescriptions nd
  ]

readNodeDescription :: (JSON nLab) => JSValue -> Result (NodeDescription nLab)
readNodeDescription (JSArray [vnd, vpds]) = case readJSON vnd of
  Ok label -> case readJSON vpds of
    Ok pds -> Ok $ NodeDescription
      { nLab = label
      , portDescriptions = pds
      }
    Error e -> Error $ "readNodeDescription: Port descriptions expected: " ++ e
  Error e -> Error $ "readNodeDescription: Label expected: " ++ e
readNodeDescription _ = Error "readNodeDescription: 2-element array expected"
\end{code}

\begin{code}
instance  (JSON nLab) => JSON (NodeDescription nLab) where
  readJSON = readNodeDescription
  showJSON = showNodeDescription
\end{code}


\begin{code}
showRule :: (JSON nLab) => Rule nLab -> JSValue
showRule = showJSON . fromRule

readRule :: (JSON nLab) => JSValue -> Result (Rule nLab)
readRule = fmap (uncurry Rule) . readJSON
\end{code}


\begin{code}
instance  (JSON nLab) => JSON (Rule nLab) where
  readJSON = readRule
  showJSON = showRule
\end{code}


\begin{code}


\end{code}

