%%%  src/INet/InetsFile/Parser.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{|INet.InetsFile.Parser| --- Parsing \texttt{.inet} Files}

\edcomm{WK}{For testing:
\begin{spec}
either print print $ runParser (pStatementExpression <* eof) [] "@" "counter = counter +1"

do bs <- mapM (\ n -> parseInetsFile' (exampleDir ++ n ++ ".inet")) exampleNames; mapM_ (putStrLn . show) (zip exampleNames bs)

 do Right cu <- parseInetsFile (exampleDir ++ "examples/Ackerman.inet") ; mapM_ print (map ruleDefAgents (cuRuleDefs cu))
\end{spec}
}%edcomm

\begin{ModuleHead}
\begin{code}
{-# LANGUAGE FlexibleContexts, NoMonomorphismRestriction #-}

module INet.InetsFile.Parser where

import INet.InetsFile.Abstract

import Text.Parsec
import Text.Parsec.Expr
import Text.Parsec.Token (GenLanguageDef(..))
import qualified Text.Parsec.Token as P

import Data.Map (Map)
import qualified Data.Map as Map
import qualified Data.Char as Char

import Control.Applicative ( (<$>), (<*>), (*>), (<*) )
import Control.Monad (guard, mplus)
\end{code}
\end{ModuleHead}

\begin{code}
parseInetsFile :: FilePath -> IO (Either ParseError CompilationUnit)
parseInetsFile path = do
   s <- readFile path
   return $ runParser pCompilationUnit [Map.empty] path s
\end{code}

\begin{code}
parseInetsFile'  :: FilePath -> IO Bool
parseInetsFile' path = do
   putStrLn $ "Parsing " ++ path
   e <- parseInetsFile path
   case e of
     Right r -> putStrLn "... success" >> return True -- |putStrLn (show r)|
     Left e -> print e >> return False
\end{code}

\begin{code}
exampleDir = "../../../svn/inets/test/"

exampleNames = map ("examples/" ++)
  [ "Ackerman"
  , "ArithExpression"
  , "factorial"
  , "fibonacci"
  , "hanoi"
  , "nesttest1"
  , "nesttest2"
  , "nonTerminate"
  , "quicktest"
  , "simple_generic"
  , "sort"
  , "triple"
  , "yale"
  , "monad/list"
  , "monad/list_generic"
  , "monad/maybe"
  , "monad/maybe_generic"
  , "monad/writer"
  , "monad/writer_generic"
  ] ++ map ("lib/" ++)
  [ "Copier"
  , "Eraser"
  , "List"
  , "Num"
  ]

\end{code}

\begin{code}
primTypes :: [(String, PrimitiveType)]
primTypes =
  [("int", Tint)
  ,("bool", Tbool)
  ,("float", Tfloat)
  ,("String", TString)
  ,("char", Tchar)
  ,("Agent", TAgent)
  ,("Agents", TAgents)
  ]
primTypeNames :: [String]
primTypeNames = map fst primTypes
\end{code}

\begin{code}
inetLanguageDef :: Stream s m Char => GenLanguageDef s u m
inetLanguageDef = LanguageDef
  { commentStart = "/*"
  , commentEnd = "*/"
  , commentLine = "//" -- \edcomm{WK}{Not recognised as first line???}
  , nestedComments = True
  , identStart = letter <|> char '_'
  , identLetter = alphaNum <|> char '_'
  , opStart = opChar
  , opLetter = opChar
  , reservedNames =  [ "module", "import", "open", "as", "export", "print", "printNet"
                     , "end", "if", "else", "read", "readc", "readln", "return"
                     , "true", "false", "Net"] ++ primTypeNames

  , reservedOpNames =  [ "><", "~", "=", "=>"
                       , "||", "&&", "!=", "==", "<", ">", "<=", ">="
                       , "+", "-", "*", "/", "%", "!" ]

  , caseSensitive = True
  }
  where
    opChar = oneOf ":!#$%&*+./<=>?@\\^|-~"
\end{code}

\begin{code}
lexer :: Stream s m Char => P.GenTokenParser s u m
lexer = P.makeTokenParser inetLanguageDef
\end{code}

\begin{code}
reserved = P.reserved lexer
reservedOp = P.reservedOp lexer
parens = P.parens lexer
braces = P.braces lexer
comma = P.comma lexer
semi = P.semi lexer
\end{code}

\begin{code}
parens' :: (Stream s m Char) => ParsecT s u m [a] -> ParsecT s u m [a]
parens' p = try (parens p) <|> return []
\end{code}

\begin{code}
(<+>) :: ParsecT s u m a -> ParsecT s u m b -> ParsecT s u m (Either a b)
pa <+> pb = fmap Left pa <|> fmap Right pb
\end{code}

\begin{code}
pCompilationUnit = (many1 pModule <+> pModuleComponents) <* eof

pModuleComponents = many pModuleComponent
\end{code}

\begin{code}
pName = P.identifier lexer
\end{code}

\begin{code}
pModule :: Stream s m Char => ParsecT s Scopes m Module
pModule = do
  reserved "module"
  name <- pName
  body <- braces pModuleComponents
  return $ Module name body
\end{code}

\begin{code}
pModuleComponent
  =    fmap MCImportStmt (reserved "import" *> sepBy1 pName (P.dot lexer) <* semi)
  <|>  try (fmap MCRule pRule)
  <|>  try (fmap MCNet pNet)
  <|>  fmap MCStatements pStatements
\end{code}

\begin{code}
type Scope = Map String PrimitiveType
type Scopes = [Scope]
\end{code}

\begin{code}
withLocalScope :: (Monad m) => Scope -> ParsecT s Scopes m a -> ParsecT s Scopes m a
withLocalScope init p = do
  modifyState (init :)
  a <- p
  modifyState tail
  return a

addToLocalScope :: (Monad m) => String -> PrimitiveType -> ParsecT s Scopes m ()
addToLocalScope name ty = modifyState (\ (sc : scs) -> Map.insert name ty sc : scs)

addParamOrArg (Param ty (Agent name params)) = do
    addToLocalScope name ty
    mapM_ addParamOrArg params
addParamOrArg (ParamArray name k) = return () -- arrays not yet supported
addParamOrArg (Arg (EVar (Agent name _))) = addToLocalScope name TAgent
addParamOrArg (Arg (EAgent (Agent name _))) = addToLocalScope name TAgent
addParamOrArg (Arg exprOrTerm) = fail $ "addParamOrArg " ++ show exprOrTerm

scopeLookup :: (Monad m) => String -> ParsecT s Scopes m (Maybe PrimitiveType)
scopeLookup name = do
  scs <- getState
  return (foldr mplus Nothing $ map (Map.lookup name) scs)
\end{code}

\begin{code}
pRule :: Stream s m Char => ParsecT s Scopes m Rule
pRule  =    fmap AgentRule pRuleDef
       --  ``or'' |fmap NamedRule pNamedRuleDef|

pRuleDef :: Stream s m Char => ParsecT s Scopes m RuleDef
pRuleDef = withLocalScope Map.empty $ do
  a@(Agent name params) <- pAgent
  addToLocalScope name TAgent
  mapM_ addParamOrArg params
  reservedOp "><"
  pre <- many pStatementOrDec
  rhss <- many1 (try pRHS)
  return $ RuleDef a pre rhss
\end{code}

\begin{code}
pBlock  :: Stream s m Char
        => ([StatementOrDec] -> a -> [StatementOrDec] -> b)
        -> ParsecT s Scopes m a -> ParsecT s Scopes m b
pBlock wrap p = braces (do
          pre <- many pStatementOrDec
          r <- p
          post <- many pStatementOrDec
          return $ wrap pre r post
      ) <|> do
          r <- p
          return $ wrap [] r []
\end{code}

\begin{code}
pRHS :: Stream s m Char => ParsecT s Scopes m RHS
pRHS  = pBlock RhsBlock pRuleRHS
\end{code}

\begin{code}
pRuleBody :: Stream s m Char => ParsecT s Scopes m RuleBody
pRuleBody = fmap RBeq pEquationList
  <|> fmap RBif pIfStatement
  <|> (semi *> return RBskip)
\end{code}

\begin{code}
pRuleRHS :: Stream s m Char => ParsecT s Scopes m RuleRHS
pRuleRHS = RuleRHS <$> (pAgent <* (optional $ reservedOp "=>"))
                   <*> many pStatementOrDec <*> pRuleBody
\end{code}

\begin{code}
pIfStatement :: Stream s m Char => ParsecT s Scopes m IfStatement
pIfStatement = do
  reserved "if"
  b <- parens pExpression
  t <- pIfBlock
  reserved "else"
  e <- fmap Left pIfBlock <|> fmap Right pIfStatement
  return $ IF b t e
\end{code}

\begin{code}
pIfBlock :: Stream s m Char => ParsecT s Scopes m IfBlock
pIfBlock = pBlock IfBlock pEquationList
\end{code}

\begin{code}
commaSemiList :: Stream s m Char => ParsecT s u m a -> ParsecT s u m [a]
commaSemiList p = sepBy1 p comma <* semi
\end{code}

\begin{code}
pEquationList :: Stream s m Char => ParsecT s Scopes m [Equation]
pEquationList = commaSemiList pEquation
\end{code}

\begin{code}
pNet :: Stream s m Char => ParsecT s Scopes m Net
pNet = try (withLocalScope Map.empty
  (NamedNet  <$> pName
             <*> parens pParamOrArgs
             <*> braces (many1 pNetBody)
  )) <|> fmap UnNamedNet pUnNamedNet
\end{code}

\begin{code}
pNetBody :: Stream s m Char => ParsecT s Scopes m (Either StatementOrDec [NetDef])
pNetBody = try pStatementOrDec <+> commaSemiList pNetDef
\end{code}

\begin{code}
pNetDef :: Stream s m Char => ParsecT s Scopes m NetDef
pNetDef = try pEquation <+> pNetInst
\end{code}

\begin{code}
pNetInst :: Stream s m Char => ParsecT s Scopes m NetInst
pNetInst = NetInst <$> pName <*> parens pParamOrArgs
\end{code}

\begin{code}
pUnNamedNet :: Stream s m Char => ParsecT s Scopes m [NetDef]
pUnNamedNet = commaSemiList pNetDef
\end{code}

\begin{code}
pEquation :: Stream s m Char => ParsecT s Scopes m Equation
pEquation = do
  t1 <- pTerm
  reservedOp "~"
  t2 <- pTerm
  return $ Equation t1 t2
\end{code}

\begin{code}
pTerm :: Stream s m Char => ParsecT s Scopes m Term
pTerm = try (fmap TermAgent pAgent)
  <|> fmap TermVar pVarAgent
\end{code}

\begin{code}
pAgent :: Stream s m Char => ParsecT s Scopes m Agent
pAgent = Agent <$> pAgentName <*> parens' pParamOrArgs

pAgentName :: Stream s m Char => ParsecT s Scopes m Name
pAgentName = do
  s@(c : _) <- pName
  guard (Char.isUpper c)
  return s

pVarAgentName :: Stream s m Char => ParsecT s Scopes m Name
pVarAgentName = do
  s@(c : _) <- pName
  guard (Char.isLower c)
  return s
\end{code}

\begin{code}
pVarAgent :: Stream s m Char => ParsecT s Scopes m Agent
pVarAgent = Agent <$> pVarAgentName <*> parens' pParamOrArgs
\end{code}

\begin{code}
pParamOrArgs :: Stream s m Char => ParsecT s Scopes m ParamOrArgs
pParamOrArgs = sepBy pParamOrArg comma
\end{code}

\begin{code}
pParamOrArg :: Stream s m Char => ParsecT s Scopes m ParamOrArg
pParamOrArg = try pParam <|> fmap Arg pExpressionOrTerm
\end{code}

\begin{code}
pParam :: Stream s m Char => ParsecT s Scopes m ParamOrArg
pParam = do
  ty <- pPrimitiveType
  v@(Agent name params) <- pVarAgent
  addToLocalScope name ty
  mapM_ addParamOrArg params
  return $ Param ty v   -- other alternatives in |Param()| left out
\end{code}

\begin{code}
inScope :: Stream s m Char => ParsecT s Scopes m Agent -> ParsecT s Scopes m Agent
inScope p = do
  a@(Agent name params) <- p
  mty <- scopeLookup name
  case mty of
    Nothing     -> return a
    Just TAgent -> return a
    _ -> fail $ "Scope |- name : " ++ show mty
\end{code}

\edcomm{WK}{|pExpressionOrTerm| needs to use the |Scopes|,
apparently to distinguish variable agent names from expression variables!}
\begin{code}
pExpressionOrTerm :: Stream s m Char => ParsecT s Scopes m ExpressionOrTerm
pExpressionOrTerm = try (fmap EAgent $ inScope pAgent)
  <|> try (fmap EVar $ inScope pVarAgent)
  <|> fmap Eexpr pExpression
\end{code}

\begin{code}
pConstants :: Stream s m Char => (String -> ParsecT s u m x) -> [(String, a)] -> ParsecT s u m a
pConstants mkP ps = foldr1 (<|>) $ map (\ (s, t) -> mkP s *> return t) ps
\end{code}

\begin{code}
pPrimitiveType :: Stream s m Char => ParsecT s Scopes m PrimitiveType
pPrimitiveType = pConstants reserved primTypes
\end{code}

\begin{code}
pStatements :: Stream s m Char => ParsecT s Scopes m [StatementOrDec]
pStatements = many1 pStatementOrDec
\end{code}

\begin{code}
pStatementOrDec :: Stream s m Char => ParsecT s Scopes m StatementOrDec
pStatementOrDec = (try pReturnStatement
  <|> try pStatementExpression
  <|> try pDeclaration
  <|> pPrintStatement
  ) <* semi
\end{code}

\begin{code}
pReturnStatement :: Stream s m Char => ParsecT s Scopes m StatementOrDec
pReturnStatement = reserved "return" *>
                   (Return <$> (fmap Just pTerm <|> return Nothing))
\end{code}

\begin{code}
pStatementExpression :: Stream s m Char => ParsecT s Scopes m StatementOrDec
pStatementExpression = StatementExpression <$> (pETerm <* reservedOp "=") <*> pExpression
\end{code}

\begin{code}
pDeclaration :: Stream s m Char => ParsecT s Scopes m StatementOrDec
pDeclaration = do
  ty <- pPrimitiveType
  assigns <- sepBy1 pAssignment comma
  mapM_ (\ (Assignment name _) -> addToLocalScope name ty) assigns
  return $ Declaration ty assigns
\end{code}

\begin{code}
pAssignment :: Stream s m Char => ParsecT s Scopes m Assignment
pAssignment = Assignment <$> pName <*> (fmap Just pExpression <|> return Nothing)
\end{code}

\begin{code}
pPrintStatement :: Stream s m Char => ParsecT s Scopes m StatementOrDec
pPrintStatement = (reserved "print" *> (Print <$> sepBy1 pExpOrString comma))
  <|> (reserved "printNet" *> (PrintNet <$> sepBy1 pTerm comma))
\end{code}

\begin{code}
pExpOrString :: Stream s m Char => ParsecT s Scopes m ExpOrString
pExpOrString = try pExpression <+> P.stringLiteral lexer
\end{code}

\begin{code}
pExpression :: Stream s m Char => ParsecT s Scopes m Expression
pExpression = buildExpressionParser table pETermChain
  where
    table = [[Infix (reservedOp "&&" *> return And)  AssocRight]
            ,[Infix (reservedOp "||" *> return Or)   AssocRight]]
\end{code}

\begin{code}
pETermChain :: Stream s m Char => ParsecT s Scopes m Expression
pETermChain = try (Atom <$> pETerm <*> many ((,) <$> pPred <*> pETerm))
\end{code}

\begin{code}
pPred :: Stream s m Char => ParsecT s Scopes m Pred
pPred = pConstants reservedOp infixPreds

infixPreds =
  [("==", Peq)
  ,("!=", Pneq)
  ,("<", Plt)
  ,(">", Pgt)
  ,("<=", Pleq)
  ,(">=", Pgeq)
  ]
\end{code}

\begin{code}
pLit :: Stream s m Char => ParsecT s Scopes m ETerm
pLit = fmap BoolLit (pConstants reserved [("false", False), ("true", True)])
  <|>  fmap IntLit (P.integer lexer)
  <|>  fmap StringLit (P.stringLiteral lexer)
\end{code}

\begin{code}
pETerm :: Stream s m Char => ParsecT s Scopes m ETerm
pETerm = buildExpressionParser table (pLit <|> fmap Var pName <|> parens pETerm)
  where
    table =
      [[  Prefix (reservedOp "!" *> return (Un Unot))
       ,  Prefix (reservedOp "-" *> return (Un Uminus))
       ,  Prefix (reservedOp "+" *> return id)
       ]
      ,[  Infix (reservedOp "*" *> return (Bin Bmult)) AssocLeft
       ,  Infix (reservedOp "/" *> return (Bin Bdiv)) AssocLeft
       ,  Infix (reservedOp "%" *> return (Bin Bmod)) AssocLeft
       ]
      ,[  Infix (reservedOp "+" *> return (Bin Bplus)) AssocLeft
       ,  Infix (reservedOp "-" *> return (Bin Bminus)) AssocLeft
       ]
      ]
\end{code}


% Local Variables:
% folded-file: t
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% fold-internal-margins: 0
% end:
