%%%  src/INet/InetsFile/ToDescription.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{|INet.InetsFile.ToDescription| --- Conversion to |NetDescription|s}

\edcomm{WK}{For testing:
\begin{spec}
:l "INet/InetsFile/ToDescription.lhs" "INet/InetsFile/Parser.lhs"
:m + INet.InetsFile.Parser

do Right cu <- parseInetsFile (exampleDir ++ "examples/Ackerman.inet") ; print (snd $ fst $ cuINetLang cu)

do Right cu <- parseInetsFile (exampleDir ++ "examples/factorial.inet") ; print (fst $ fst $ cuINetLang cu)

\end{spec}
}%edcomm

\begin{ModuleHead}
\begin{code}
{-# LANGUAGE Rank2Types #-}
module INet.InetsFile.ToDescription where

import INet.InetsFile.Abstract

import INet.Description hiding (ruleRHS)
import INet.Description.Check (checkNetDescription)
import INet.Description.Utils (collectArityFromNetDescription)
import INet.Description.Show () -- instances only

import INet.Polarity

import INet.PTerm

import INet.Utils.Vector  (Vector) -- |, (!?), atErr, bounds, (!))|
import qualified INet.Utils.Vector as V

import Data.Map (Map)
import qualified Data.Map as Map
import Data.List (nub, intersperse)

import Control.Monad (mplus)
import Control.Arrow (first, second)

-- |import Debug.Trace|
\end{code}
\end{ModuleHead}

\begin{code}
data Value
  = ValInt Integer
  | ValFloat Double
  | ValBool Bool
  | ValChar Char
  | ValString String
  deriving (Eq, Ord, Show)

valTy :: Value -> PrimitiveType
valTy (ValInt _) = Tint
valTy (ValFloat _) = Tfloat
valTy (ValBool _) = Tbool
valTy (ValChar _) = Tchar
valTy (ValString _) = TString
\end{code}

%{{{ valBinOp
\begin{code}
valBinOp :: BinOp -> Value -> Value -> Either String Value
valBinOp Bplus (ValInt i1) (ValInt i2) = Right $ ValInt $ i1 + i2
valBinOp Bplus (ValFloat d1) (ValFloat d2) = Right $ ValFloat $ d1 + d2
valBinOp Bplus (ValInt i1) (ValFloat d2) = Right $ ValFloat $ fromInteger i1 + d2
valBinOp Bplus (ValFloat d1) (ValInt i2) = Right $ ValFloat $ d1 + fromInteger i2
valBinOp Bplus (ValString s1) (ValString s2) = Right $ ValString $ s1 ++ s2
valBinOp Bplus (ValChar c1) (ValString s2) = Right $ ValString $ c1 : s2
valBinOp Bplus (ValString s1) (ValChar c2) = Right $ ValString $ s1 ++ [c2]
valBinOp Bplus (ValInt i1) (ValString s2) = Right $ ValString $ show i1 ++ s2
valBinOp Bplus (ValString s1) (ValInt i2) = Right $ ValString $ s1 ++ show i2
valBinOp Bplus (ValFloat d1) (ValString s2) = Right $ ValString $ show d1 ++ s2
valBinOp Bplus (ValString s1) (ValFloat d2) = Right $ ValString $ s1 ++ show d2
valBinOp Bplus (ValBool b1) (ValString s2) = Right $ ValString $ show b1 ++ s2
valBinOp Bplus (ValString s1) (ValBool b2) = Right $ ValString $ s1 ++ show b2

valBinOp Bminus (ValInt i1) (ValInt i2) = Right $ ValInt $ i1 - i2
valBinOp Bminus (ValFloat d1) (ValFloat d2) = Right $ ValFloat $ d1 - d2
valBinOp Bminus (ValInt i1) (ValFloat d2) = Right $ ValFloat $ fromInteger i1 - d2
valBinOp Bminus (ValFloat d1) (ValInt i2) = Right $ ValFloat $ d1 - fromInteger i2

valBinOp Bmult (ValInt i1) (ValInt i2) = Right $ ValInt $ i1 * i2
valBinOp Bmult (ValFloat d1) (ValFloat d2) = Right $ ValFloat $ d1 * d2
valBinOp Bmult (ValInt i1) (ValFloat d2) = Right $ ValFloat $ fromInteger i1 * d2
valBinOp Bmult (ValFloat d1) (ValInt i2) = Right $ ValFloat $ d1 * fromInteger i2

valBinOp Bdiv (ValInt i1) (ValInt i2) = Right $ ValInt $ i1 `div` i2
valBinOp Bdiv (ValFloat d1) (ValFloat d2) = Right $ ValFloat $ d1 / d2
valBinOp Bdiv (ValInt i1) (ValFloat d2) = Right $ ValFloat $ fromInteger i1 / d2
valBinOp Bdiv (ValFloat d1) (ValInt i2) = Right $ ValFloat $ d1 / fromInteger i2

valBinOp Bmod (ValInt i1) (ValInt i2) = Right $ ValInt $ i1 `mod` i2

valBinOp op v1 v2 = Left $ unwords ["valBinOp undefined: ", show v1, show op, show v2]
\end{code}
%}}}

%{{{ valUnOp
\begin{code}
valUnOp :: UnOp -> Value -> Either String Value
valUnOp Uminus (ValInt i) = Right $ ValInt $ negate i
valUnOp Uminus (ValFloat d) = Right $ ValFloat $ negate d
valUnOp Unot (ValBool b)  = Right $ ValBool $ not b
valUnOp op v = Left $ unwords ["valUnOp undefined: ", show op, show v]
\end{code}
%}}}

%{{{ valPred
\begin{code}
valPred :: Pred -> Value -> Value -> Either String Bool
valPred p  (ValInt i1)     (ValInt i2)     = Right $ interpPred p i1 i2
valPred p  (ValFloat d1)   (ValFloat d2)   = Right $ interpPred p d1 d2
valPred p  (ValBool b1)    (ValBool b2)    = Right $ interpPred p b1 b2
valPred p  (ValChar c1)    (ValChar c2)    = Right $ interpPred p c1 c2
valPred p  (ValString s1)  (ValString s2)  = Right $ interpPred p s1 s2

valPred p v1 v2 = Left $ unwords ["valPred undefined: ", show v1, show p, show v2]
\end{code}
%}}}

%{{{ evalExpression, evalETerm
\begin{code}
evalExpression :: Map Name Value -> Expression -> Either String Value
evalExpression m (Or e1 e2) = do
  v1 <- evalExpression m e1
  case v1 of
    ValBool True -> Right $ ValBool True
    _ -> evalExpression m e2
evalExpression m e@(And e1 e2) = do
  v1 <- evalExpression m e1
  case v1 of
    ValBool False -> Right $ ValBool False
    ValBool True -> evalExpression m e2
    v2 -> Left $ unwords ["evalExpression", show e, ":  ill-typed conjunction of", show v1, "and", show v2]
evalExpression m (Atom e []) = evalETerm m e
evalExpression m (Atom e ps) = do
    v <- evalETerm m e
    h v ps
  where
    h v1 ((pred1, e2) : ps) = do
      v2 <- evalETerm m e2
      r1 <- valPred pred1 v1 v2
      if r1
        then if null ps
               then Right $ ValBool True
               else h v2 ps
        else Right $ ValBool False
    h v1 [] = error "evalExpression: IMPOSSIBLE"

evalETerm :: Map Name Value -> ETerm -> Either String Value
evalETerm m (Bin op e1 e2) = do
  v1 <- evalETerm m e1
  v2 <- evalETerm m e2
  valBinOp op v1 v2
evalETerm m (Un op e) = do
  v <- evalETerm m e
  valUnOp op v
evalETerm m (Var n) = case Map.lookup n m of
  Nothing -> Left $ unwords ["evalETerm: Variable", n, "undefined in", show m]
  Just v -> Right v
evalETerm m (BoolLit b) = Right $ ValBool b
evalETerm m (IntLit i) = Right $ ValInt i
evalETerm m (StringLit s) = Right $ ValString s
\end{code}
%}}}

\begin{code}
data NLab arg = NLab
  { nLabName :: !Name
  , nLabAttrs :: ![arg]
  }
  deriving (Eq, Ord)

instance Show arg => Show (NLab arg) where
  showsPrec _ (NLab name []) = (name ++)
  showsPrec _ (NLab name args) = (name ++) . ('(' :) .
       (foldr ($) `flip` intersperse (',' :) (map shows args)) . (')' :)

instance Functor NLab where
  fmap f (NLab n as) = NLab n $ map f as
\end{code}

%{{{ eVars :: [(PrimitiveType, Name)] -> [Value] -> Either String [(Name, Value)]
\begin{code}
eVars :: [(PrimitiveType, Name)] -> [Value] -> Either String [(Name, Value)]
eVars tyns attrs = sequence $ zipWith f tyns attrs
                          -- \edcomm{WK}{Does not yet capture different lengths!}
  where
    f (Tint, n)     v@(ValInt _)     = Right (n, v)
    f (Tfloat, n)   v@(ValFloat _)   = Right (n, v)
    f (Tfloat, n)   (ValInt i)       = Right (n, ValFloat $ fromInteger i)
    f (Tbool, n)    v@(ValBool _)    = Right (n, v)
    f (Tchar, n)    v@(ValChar _)    = Right (n, v)
    f (TString, n)  v@(ValString _)  = Right (n, v)
    f (ty, n) v = Left $ unwords ["eVars: Variable", n, "expected:", show ty, "found:", show v]
\end{code}
%}}}

%{{{ type SplitAttribs, mkSplitAttribs
\begin{code}
type SplitAttribs = [ParamOrArg] -> ([ParamOrArg], [ParamOrArg])
    -- |forall a . [a] -> ([a], [a])|
type SplitAttribsMap = Map Name SplitAttribs
\end{code}

\begin{code}
mkSplitAttribs :: [ParamOrArg] -> SplitAttribs
mkSplitAttribs (Param TAgent ag : pas) (x : xs) = second (x :) $ mkSplitAttribs pas xs
mkSplitAttribs (Param ty (Agent name []) : pas) (x : xs) = first (x:) $ mkSplitAttribs pas xs
mkSplitAttribs (_ : pas) (x : xs) = second (x :) $ mkSplitAttribs pas xs
mkSplitAttribs [] [] = ([], [])
mkSplitAttribs (_ : pas) [] = error $ "mkSplitAttribs: Argument list too short"
mkSplitAttribs [] _         = error $ "mkSplitAttribs: Argument list too long"
\end{code}
%}}}

%{{{ getSplitAttribs :: [RuleDef] -> Map Name SplitAttribs
\begin{code}
getSplitAttribs :: [RuleDef] -> Map Name SplitAttribs
getSplitAttribs = foldr enter Map.empty . concatMap fromRuleDef
  where
    fromRuleDef (RuleDef f@(Agent fName fPAs) _ rhss) = (fName, fPAs)
                    : map (fromRHS . ruleRHS) rhss
    fromRHS (RuleRHS (Agent cName cPAs) _ _) = (cName, cPAs)
    enter (name, pas) m = case Map.lookup name m of
      Nothing -> Map.insert name (mkSplitAttribs pas) m
      Just _ -> m
\end{code}
%}}}


%{{{ data Conditional expr r
\begin{code}
data Conditional expr r
  = CondLeaf r
  | CondBranch expr (Conditional expr r) (Conditional expr r)
  deriving (Show)

instance Functor (Conditional expr) where
  fmap f (CondLeaf r) = CondLeaf (f r)
  fmap f (CondBranch b t e) = CondBranch b (fmap f t) (fmap f e)
\end{code}
%}}}

%{{{ listFromConditional
\begin{code}
listFromConditional :: Conditional expr r -> [r]
listFromConditional cond = h cond []
  where
    h (CondLeaf r) = (r :)
    h (CondBranch b t e) = h t . h e
\end{code}
%}}}

%{{{ evalConditional
\begin{code}
evalConditional :: (expr -> Bool) -> Conditional expr r -> r
evalConditional eval (CondLeaf r) = r
evalConditional eval (CondBranch b t e) = if eval b  then evalConditional eval t
                                                     else evalConditional eval e
\end{code}
%}}}

%{{{ extractConditional :: RuleBody -> Conditional Expression [Equation]
\begin{code}
extractConditional :: RuleBody -> Conditional Expression [Equation]
extractConditional (RBeq eqs) = CondLeaf eqs
extractConditional RBskip     = CondLeaf []
extractConditional (RBif s) = fromIfStatement s
  where
    fromIfStatement (IF b t e) = CondBranch b (fromIfBlock t)
        (either fromIfBlock fromIfStatement e)
    fromIfBlock (IfBlock _ eqs _) = CondLeaf eqs
\end{code}
%}}}

%{{{ getAttrib, getArg, etc.
\edcomm{WK}{The |String| arguments in the following are only
for debugging during development,
while I am not fully understanding the Inets grammar and language definition.}

\begin{code}
getAttrib :: String -> ParamOrArg -> Expression
getAttrib _ (Arg (Eexpr e)) = e
-- |getAttrib (Arg (EAgent ag)) = |
getAttrib caller pa = error $ unwords ["getAttrib: unexpected", show pa, "\n   in", caller]

getArg :: String -> ParamOrArg -> Term
getArg _ (Arg (EAgent ag)) = TermAgent ag
getArg _ (Arg (EVar ag)) = TermVar ag
getArg _ (Arg (Evariadic ag)) = TermVariadic ag
getArg _ (Param TAgent ag@(Agent name [])) = TermVar ag
getArg caller pa = error $ unwords ["getArg: unexpected", show pa, "\n   in", caller]

getDecl :: String -> ParamOrArg -> (PrimitiveType, Name)
getDecl _ (Param ty (Agent name [])) = (ty, name)
getDecl caller pa = error $ unwords ["getArg: unexpected", show pa, "\n   in", caller]

mkSplitAttribsL :: String -> Map Name SplitAttribs -> Name -> [ParamOrArg] -> ([(PrimitiveType, Name)],[Term])
mkSplitAttribsL _ _ name [] = ([], [])
mkSplitAttribsL caller splitAttribsMap name pas = case Map.lookup name splitAttribsMap of
  Nothing -> error $ unwords ["splitAttribsL:", name, "not found in splitAttribsMap"]
  Just split -> case split pas of
    (attribs, args) -> (map (getDecl caller) attribs, map (getArg caller) args)

mkSplitAttribsR :: String -> Map Name SplitAttribs -> Name -> [ParamOrArg] -> ([Expression],[Term])
mkSplitAttribsR _ _ name [] = ([], [])
mkSplitAttribsR caller splitAttribsMap name pas = case Map.lookup name splitAttribsMap of
  Nothing -> error $ unwords ["splitAttribsR:", name, "not found in splitAttribsMap"]
  Just split -> case split pas of
    (attribs, args) -> (map (getAttrib caller) attribs, map (getArg caller) args)

argVars :: (PI -> PortTargetDescription) -> [Term] -> [(Name, PortTargetDescription)]
argVars mkPTD = h 1
  where
    h pi [] = []
    h pi (TermAgent ag : ts) = h (succ pi) ts
    h pi (TermVar (Agent name []) : ts) = (name, mkPTD pi) : h (succ pi) ts
    h pi (_ : ts) = h (succ pi) ts

toPTerm :: String -> Map Name SplitAttribs -> Term -> PTerm (NLab Expression) Name
toPTerm _ _ (TermVar (Agent name [])) = ConnVar name
toPTerm caller splitAttribsMap (TermAgent (Agent name pas))
  = case mkSplitAttribsR caller splitAttribsMap name pas of
     (attribs, args) -> PNode (NLab name attribs) (map (toPTerm caller splitAttribsMap) args)
toPTerm _ _ t = error $ "toPTerm: unexpected argument: " ++ show t

toPTermEq :: String -> Map Name SplitAttribs ->  Equation -> PTermEq (NLab Expression) Name
toPTermEq caller splitAttribsMap (Equation t1 t2)
   =  ( toPTerm caller splitAttribsMap t1
      , toPTerm caller splitAttribsMap t2)
\end{code}
%}}}

%{{{ convertRuleDef
In the Inets source,
|src/inets/transform/NestedToOrd.java| documents the following decisions:
\begin{enumerate}
\item All rules with more than 2 agents on the LHS \emph{and} also
   having an RHS net will be considered as rules with nested patterns.
\item All rules that have more than 2 agents on the LHS \emph{and}
   do not have an RHS net will be considered as ordinary rules in Lafont style.
\item All rules with exactly two agents on the LHS and an equation list as the RHS
   net will be considered as ordinary rules.
\end{enumerate}
A twist is the following rule in |lib/List.inet|:

  |App(b,b) ><  Nil;|

This can be recognised as being in Lafont-style
only from the fact that its variable occurs twice.

\begin{code}
convertRuleDef :: Map Name SplitAttribs
               -> RuleDef
               -> (    NLab (PrimitiveType, Name)
                  , [  ( NLab (PrimitiveType, Name)
                       , Conditional Expression (NetDescription (NLab Expression)) )])
convertRuleDef splitAttribsMap rd@(RuleDef f@(Agent fName fPAs) _ rhss) = let
  (fParams, fArgs0) = splitAttribsL fName fPAs
  fArgVars = argVars SourcePort fArgs0

  splitAttribsL :: Name -> [ParamOrArg] -> ([(PrimitiveType, Name)],[Term])
  splitAttribsL = mkSplitAttribsL (show rd) splitAttribsMap

  mkPTerm :: Term -> PTerm (NLab Expression) Name
  mkPTerm = toPTerm (show rd) splitAttribsMap

  mkPTermEq :: Equation -> PTermEq (NLab Expression) Name
  mkPTermEq = toPTermEq (show rd) splitAttribsMap

  convertRHS :: RuleRHS ->  ( NLab (PrimitiveType, Name)
                            , Conditional Expression (NetDescription (NLab Expression)))
  convertRHS rhs@(RuleRHS c@(Agent cName cPAs) _ RBskip)
    | length fArgVars < length fArgs0 || length cArgVars < length cArgs0
      || length (nub vNames) < length vNames
    = let -- this is a Lafont-style rule
        f', c' :: PTerm (NLab Expression) Name
        f' = PNode (NLab fName $ error "fParams") $ map mkPTerm fArgs0
        c' = PNode (NLab cName $ error "cParams") $ map mkPTerm cArgs0
        (_, nd) = buildLafontRule (f', c')
      in -- |trace (unwords ["\nconvertRuleDef: Lafont:", show f, "><", show c]) $|
         case checkNetDescription nd of
                 [] -> (,) (NLab cName cParams) $ CondLeaf nd
                 clashes -> error $ unlines $
                   ("\nconvertRHS: Inconsistent net description in Lafont-style rule:" ++ fName ++ " >< " ++ cName)
                   : unwords ["   ", show f, "><", show c]
                   : show nd : "Clashes:" : map (("  " ++) . show) clashes
      where
        (cParams, cArgs0) = splitAttribsL cName cPAs
        cArgVars = argVars TargetPort cArgs0
        vNames = map fst $ fArgVars ++ cArgVars
  convertRHS rhs@(RuleRHS c@(Agent cName cPAs) _ body) -- |RBskip| is now an empty list of equations
    = (,) (NLab cName cParams) $ let
        patArgVars :: Map Name PortTargetDescription
        patArgVars = Map.fromList $ fArgVars ++ cArgVars
      in if length fArgVars < length fArgs0 || length cArgVars < length cArgs0
         then error $ unwords ["Nesting not yet implemented:", show f, show rhs]
         else let
             mkNet :: [Equation] -> NetDescription (NLab Expression)
             mkNet eqs = let nd = buildOrdinaryRHS patArgVars $ map mkPTermEq eqs
               in case checkNetDescription nd of
                 [] -> nd
                 clashes -> error $ unlines $
                   ("\nconvertRHS: Inconsistent net description: in " ++ fName ++ " >< " ++ cName)
                   : "Equations:" : map (("  " ++) . show) eqs
                   ++ show nd : "Clashes:" : map (("  " ++) . show) clashes
           in fmap mkNet $ extractConditional body
      where
        (cParams, cArgs0) = splitAttribsL cName cPAs
        cArgVars = argVars TargetPort cArgs0
  in (NLab fName fParams, map (convertRHS . ruleRHS) rhss)
\end{code}
%}}}

%{{{ type SymbLangMap
\begin{code}
type SymbLangSubMap = Map Name  (  ([(PrimitiveType, Name)], [(PrimitiveType, Name)])
                                ,  Conditional Expression (NetDescription (NLab Expression)))
type SymbLangMap = Map Name SymbLangSubMap
\end{code}
%}}}

%{{{ arityFromSymbLangMap
\begin{code}
arityFromSymbLangMap :: SymbLangMap -> Map Name Int
arityFromSymbLangMap = let
  fromConditional :: Conditional Expression (NetDescription (NLab Expression))
                  -> [(Name, Int)] -> [(Name, Int)]
  fromConditional (CondLeaf nd) = collectArityFromNetDescription nLabName nd
  fromConditional (CondBranch b t e) = fromConditional t . fromConditional e
  condIfaceLengths cond =  let nd = head $ listFromConditional cond
                           in (V.length (source nd), V.length (target nd))
  fromSubMap :: SymbLangSubMap -> (Int, [(Name, Int)] -> [(Name, Int)])
  fromSubMap sm = case Map.toList sm of
      ps@(~((cName1, (_, cond1)) : _)) -> case condIfaceLengths cond1 of
        (fSize, cSize) -> (,) (succ fSize) $ foldr h `flip` ps
    where
      h (cName, (_, cond)) = ((cName, succ $ snd $ condIfaceLengths cond) :) . fromConditional cond

  g fName sm = case fromSubMap sm of
      (fSize, collect_sm) -> ((fName, fSize) :) . collect_sm
  in Map.fromList . Map.foldrWithKey g []
\end{code}
%}}}

%{{{ cuINetLang
\begin{code}
cuINetLang :: CompilationUnit
           -> (  (SymbLangMap, Map Name (Vector Polarity))
              ,  (SplitAttribsMap, INetLang (NLab Value))
              )
cuINetLang cu = let
    rds = cuRuleDefs cu
    splitAttribsMap = getSplitAttribs rds
    addToFMap :: (    NLab (PrimitiveType, Name)
                  , [  ( NLab (PrimitiveType, Name)
                       , Conditional Expression (NetDescription (NLab Expression)) )]
                 ) -> SymbLangMap -> SymbLangMap
    addToFMap (NLab fName fParams, ps)
      = Map.insertWith  Map.union -- |(\ newsub subm -> foldr (addToCMap fParams) subm ps)|
                 fName  (foldr (addToCMap fParams) Map.empty ps)
    addToCMap :: [(PrimitiveType, Name)]
              -> ( NLab (PrimitiveType, Name)
                  , Conditional Expression (NetDescription (NLab Expression)) )
              -> SymbLangSubMap -> SymbLangSubMap
    addToCMap fParams (NLab cName cParams, cond) = Map.insert cName ((fParams, cParams), cond)
    symbLangMap :: SymbLangMap
    symbLangMap = foldr addToFMap Map.empty $ map (convertRuleDef splitAttribsMap) rds
    find :: NLab Value -> NLab Value -> NetDescription (NLab Value)
    find (NLab fName fAttribs) = case Map.lookup fName symbLangMap of
      Nothing -> error $ "cuINetLang: Undefined function label " ++ fName
      Just cMap -> \ (NLab cName cAttribs) -> case Map.lookup cName cMap of
        Nothing -> error $ "uINetLang: Undefined constructor label " ++ cName
        Just ((fParams, cParams), cond) -> case eVars fParams fAttribs of
          Left e -> error $ "cuINetLang: Function parameters mismatch: " ++ e
          Right fVal -> case eVars cParams cAttribs of
            Left e -> error $ "cuINetLang: Constructor parameters mismatch: " ++ e
            Right cVal -> let
                val :: Map Name Value
                val = Map.fromList $ fVal ++ cVal
                evalExpr :: Expression -> Value
                evalExpr e = case evalExpression val e of
                   Left err -> error $ unwords ["cuINetLang: evaluating", show e, ":", err]
                   Right v -> v
                evalBool :: Expression -> Bool
                evalBool e = case evalExpr e of
                   ValBool b -> b
                   v -> error $ unwords ["cuINetLang: condition", show e, "evaluates to", show v]
                ndE :: NetDescription (NLab Expression)
                ndE = evalConditional evalBool cond
              in fmap (fmap evalExpr) ndE
    polMap = extractPolarity symbLangMap
    pol :: NLab Value -> Vector Polarity
    pol (NLab name _) = polMap Map.! name
  in ((symbLangMap, polMap), (splitAttribsMap, INetLang pol find))
\end{code}
%}}}

%{{{ mmInsert, mmLookup
\begin{code}
mmInsert :: (Ord a, Ord b, Eq c, Show a, Show b, Show c)
         => String -> a -> b -> c -> Map a (Map b c) -> Map a (Map b c)
mmInsert msg a b c m = -- |trace (unwords ["\nTrace mmInsert:\n  ", msg, "\n  ", show a, show b, "->", show c, "\n"]) $|
                       Map.insertWith (Map.unionWith h) a (Map.singleton b c) m
   where
     h c2 c1 = if c1 == c2 then c1
       else error $ unwords ["mmInsert", show a, show b, ":", show c1, "- ->", show c2
                            ,"\n   ",msg
                            ,"\n",unlines $ map show $ Map.toList m]

mmLookup :: (Ord a, Ord b) => a -> b -> Map a (Map b c) -> Maybe c
mmLookup a b m = Map.lookup a m >>= Map.lookup b
\end{code}
%}}}

%{{{ vectorFromMap
\begin{code}
vectorFromMap :: Map PI a -> Vector a
vectorFromMap = V.fromList . h 0 . Map.toAscList
  where
    h i [] = []
    h i ((j, x) : ps) = if j > i
      then error $ unwords ["vectorFromMap: Gap from", show i, "to below", show j]
      else x : h (succ i) ps
\end{code}
%}}}

%{{{ extractPolarity
\begin{code}
extractPolarity :: SymbLangMap -> Map Name (Vector Polarity)
extractPolarity symbLangMap = let
    pports :: Map Name (Map PI Polarity)
    pports = Map.foldrWithKey (\ fName subMap -> h subMap . mmInsert "fName PP" fName 0 Neg) Map.empty symbLangMap
       where
         h sm m = Map.foldrWithKey (\ cName (_, cond) -> mmInsert "cName PP" cName 0 Pos) m sm
    subrules :: [((Name, Name), NetDescription (NLab Expression))]
    subrules = concatMap flattenSubMap $ Map.toList symbLangMap
      where
        flattenSubMap (fName, cMap) = concatMap (flattenEntry fName) $ Map.toList cMap
        flattenEntry fName (cName, (_, cond)) = map ((,) (fName, cName)) $ listFromConditional cond
    addSubRule  :: ((Name, Name), NetDescription (NLab Expression))
                -> Map Name (Map PI Polarity) -> Map Name (Map PI Polarity)
    addSubRule ((fName, cName), nd) = (V.ifoldr addNode `flip` (nodes nd))
                                      . addIface fName source
                                      . addIface cName target
      where
        -- |(fAuxSize, cAuxSize) = (V.length (source nd), V.length (target nd))|
        mkMsg s = fName ++ " >< " ++ cName ++ " : " ++ s
        addIface name getPTDs m = let
            pols = Map.findWithDefault Map.empty name m
            ptds = getPTDs nd
            addPort' i ptd = case Map.lookup i pols of
               Nothing -> id
               Just pol -> addPort (unwords ["f:", show i, show ptd]) ptd (opposite pol)
          in foldr ($) m $ zipWith addPort' [1..] (V.toList ptds)

        addPort  :: String -> PortTargetDescription -> Polarity
                -> Map Name (Map PI Polarity) -> Map Name (Map PI Polarity)
        addPort msg0 ptd pol m = case ptd of
          SourcePort pi -> mmInsert (mkMsg "addPort: SourcePort") fName pi pol m
          TargetPort pi -> mmInsert (mkMsg "addPort: TargetPort") cName pi pol m
          InternalPort ni pi -> let
               msg2 = unwords [msg1, "addPort: InternalPort", show ni, nL', show pi]
               nL' = nLabName . nLab $ nodes nd V.! ni
             in mmInsert (mkMsg msg2) (nLabName . nLab $ nodes nd V.! ni) pi (opposite pol) m
          where
            msg1 = unwords ["\n", msg0, show ptd, show pol, "\n  ", show m, "\n  "]

        addNode ni0 (NodeDescription nL pds) m = case Map.lookup (nLabName nL) m of
            Nothing -> m
            Just pols -> Map.foldrWithKey addPol m pols
          where
            addPol  :: PI -> Polarity
                    -> Map Name (Map PI Polarity) -> Map Name (Map PI Polarity)
            addPol i pol = addPort msg0 (V.atErr "addPol" pds i) pol
              where
                msg0 = unwords [show nL, show ni0, show i, show pds, "\n  "]

    arityMap = arityFromSymbLangMap symbLangMap
    arityCount = Map.size arityMap

    inComplete :: Map Name (Map PI Polarity) -> Map Name (Map PI Polarity)
    inComplete = Map.filterWithKey h
      where
        h name sm  =  Map.size sm /= arityMap Map.! name

    -- The following material is heuristic in nature, for filling in gaps in the
    -- polarity information that are not covered by |addSubRule|.
                      
    -- |singleGapAllNeg nL sm| returns |Just pi| if |pi| is the only port index  of |nL|
    -- without polarity in |sm|, and all polarities in |sm| are |Neg|.
    singleGapAllNeg :: Name -> Map PI Polarity -> Maybe (Name, PI)
    singleGapAllNeg name sm = if succ (Map.size sm) /= arity
         then Nothing
         else -- exactly one gap
           h 0 $ Map.toAscList sm -- find that gap
      where
        arity = arityMap Map.! name
        h i [] = Nothing -- don't consider zero-ary agents to avoid terminators.
        h i ((i',pol) : ps) = case pol of
          Pos -> Nothing
          Neg -> if i == i'
                 then h' (succ i) ps
                 else if all (Neg ==) $ map snd ps
                      then Just (name, i)
                      else Nothing -- not all negative
        h' i [] = Just (name, i) -- no earlier gap found
        h' i ps = h i ps

    add i m = let m' = foldr addSubRule m subrules
      in if m /= m'
         then add (succ i) m'
         else let
              mIncomplete = inComplete m
              mIncCount = Map.size mIncomplete
            in -- |trace (unlines|
               -- |( ("\nextractPolarity: " ++ shows i " iterations.\n")|
               -- |: map show (Map.toList m)|
               -- |++ [unwords [show (Map.size m), "entries;"|
               -- |            , show mIncCount, "incomplete;"|
               -- |            , show arityCount, "needed."]]|
               -- |)) $|
               if Map.size m == arityCount -- |&& mIncCount == 0|
                     then m
                     else case Map.foldrWithKey (\ n sm r -> singleGapAllNeg n sm `mplus` r) Nothing m of
                        Just (name, pi) -> -- |trace (unwords ["singleGapAllNeg:", name, show pi, "-> Pos"]) $|
                                           add 0 $ mmInsert "singleGapAllNeg" name pi Pos m
                        Nothing -> error $ unlines $
                                     "Don't know how to complete polarity.\n"
                                     : map show (Map.toList arityMap)
                                     ++ [unwords [show (Map.size m), "entries;"
                                                 , show mIncCount, "incomplete;"
                                                 , show arityCount, "needed."]]
  in Map.map vectorFromMap $ add 1 pports
\end{code}
%}}}

% Local Variables:
% folded-file: t
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% fold-internal-margins: 0
% end:
