\section{GetModuleExports}

%{{{ module GetModuleExports
\begin{code}
module GetModuleExports
  ( Exports(..), moduleExports, getExports
  , getExportNames, getTypedExportNames
  ) where

import qualified GHC.Paths (libdir)

-- package GHC
import GHC (depanal, getModSummary, parseModule, typecheckModule
           , typecheckedSource
           , getSessionDynFlags, setSessionDynFlags
           , Target(..), TargetId(..), addTarget, getTargets
           , LoadHowMuch(LoadAllTargets), load
           , getModuleInfo, modInfoExports
           , runGhc, GhcMonad, getSession
           , printExceptionAndWarnings
           , handleSourceError
           , Class
           , DataCon
           , TyCon
           , TyThing(..)
           , Id
           , lookupGlobalName -- |:: GhcMonad m => Name -> m (Maybe TyThing)|
           )
import Name (Name)
import Finder (findImportedModule, FindResult(..), cannotFindInterface)
import LoadIface (loadSrcInterface, findAndReadIface)
import IfaceSyn (IfaceInst(ifDFun))
import TcIface (typecheckIface)
import TcEnv (iDFunId)
import TcRnMonad (initTc, addWarnTc, getTopEnv, TcRn)
import Inst (tcGetInstEnvs)
import InstEnv (Instance(..), InstEnv, emptyInstEnv, instEnvElts)
import HsBinds (HsBindLR(..))
import SrcLoc (Located(L))
import HscTypes ( IsBootInterface, HscSource(HsSrcFile)
                , ModSummary (ms_mod)
                , ModIface, mi_insts
                , ModDetails (md_exports)
                , availNames
                , typeEnvElts
                , hsc_dflags, hsc_global_type_env
                , hptInstances
                , ioMsg, ioMsgMaybe
                )
import Module   ( Module, mkModule, mainPackageId
                , ModuleName, moduleNameString )
import Outputable
import MonadUtils (MonadIO(liftIO))
import Maybes (MaybeErr(..))
import Bag (Bag, bagToList)
import FastString (FastString)

import PprUtils (pprint)


import Data.Maybe (catMaybes)
import Data.Set (Set)
import qualified Data.Set as Set

import Control.Monad.State (modify, execState)

import System.IO

import Prelude hiding (mod)
\end{code}
%}}}

\begin{code}
moduleExports :: [ModuleName] -> IO [Exports]
moduleExports modNames = runGhc (Just GHC.Paths.libdir) $
    handleSourceError srcErrHandle $ mapM getExports modNames
  where
    srcErrHandle srcErr = do
      printExceptionAndWarnings srcErr
      fail $ "moduleExports " ++ unwords (map moduleNameString modNames) ++ " failed."
\end{code}

\begin{code}
getExports :: (GhcMonad m) => ModuleName -> m Exports
getExports modName = do
    (_mod, ((names, _globals), _inst_envs)) <- getExportNames modName
    tyThings <- fmap catMaybes $ mapM lookupGlobalName names
    return $ eCollect tyThings
\end{code}

%{{{ data Exports
\begin{code}
data Exports = Exports
  {exportIds :: Set Id
  ,exportDataCons :: Set DataCon
  ,exportTyCons :: Set TyCon
  ,exportClasses :: Set Class
  }

emptyExports :: Exports
emptyExports = Exports Set.empty Set.empty Set.empty Set.empty
\end{code}
%}}}

%{{{ eCollect
\begin{code}
eInsert :: TyThing -> Exports -> Exports
eInsert (AnId ident)   e = e {exportIds = Set.insert ident $ exportIds e}
eInsert (ADataCon dc)  e = e {exportDataCons = Set.insert dc $ exportDataCons e}
eInsert (ATyCon tc)    e = e {exportTyCons = Set.insert tc $ exportTyCons e}
eInsert (AClass c)     e = e {exportClasses = Set.insert c $ exportClasses e}
\end{code}

\begin{code}
eCollect :: [TyThing] -> Exports
eCollect tyThings = execState (mapM_ (modify . eInsert) tyThings) emptyExports
\end{code}
%}}}

\begin{code}
type InstInfo = IfaceInst

type InstEnvInfo = (Instance, Id)
\end{code}

%{{{ getTypedExportNames
\begin{code}
getTypedExportNames :: (GhcMonad m)
  => ModuleName -> m (Module, (([(Name, Maybe TyThing)],[(InstInfo, Id)]),([InstEnvInfo],[InstEnvInfo])))
getTypedExportNames modName = do
  (m, ((exports, insts), (instEnv1, instEnv2))) <- getExportNames modName
  let  tyName n = do
           mty <- lookupGlobalName n
           return (n, mty)
       tyInst getDictId i = do
           mty <- lookupGlobalName (getDictId i)
           case mty of
             Just (AnId d) -> return (i, d)
             _ -> return (i, error "getTypedExportNames: funny instance")
       tyInstE = map (\ i -> (i, is_dfun i))
  exports' <- mapM tyName exports
  insts' <- mapM (tyInst ifDFun)  insts
  let  ie1 = tyInstE instEnv1
       ie2 = tyInstE instEnv2
  return (m, ((exports', insts'),(ie1, ie2)))
\end{code}
%}}}

%{{{ getExportNames
\begin{code}
getExportNames :: (GhcMonad m) => ModuleName -> m (Module, (([Name],[InstInfo]),([Instance],[Instance])))
getExportNames modName = do
    getSessionDynFlags >>= setSessionDynFlags
    -- |liftIO $ putStrLn "after setSessionDynFlags"|
    ifaceResult <- getInterfaceExports modName Nothing
    case ifaceResult of
      Just (m,(p,(ie1,ie2))) -> return (m,(p,(instEnvElts ie1,instEnvElts ie2)))
      Nothing -> getSourceExports modName
\end{code}
%}}}

%{{{ getSourceExports
\begin{code}
getSourceExports :: (GhcMonad m) => ModuleName -> m (Module, (([Name],[InstInfo]),([Instance],[Instance])))
getSourceExports modName = do
    addTarget $ Target (TargetModule modName) True Nothing
    -- |liftIO $ putStrLn "after addTarget"|
    targets <- getTargets
    liftIO $ pprint $ vcat $ map ppr targets
    load LoadAllTargets
    -- |liftIO $ putStrLn "after load LoadAllTargets"|
    -- Then find dependencies
    _modGraph <- depanal [] True
    -- |liftIO $ putStrLn "after depanal"|
    modSummary  <- getModSummary modName
    let mod = ms_mod modSummary
    -- |liftIO $ putStrLn "after getModSummary"|
    modTC       <- typecheckModule =<< parseModule modSummary
    -- |liftIO $ putStrLn "after typecheckModule"|
    mModInfo <- getModuleInfo mod
    let  insts = [] -- |iDFunId| -- |mi_insts modIface|
    hsc_env <- getSession
    let  inst_envs = ([], fst $ hptInstances hsc_env (const True))
         -- use |eps_inst_env|?
    case mModInfo of
      Just modInfo -> return $ (mod, ((modInfoExports modInfo, insts), inst_envs))
      Nothing -> do
        liftIO $ putStrLn $ "No ModuleInfo found for " ++ moduleNameString modName
        liftIO . pprint . outputLHsBinds . bagToList $ typecheckedSource modTC
        return (mod, (([], insts), inst_envs))
\end{code}
%}}}

%{{{ nonVarBinds
\begin{code}
nonVarBinds :: Bag (Located (HsBindLR lId rId)) -> [Located (HsBindLR lId rId)]
nonVarBinds = foldr h [] . bagToList
  where
    h (L _ (VarBind _ _)) r = r
    h b r = b : r
\end{code}
%}}}

%{{{ outputLHsBinds
\begin{code}
outputLHsBinds :: (OutputableBndr t) => [Located (HsBindLR t t)] -> SDoc
outputLHsBinds = vcat . map outputLHsBind

outputLHsBind :: (OutputableBndr t) => Located (HsBindLR t t) -> SDoc
outputLHsBind (L srcSpan hsbind) = ppr srcSpan <+> case hsbind of
  FunBind funId _funFix _funMatchGroup _coerce _mutuals _tckNo -> text "FunBind:" <+> ppr funId
  PatBind patLhs _patRhs _patRhsTy _mutuals -> text "PatBind:" <+> ppr patLhs
  VarBind varId _varRhs -> text "VarBind:" <+> ppr varId
  AbsBinds absTyvars absDicts absExports absBinds -> text "AbsBinds:" <+> nest 9
    (ppr absTyvars $+$ ppr absDicts $+$ vcat (map (ppr . (\ (a,b,c,_) -> (a,b,c))) absExports) $+$ outputLHsBinds (nonVarBinds absBinds))
\end{code}
%}}}

%{{{ loadFullInterface
Since |LoadIface.loadInterface| and |LoadIface.loadSrcInterface|
strip out the interesting parts,
we have to roll our own |loadFullInterface|.

|maybe_pkg|: If nothing, search path and exposed modules;
if |Just "this"|, then only search path;
if |Just pkg|, then only |pkg| is searched.

\begin{code}
loadFullInterface' :: ModuleName -> Maybe FastString -> TcRn (Maybe ((Module, ModIface), (InstEnv, InstEnv)))
loadFullInterface' modName maybe_pkg = do
  hsc_env <- getTopEnv
  res <- liftIO $ findImportedModule hsc_env modName maybe_pkg
  case res of
    Found _ mod -> do
      read_result <- findAndReadIface (text "loadFullIface") mod (False :: IsBootInterface)
      case read_result of
        Failed err      -> do
          addWarnTc err
          return Nothing
        Succeeded  (iface, file_path) -> do
          liftIO $ hPutStrLn stderr $ "Found interface for " ++ moduleNameString modName ++ " at " ++ file_path
          inst_envs <- tcGetInstEnvs
          return $ Just ((mod, iface), inst_envs)
    err ->
        let dflags = hsc_dflags hsc_env in
        addWarnTc (cannotFindInterface dflags modName err) >> return Nothing
\end{code}

\begin{code}
_loadFullInterface :: (GhcMonad m)
  => ModuleName -> Maybe FastString -> m (Maybe ((Module, ModIface), (InstEnv, InstEnv)))
_loadFullInterface modName maybe_pkg = let
   tcInitMod = mkModule (error "loadFullInterface': initTc really needs a packageId") modName
 in do
  hsc_env <- getSession
  ioMsgMaybe $
    initTc hsc_env HsSrcFile True tcInitMod $ -- saving renamed syntax
    -- |initTcRnIf "loadFullInterface" hsc_env gbl_env lcl_env|
    loadFullInterface' modName maybe_pkg
\end{code}
%}}}

%{{{ getInterfaceExports
\begin{code}
getInterfaceExports :: (GhcMonad m)
  => ModuleName -> Maybe FastString -> m (Maybe (Module, (([Name], [InstInfo]), (InstEnv, InstEnv))))
getInterfaceExports modName maybe_pkg = let
   tcInitMod = mkModule mainPackageId modName -- |mainPackageId| works, but is hopefully not used.
 in do
  hsc_env <- getSession
  let _globals = typeEnvElts $ hsc_global_type_env hsc_env
  ioMsg $ initTc hsc_env HsSrcFile True tcInitMod $ do -- saving renamed syntax
    _modIface <- loadSrcInterface (text "getInterface") modName (False :: IsBootInterface) maybe_pkg
    -- |loadSrcInterface| apparently is necessary to populate internal structures,
    -- but not sufficient to allow access via |md_exports|.
    mp <- loadFullInterface' modName maybe_pkg
    case mp of
      Nothing -> fail "loadFullInterface'"
      Just ((mod, modIface), inst_envs) -> do
        -- |liftIO $ putStrLn $ "after loadSrcInterface " ++ moduleNameString modName|
        modDetails <- typecheckIface modIface
        let  insts = mi_insts modIface
        -- |liftIO $ putStrLn $ "after typecheckIface " ++ moduleNameString modName|
        return $ (mod, ((md_exports modDetails >>= availNames, insts), inst_envs))
\end{code}
%}}}

%{{{ EMACS lv
% Local Variables:
% folded-file: t
% fold-internal-margins: 0
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% end:
%}}}
