\section{MatchTypes}

\begin{code}
module MatchTypes where

import PprUtils (traceSDoc)
import IfChanged

-- pachage ghc:
import Type  (mkTyVarTy, PredType(..), ThetaType, tcEqPred
             , mkTyConApp, mkFunTy, mkAppTy)
import TcType (tcEqType, tcSplitSigmaTy
  , tcGetTyVar_maybe, tcSplitFunTy_maybe, tcSplitTyConApp_maybe, tcSplitAppTy_maybe
  , TcTyVarSet
  , tyVarsOfTypes, tyVarsOfType
  , tcTyVarsOfType, tcTyVarsOfTypes
  , mkSigmaTy)
import Var (TyVar)
import TyCon (TyCon)
import TypeRep (Type(..))
import VarSet (elemVarSet, unionVarSet, emptyVarSet)
import Outputable

import Data.Map (Map)
import qualified Data.Map as Map
import Data.Set (Set)
import qualified Data.Set as Set

import Data.Maybe (mapMaybe)

import Control.Arrow ( second, (***) )
import Control.Monad.State
import Control.Monad.Trans (lift)
\end{code}


%{{{ matchType :: Type -> Type -> Maybe Subst
\begin{code}
type Match = StateT Subst Maybe
\end{code}

\begin{code}
addToMatch :: TyVar -> Type -> Match ()
addToMatch v ty = do
  m <- get
  case Map.lookup v m of
    Nothing -> put (Map.insert v ty m)
    Just ty' -> if tcEqType ty ty' then return ()
      else mzero
\end{code}

\begin{code}
matchType :: Type -> Type -> Maybe Subst
matchType ty1 ty2  = flip execStateT Map.empty
                   $ matchTau (stripSigma ty1) (stripSigma ty2)
\end{code}

\begin{code}
matchTau :: Type -> Type -> Match ()
matchTau ty1 ty2 = case tcGetTyVar_maybe ty1 of
  Just tv1 -> addToMatch tv1 ty2
  Nothing -> case tcSplitFunTy_maybe ty1 of
    Just (f1, a1) -> do
      (f2, a2) <- lift $ tcSplitFunTy_maybe ty2
      matchTau f1 f2
      matchTau a1 a2
    Nothing -> case tcSplitTyConApp_maybe ty1 of
      Just (tc1, tys1) -> do
        (tc2, tys2) <- lift $ tcSplitTyConApp_maybe ty2
        guard $ showSDoc (ppr tc1) == showSDoc (ppr tc2) -- |TyCon|s from separate runs
        sequence_ $ zipWith matchTau tys1 tys2
      Nothing -> case tcSplitAppTy_maybe ty1 of
        Just (f1, a1) -> do
          (f2, a2) <- lift $ tcSplitAppTy_maybe ty2
          matchTau f1 f2
          matchTau a1 a2
        Nothing -> -- |trace ("matchTau: unexpected type: " ++ showSDoc (ppr ty1))|
                   mzero
\end{code}
%}}}

%{{{ freeInType :: TyVar -> Type -> Bool
\edcomm{WK}{|freeInType' v (mkTyVarTy v)| reports different results!}

\begin{code}
freeInType' :: TyVar -> Type -> Bool
freeInType' v ty = let
    result1 = v `elemVarSet` tcTyVarsOfType ty
    result2 = freeInType v ty
  in if result1 == result2 then result2
    else traceSDoc
  (text "freeInType'" <+> ppr v <+> ppr ty <+> text "=" <+> ppr result1 <+> ppr result2)
  result2
\end{code}

\begin{code}
freeInType :: TyVar -> Type -> Bool
freeInType v ty = case tcGetTyVar_maybe ty of
  Just tv -> v == tv
  Nothing -> case tcSplitFunTy_maybe ty of
    Just (f, a) -> freeInType v f || freeInType v a
    Nothing -> case tcSplitTyConApp_maybe ty of
      Just (_tc, tys) -> freeInTypes v tys
      Nothing -> case tcSplitAppTy_maybe ty of
        Just (f, a) -> freeInType v f || freeInType v a
        Nothing -> error $ "freeInType: unexpected type: " ++ showSDoc (ppr ty)

freeInTypes :: TyVar -> [Type] -> Bool
freeInTypes v = any (freeInType v)

freeInSubst :: TyVar -> Subst -> Bool
freeInSubst v = freeInTypes v . Map.elems
\end{code}
%}}}

%{{{ type Subst
\begin{code}
type Subst = Map TyVar Type
\end{code}

The implementation of |substComp| uses the fact that |Map.union| is left-biased.
\begin{code}
substComp :: Subst -> Subst -> Subst
substComp subst1 subst2 = Map.map (substType subst2) subst1 `Map.union` subst2
\end{code}

\begin{code}
pprSubst :: Subst -> SDoc
pprSubst = ppr . Map.toList
\end{code}

\begin{code}
substEq :: Subst -> Subst -> Bool
substEq su1 su2 = h (Map.toAscList su1) (Map.toAscList su2)
  where
    h [] [] = True
    h [] _  = False
    h _  [] = False
    h ((x,t) : pxs) ((y,u) : pys) = x == y && tcEqType t u && h pxs pys
\end{code}
%}}}

%{{{ substIfChType :: Subst -> Type -> Maybe Type
\begin{code}
substIfChTyVar :: Subst -> TyVar -> Maybe Type
substIfChTyVar subst tv = Map.lookup tv subst
\end{code}

\begin{code}
substIfChType :: Subst -> Type -> Maybe Type
\end{code}

%{{{ {- bypass |tcView| -}
\begin{verbatim}
{-
\end{verbatim}

Try to bypass |tcView| to preserve type synonoyms:
\edcomm{WK}{Apparently not so easy.}

\begin{spec}
substType subst ty = case ty of
  TyVarTy tv -> case Map.lookup tv subst of
      Just ty2 -> ty2
      Nothing -> ty
  FunTy f a -> FunTy (substType subst f) (substType subst a)
  TyConApp tc tys -> TyConApp tc $ substTypes subst tys
  AppTy f a -> mkAppTy (substType subst f) (substType subst a)
  PredTy pr -> PredTy $ substPredType subst pr
  ForAllTy tv ty' -> let subst' = Map.delete tv subst
      in if tv `freeInSubst` subst'
         then error $ "substType captures: " ++ showSDoc (ppr tv)
         else ForAllTy tv $ substType subst' ty'
\end{spec}

\begin{verbatim}
-}
\end{verbatim}
%}}}

\begin{code}
substIfChType subst ty = case tcGetTyVar_maybe ty of
  Just tv -> substIfChTyVar subst tv
  Nothing -> let
   (forallTvs, theta, ty') = tcSplitSigmaTy ty
   substVars = tcTyVarsOfTypes $ Map.elems subst
   badBinders = filter (`elemVarSet` substVars) forallTvs
   forallTvs' = if null badBinders then forallTvs
      else error $ "substType captures: " ++ showSDoc (ppr badBinders)
   in ifChBin (mkSigmaTy forallTvs') (substIfChTheta subst)
      ((if null forallTvs && null theta then substIfChType1 else substIfChType) subst) theta ty'

substIfChType1 :: Subst -> Type -> Maybe Type
substIfChType1 subst ty = case tcSplitFunTy_maybe ty of
    Just (f, a) -> ifChBin mkFunTy (substIfChType subst) (substIfChType subst) f a
    Nothing -> case tcSplitTyConApp_maybe ty of
      Just (tc, tys) -> fmap (mkTyConApp tc) $ substIfChTypes subst tys
      Nothing -> case tcSplitAppTy_maybe ty of
        Just (f, a) -> ifChBin mkAppTy (substIfChType subst) (substIfChType subst) f a
        Nothing -> error $ "substIfChType: unexpected type: " ++ showSDoc (ppr ty)
\end{code}

\begin{code}
substIfChTypes :: Subst -> [Type] -> Maybe [Type]
substIfChTypes = ifChList . substIfChType

substIfChMap :: Ord a => Subst -> Map a Type -> Maybe (Map a Type)
substIfChMap = ifChMap . substIfChType
\end{code}

\begin{code}
substIfChPredType :: Subst -> PredType -> Maybe PredType
substIfChPredType subst (ClassP c tys) = fmap (ClassP c) $ substIfChTypes subst tys
substIfChPredType subst (IParam param ty) = fmap (IParam param) $ substIfChType subst ty
substIfChPredType subst (EqPred ty1 ty2) = ifChBin EqPred (substIfChType subst) (substIfChType subst) ty1 ty2
\end{code}

\begin{code}
substIfChTheta :: Subst -> [PredType] -> Maybe [PredType]
substIfChTheta = ifChList . substIfChPredType
\end{code}
%}}}

%{{{ substType :: Subst -> Type -> Type
\begin{code}
substTyVar :: Subst -> TyVar -> Type
substTyVar subst tv = case Map.lookup tv subst of
      Just ty2 -> ty2
      Nothing -> mkTyVarTy tv
\end{code}

\begin{code}
substType :: Subst -> Type -> Type
substType = forceIfCh . substIfChType
\end{code}

\begin{code}
substTypes :: Subst -> [Type] -> [Type]
substTypes = forceIfCh . substIfChTypes

substMap :: Subst -> Map a Type -> Map a Type
substMap = Map.map . substType     -- using |Map.map| saves |Ord a| and is likely much more time efficient.
\end{code}

\begin{code}
substPredType :: Subst -> PredType -> PredType
substPredType = forceIfCh . substIfChPredType
\end{code}

\begin{code}
substTheta :: Subst -> [PredType] -> [PredType]
substTheta = forceIfCh . substIfChTheta
\end{code}
%}}}

%{{{ unifySubsts
|unifySubsts| really calculates the pushout of two substitutions,
which are assumed to both start from |dom|.
\begin{code}
unifySubsts :: Subst -> Subst -> Maybe ((Subst, Subst), Subst)
unifySubsts su1 su2 = let
    rightOnly (y, u) = (mkTyVarTy y, u)
    leftOnly  (x, t) = (t, mkTyVarTy x)
    h [] pys = map rightOnly pys
    h pxs [] = map leftOnly pxs
    h pxs@(px@(x, t) : pxs') pys@(py@(y,u) : pys') = case compare x y of
      EQ -> (t,u) : h pxs' pys'
      LT -> leftOnly px : h pxs' pys
      GT -> rightOnly py : h pxs pys'
    dom = Map.keysSet su1 `Set.union` Map.keysSet su2
    inDom k _ = k `Set.member` dom
    ps = h (Map.toAscList su1) (Map.toAscList su2)
  in do
    p@(s1, s2) <- unify' Map.empty Map.empty ps
    let  su1' = Map.filterWithKey inDom $ su1 `substComp` s1
         su2' = Map.filterWithKey inDom $ su2 `substComp` s2
    if substEq su1' su2' then Just (p, su1')
         else error . showSDoc . withPprStyle defaultDumpStyle $ vcat
           [ text "unifySubsts:"
           , pprSubst su1
           , pprSubst su2
           , text "completed:"
           , vcat $ map ppr ps
           , text "->"
           , pprSubst su1'
           , pprSubst su2'
           , text "via"
           , pprSubst s1
           , pprSubst s2
           ]
\end{code}
%}}}

%{{{ unify
\begin{code}
unify :: Type -> Type -> Maybe ((Subst, Subst), Type)
unify ty1 ty2 = case unify' Map.empty Map.empty [(ty1, ty2)] of
  Nothing -> Nothing
  Just p@(su1, su2) -> let 
      ty1' = substType su1 ty1
      ty2' = substType su2 ty2 -- for sanity check
    in if tcEqType ty1' ty2' then Just (p, ty1')
       else error . showSDoc . withPprStyle defaultDumpStyle $ vcat
           [ text "unify:"
           , ppr ty1
           , ppr ty2
           , text "->"
           , ppr ty1'
           , ppr ty2'
           , text "via"
           , pprSubst su1
           , pprSubst su2
           ]
\end{code}
%}}}

%{{{ unify' --- worker
The returned substitutions are ``domain-precise'',
i.e., they contain ``identical'' mappings
(which are redundant for substitution application)
for all variables in their domain
for which there is no other image.

The advantage of domain-precise substitutions
is that |substComp|, used on consecutive substitutions,
does not change the domain.

\begin{code}
unify' :: Subst -> Subst
  -> [(Type, Type)] -> Maybe (Subst, Subst)
unify' su1 su2 [] = Just (su1, su2)
unify' su1 su2 ((ty1, ty2) : tps) = case tcGetTyVar_maybe ty1 of
  Just tv1 -> let
                   incr   = Map.singleton tv1 ty2
                   sIncr  = substType incr
                   su1'   = su1 `substComp` incr
                   (tps', su2')   = case ifChBin (,) (ifChList . ifChSecond $ substIfChType incr) (substIfChMap incr) tps su2 of
                     Nothing -> (tps, su2)
                     Just ch -> second (Map.insert tv1 ty2) ch
                   result f = unify' su1' (f su2') $ map (sIncr *** sIncr) tps'
    in case tcGetTyVar_maybe ty2 of
      Just tv2 ->  result (Map.insert tv2 ty2)
      _        ->  if tv1 `freeInType` ty2 -- occur check
                   then mzero else result id
  Nothing -> case tcGetTyVar_maybe ty2 of
    Just tv2 -> if tv2 `freeInType` ty1 -- occur check
                then mzero
                else let
                   incr   = Map.singleton tv2 ty1
                   sIncr  = substType incr
                   su2'   = su2 `substComp` incr
                   su1'   = Map.map sIncr su1
                 in unify' su1' su2' $ map (sIncr *** sIncr) tps
    Nothing -> case tcSplitFunTy_maybe ty1 of
      Just (f1, a1) -> case tcSplitFunTy_maybe ty2 of
        Just (f2, a2) -> unify' su1 su2 $ (f1,f2) : (a1,a2) : tps
        Nothing -> mzero -- mismatch
      Nothing -> case tcSplitTyConApp_maybe ty1 of
        Just (tc1, tys1) -> case tcSplitTyConApp_maybe ty2 of
          Just (tc2, tys2)  | showSDoc (ppr tc1) == showSDoc (ppr tc2)
                            -> unify' su1 su2 (zip tys1 tys2 ++ tps)
          _ -> mzero -- mismatch
        Nothing -> case tcSplitAppTy_maybe ty1 of
          Just (f1, a1) -> case tcSplitAppTy_maybe ty2 of
            Just (f2, a2) -> unify' su1 su2 $ (f1,f2) : (a1,a2) : tps
            Nothing -> mzero -- mismatch
          Nothing -> error $ "unify: unexpected type: " ++ showSDoc (ppr (ty1,ty2))
                     -- mzero -- no other options?
\end{code}
%}}}

%{{{ myThetaTyVars, myPredTyVars
\edcomm{WK}{For |myThetaTyVars| and |myPredTyVars|,
I use |tyVarsOfType*|,
whereas elsewhere I always use |tcTyVarsOfType*|,
and it seems to be necessary to do it this way.
-- See comment at |freeInType'|.}

\begin{code}
myThetaTyVars :: [PredType] -> TcTyVarSet
myThetaTyVars = foldr (unionVarSet . myPredTyVars) emptyVarSet

myPredTyVars :: PredType -> TcTyVarSet
myPredTyVars (ClassP _c tys)     = tyVarsOfTypes tys
myPredTyVars (IParam _param ty)  = tyVarsOfType  ty
myPredTyVars (EqPred ty1 ty2)    = tyVarsOfTypes [ty1, ty2]
\end{code}
%}}}

\begin{code}
elemTheta :: PredType -> [PredType] -> Bool
elemTheta = any . tcEqPred
\end{code}

\begin{code}
stripSigma :: Type -> Type
stripSigma ty = let (forallTvs, theta, ty') = tcSplitSigmaTy ty
  in if null forallTvs && null theta then ty' else stripSigma ty'
\end{code}

%{{{ typeTyCons
\begin{code}
typeTyCons, typeTyCons1 :: Type -> Set TyCon
typeTyCons ty = case tcGetTyVar_maybe {- |getTyVar_maybe| -} ty of
  Just _tv -> Set.empty
  Nothing -> let
     (forallTvs, theta, ty') = tcSplitSigmaTy ty
   in thetaTyCons theta `Set.union`
      (if null forallTvs && null theta then typeTyCons1 else typeTyCons) ty'

typeTyCons1 ty = case tcSplitFunTy_maybe {- |splitFunTy_maybe| -} ty of
    Just (f, a) -> typesTyCons [f,a]
    Nothing -> case tcSplitTyConApp_maybe {- |splitTyConApp_maybe| -} ty of
      Just (tc, tys) -> Set.insert tc $ typesTyCons tys
      Nothing -> case tcSplitAppTy_maybe {- |splitAppTy_maybe| -} ty of
        Just (f, a) -> typesTyCons [f,a]
        Nothing -> error $ "typeTyCons: unexpected type: " ++ showSDoc (ppr ty)

typesTyCons :: [Type] -> Set TyCon
typesTyCons = Set.unions . map typeTyCons
\end{code}

\begin{code}
thetaTyCons :: ThetaType -> Set TyCon
thetaTyCons = Set.unions . map predTyCons

predTyCons :: PredType -> Set TyCon
predTyCons (ClassP _c tys) = typesTyCons tys
predTyCons (IParam _param ty) = typeTyCons ty
predTyCons (EqPred ty1 ty2) = typesTyCons [ty1, ty2]
\end{code}
%}}}

\begin{code}
data TyPos
  = FunTyPos Bool           -- right?
  | TyConAppPos TyCon Int   -- starting at 0
  | AppTyPos Bool           -- right?
  deriving (Eq, Ord)
\end{code}

%{{{ typePoss :: Type -> [([TyPos],Type)]
\begin{code}
typePoss :: Type -> [([TyPos],Type)]
typePoss ty = prependTypePoss id ty []
\end{code}

\begin{code}
prependTypePoss :: ([TyPos] -> [TyPos]) -> Type -> [([TyPos],Type)] -> [([TyPos],Type)]
prependTypePoss w ty = ((w [],ty) :) . case tcGetTyVar_maybe ty of
  Just _tv -> id
  Nothing -> case tcSplitFunTy_maybe ty of
    Just (f, a)  ->  prependTypePoss (w . (FunTyPos False :)) f
                 .   prependTypePoss (w . (FunTyPos True :)) a
    Nothing -> case tcSplitTyConApp_maybe ty of
      Just (tc, tys) -> foldr (.) id $ zipWith f [0..] tys
        where f i = prependTypePoss (w . (TyConAppPos tc i :))
      Nothing -> case tcSplitAppTy_maybe ty of
        Just (f, a)  ->  prependTypePoss (w . (AppTyPos False :)) f
                     .   prependTypePoss (w . (AppTyPos True :)) a
        Nothing -> error $ "prependTypePoss: unexpected type: " ++ showSDoc (ppr ty)
\end{code}
%}}}

%{{{ tyVarPoss :: TyVar -> Type -> [[TyPos]]
\begin{code}
tyVarPoss :: TyVar -> Type -> [[TyPos]]
tyVarPoss tv ty = prependTyVarPoss id tv ty []
\end{code}

\begin{code}
prependTyVarPoss :: ([TyPos] -> [TyPos]) -> TyVar -> Type -> [[TyPos]] -> [[TyPos]]
prependTyVarPoss w tv ty = case tcGetTyVar_maybe ty of
  Just tv' -> if tv == tv'
    then (w [] :)
    else id
  Nothing -> case tcSplitFunTy_maybe ty of
    Just (f, a)  ->  prependTyVarPoss (w . (FunTyPos False :)) tv f
                 .   prependTyVarPoss (w . (FunTyPos True :)) tv a
    Nothing -> case tcSplitTyConApp_maybe ty of
      Just (tc, tys) -> foldr (.) id $ zipWith f [0..] tys
        where f i = prependTyVarPoss (w . (TyConAppPos tc i :)) tv
      Nothing -> case tcSplitAppTy_maybe ty of
        Just (f, a)  ->  prependTyVarPoss (w . (AppTyPos False :)) tv f
                     .   prependTyVarPoss (w . (AppTyPos True :)) tv a
        Nothing -> error $ "prependTyVarPoss " ++ showSDoc (ppr tv) ++ ": unexpected type: " ++ showSDoc (ppr ty)
\end{code}
%}}}

%{{{ matchSubTypes
\begin{code}
matchSubTypes :: Type -> Type -> [([TyPos], Subst)]
matchSubTypes patTy = let
  matchPos (typos, ty') = fmap ((,) typos) $ matchType patTy ty'
  in mapMaybe matchPos . typePoss

matchesSubType :: Type -> Type -> Bool
matchesSubType patTy = not . null . matchSubTypes patTy
\end{code}
%}}}

%{{{ atTyPos
\begin{code}
atTyPoss :: [TyPos] -> Type -> Maybe Type
atTyPoss [] ty = Just ty
atTyPoss (p : ps) ty = atTyPos p ty >>= atTyPoss ps

atTyPos :: TyPos -> Type -> Maybe Type
atTyPos (FunTyPos right) ty = case tcSplitFunTy_maybe ty of
    Just (f, a)  -> Just $ if right then a  else f
    Nothing -> Nothing
atTyPos (TyConAppPos tc k) ty = case tcSplitTyConApp_maybe ty of
    Just (tc', tys) | tc == tc' && k < length tys  -> Just $ tys !! k
    _ -> Nothing
atTyPos (AppTyPos right) ty = case tcSplitAppTy_maybe ty of
    Just (f, a)  ->  Just $ if right then a  else f
    Nothing -> Nothing
\end{code}
%}}}

%{{{ EMACS lv
% Local Variables:
% folded-file: t
% fold-internal-margins: 0
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% end:
%}}}
