-- Copyright 2006 by Wolfram Kahl, all rights reserved


-- The ``SImple IMperative Programming Language'' SImPL

module SImPL
  ( module SImPLTokens    -- re-export to save imports on users
  , module SImPL          -- hiding nothing from current module
                    -- (necessary to make explicit when using an export list)
  ) where

import SImPLTokens

-- SImPL Abstract Syntax Types:

data Program     = MkProgram     [Declaration] Block
data Declaration = MkDeclaration Variable      Type
data Statement   = MkBlock     Block
                 | Assignment  Variable   Expression
                 | Conditional Expression Statement Statement
                 | Loop        Expression Statement
type Block       = [Statement]
data Expression  = Var    Variable
                 | Value  Literal
                 | Binary BinOp Expression Expression
                 | Unary  UnaryOp Expression

-- Show instances, programmed for easy understanding:

instance Show Declaration where
  show (MkDeclaration var ty) = show ty ++ ' ' : var ++ ";"

showExpr :: Expression -> String
showExpr (Var v) = v
showExpr (Value lit) = show lit
showExpr (Binary op e1 e2) =
   '(' : showExpr e1 ++ ' ' : show op ++ ' ' : showExpr e2 ++ ")"
showExpr (Unary op e) = show op ++ show e

instance Show Expression where show = showExpr

-- for statements, we generate lines first, so we can produce indentation:

linesFromStmts :: [Statement] -> [String]
linesFromStmts = concatMap linesFromStmt

linesFromBlock :: [Statement] -> [String]
linesFromBlock = braceLines . linesFromStmts

prefixLines :: String -> [String] -> [String]
prefixLines prefix [] = [prefix]
prefixLines prefix [s] = [prefix ++ s]
prefixLines prefix (s:ss) = (prefix ++ s) : map (indent ++) ss
  where indent = replicate (length prefix) ' '

braceLines :: [String] -> [String]
braceLines [] = ["{}"]
braceLines [s] = ["{ " ++ s ++ " }"]
braceLines ss = prefixLines "{ " ss ++ ["}"]


linesFromStmt :: Statement -> [String]
linesFromStmt (MkBlock b) = linesFromBlock b
linesFromStmt (Assignment  var e) = [var ++ " := " ++ show e ++ ";"]
linesFromStmt (Conditional cond sThen sElse) =
  ("if " ++ show cond)
  :  prefixLines "then " (linesFromStmt sThen)
  ++ prefixLines "else " (linesFromStmt sElse)
  ++ ["fi"]
linesFromStmt (Loop        cond body) =
  ("while " ++ show cond)
  : prefixLines "do " (linesFromStmt body)

instance Show Statement where
  show = unlines . linesFromStmt


instance Show Program where
  show (MkProgram decls body) = unlines $
    "program" : map (("  " ++) . show) decls ++ linesFromBlock body
