-- Copyright 2006 by Wolfram Kahl, all rights reserved

module SImPLParser where

import SImPL
import SImPLLexer


readSImPL :: FilePath -> IO Program
readSImPL file = do
    s <- readFile file
    case parseProgram $ lexSImPL s of
      Just (p, toks) -> case toks of
         [] -> return p
         _  -> fail $ "Garbage after end of program: " ++
                       take 40 (unwords $ map show toks)
      Nothing -> fail $ "Could not parse SImPL program from file " ++ file

-- For syntax checking and pretty printing:
prettySImPL :: FilePath -> IO ()
prettySImPL file = do
    prog <- readSImPL file
    print prog
  `catch`
    \ err -> putStrLn $ "Error in file " ++ file ++ ": " ++ show err

type Parser a = [Token] -> Maybe (a, [Token])

-- Program ::= program Declaration* '{' Statement* '}'
parseProgram :: Parser Program
parseProgram (TokPROGRAM : rest1) = case parseMany parseDeclaration rest1 of
    Just (decls, TokSep '{' : rest2) -> case parseMany parseStatement rest2 of
        Just (ss, TokSep '}' : rest3) -> Just (MkProgram decls ss, rest3)
        _                             -> Nothing
    _ -> Nothing
parseProgram _ = Nothing

-- Declaration ::= Type Variable ';'
parseDeclaration :: Parser Declaration
parseDeclaration (TokType ty : TokIdent var : TokSep ';' : rest) =
  Just (MkDeclaration var ty, rest)
parseDeclaration _ = Nothing


-- Statement ::= Variable := Expression ';'
--             | '{' Statement* '}'
--             | if Expression then Statement else Statement
--             | while Expression do Statement
parseStatement ::  Parser Statement
parseStatement (TokIdent var : TokASSIGN : rest1) = case parseExpression rest1 of
    Just (e, TokSep ';' : rest2) -> Just (Assignment var e, rest2)
    _               -> Nothing
parseStatement (TokSep '{' : rest1) = case parseMany parseStatement rest1 of
    Just (ss, TokSep '}' : rest2) -> Just (MkBlock ss, rest2)
    _                             -> Nothing
parseStatement (TokIF : rest1) = case parseExpression rest1 of
    Just (cond, TokTHEN : rest2) -> case parseStatement rest2 of
        Just (sThen, TokELSE : rest3) -> case parseStatement rest3 of
            Just (sElse, rest4) -> Just (Conditional cond sThen sElse, rest4)
            _ -> Nothing
        _ -> Nothing
    _ -> Nothing
parseStatement (TokWHILE : rest1) = case parseExpression rest1 of
    Just (cond, TokDO : rest2) -> case parseStatement rest2 of
        Just (body, rest3) -> Just (Loop cond body, rest3)
        _ -> Nothing
    _ -> Nothing
parseStatement _ = Nothing

parseMany :: Parser a -> Parser [a]
parseMany p input = case p input of
    Nothing -> Just ([], input)
    Just (s, rest1) -> case parseMany p rest1 of
        Nothing -> Just ([s], rest1)
        Just (ss, rest2) -> Just (s : ss, rest2)


-- Expression ::= RelExpr
--              | RelExpr BoolOp Expression    -- right recursion
parseExpression ::  Parser Expression
parseExpression input = case parseRelExpr input of
    Nothing -> Nothing
    Just (e1, rest1) -> case rest1 of
        TokBinOp op@(MkBoolOp _) : rest2 -> case parseExpression rest2 of
            Nothing -> Nothing                         -- Syntax Error!
            Just (e2, rest3) -> Just ( Binary op e1 e2,  rest3)
        _ -> Just (e1, rest1)


-- RelExpr ::= Expr
--           | Expr RelOp Expr                -- no recursion
parseRelExpr  ::  Parser Expression
parseRelExpr input = case parseExpr input of
    Nothing -> Nothing
    Just (e1, rest1) -> case rest1 of
        TokBinOp op@(MkRelOp _) : rest2 -> case parseExpr rest2 of
            Just (e2, rest3) -> Just (Binary op e1 e2, rest3)
            Nothing -> Nothing                         -- Syntax Error!
        _ -> Just (e1, rest1)

-- Only for illustration:
-- right-recursive parsing producing right-recursive syntax tree
parseExpr1 :: Parser Expression
parseExpr1 input = case parseTerm input of
  Nothing -> Nothing
  Just (e1, rest1) -> case parseAddOp rest1 of
    Nothing -> Just (e1, rest1)
    Just (op, rest2) -> case parseExpr1 rest2 of
      Nothing -> Nothing                         -- Syntax Error!
      Just (e2, rest3) -> Just ( Binary op e1 e2,  rest3)

-- Only for illustration:
-- left-recursive parsing producing no syntax tree ;-)
parseExpr2 :: Parser Expression
parseExpr2 input = case parseExpr2 input of
  Nothing -> parseTerm input
  Just (e1, rest1) -> case parseAddOp rest1 of
    Nothing -> Just (e1, rest1)
    Just (op, rest2) -> case parseTerm rest2 of
      Nothing -> parseTerm input
      Just (e2, rest3) -> Just ( Binary op e1 e2,  rest3)

-- Expr ::= Term {AddOp Term}*
parseExpr ::  Parser Expression
parseExpr = parseTerms id

-- right-recursive parsing producing left-recursive syntax tree
parseTerms :: (Expression -> Expression) -> Parser Expression
parseTerms wrap input = case parseTerm input of
    Nothing -> Nothing
    Just (e1, rest1) -> case parseAddOp rest1 of
        Nothing -> Just (wrap e1, rest1)
        Just (op, rest2) -> parseTerms (Binary op (wrap e1)) rest2


-- Term ::= Factor {AddOp Factor}*
parseTerm :: [Token] -> Maybe (Expression, [Token])
parseTerm = parseFactors id

-- right-recursive parsing producing left-recursive syntax tree
parseFactors :: (Expression -> Expression) -> Parser Expression
parseFactors wrap input = case parseFactor input of
    Nothing -> Nothing
    Just (e1, rest1) -> case parseMultOp rest1 of
        Nothing -> Just (wrap e1, rest1)
        Just (op, rest2) -> parseFactors (Binary op e1) rest2

-- Factor ::= Variable
--          | Literal
--          | ( Expression ) 
parseFactor :: Parser Expression
parseFactor ((TokIdent v)  : rest) = Just (Var v, rest)
parseFactor ((TokLit lit) : rest) = Just (Value lit, rest)
parseFactor ((TokSep '(')  : rest) = case parseExpression rest of
    Just (e, TokSep ')' : rest1) -> Just (e, rest1)
    _                         -> Nothing
parseFactor _                   = Nothing

-- AddOp ::= Plus | Minus
parseAddOp :: Parser BinOp
parseAddOp (TokBinOp op@(MkArithOp Plus ) : rest) = Just (op, rest)
parseAddOp (TokBinOp op@(MkArithOp Minus) : rest) = Just (op, rest)
parseAddOp _                  = Nothing

-- MultOp ::= Times | Div
parseMultOp :: Parser BinOp
parseMultOp (TokBinOp op@(MkArithOp Times) : rest) = Just (op, rest)
parseMultOp (TokBinOp op@(MkArithOp Div  ) : rest) = Just (op, rest)
parseMultOp _                  = Nothing


-- Commands for interactive expression evaluator

data Command = Let Variable Expression
             | Eval Expression

parseLet :: Parser Command
parseLet (TokIdent "let" : TokIdent v : TokSep '=' : rest) =
  case parseExpr rest of
    Nothing -> Nothing
    Just (e, rest') -> Just (Let v e, rest')
parseLet _ = Nothing

parseCommand :: Parser Command
parseCommand toks = case parseLet toks of
  Nothing -> mapParser Eval parseExpr toks
  j -> j

mapParser :: (a -> b) -> Parser a -> Parser b
mapParser f p toks = case p toks of
    Nothing        -> Nothing
    Just (x, rest) -> Just (f x, rest)
