/*************************************************************************
  FILE:  BddHiscLowSub2.cpp
  DESCR: Verify this low-level interface (Command-pair) 
  AUTH:  Raoguang Song
  Supervisor: Dr. Ryan Leduc
  DATE:  (C) Jan, 2006
*************************************************************************/

#include "BddHiscLowSub.h"
#include "BddHiscSub.h"
#include "BddHiscPubfunc.h"
#include "BddHiscType.h"
#include <fdd.h>
#include "BddHiscProject.h"
#include "BddHiscErrmsg.h"
#include <string>

using namespace std;

namespace BDDHISC
{

extern CProject *pPrj;

/*
 * DESCR:   Verify if this low-level interface is a command-pair interface
 * PARA:    None
 * RETURN:  0: This low-level interface is a command-pair interface
 *          < 0: Not a command-pair interface
 * ACCESS: private
 */
int CLowSub::CheckIntf()
{
    int iRet = 0;
    string sErr = "Interface for low-level ";
    sErr += this->GetSubName();
    sErr += ": ";
    bddPair *pPair = bdd_newpair();
    SetBddPairs(pPair, m_bddIVarPrim, m_bddIVar);
    bdd P_Ij = bddfalse;
    bdd P_Rj = bddfalse;
    bdd P_Aj = bddfalse;
    bdd P_RAj = bddfalse;
    bdd P_ARj = bddfalse;
    bdd P_Ijm = bddfalse;
    
    try
    {
        if (m_bddIntfInit == bddfalse)
        {
            pPrj->SetErr(sErr + "No reachable states.", HISC_BAD_INTERFACE);
            throw -1;
        }
        P_Ij = intf_r(bddtrue, iRet);
        if (iRet < 0)
            throw -1;
        P_Ijm = m_bddIntfMarking & P_Ij;
        
        for (int i = 1; i <= (unsigned short)(m_usiMaxCon[A_EVENT] + 1); i += 2)
            P_Rj |= bdd_exist(m_pbdd_ATrans[1][(i - 1)/2], m_bddIVar);
        for (int i = 2; i <= (unsigned short)(m_usiMaxUnCon[A_EVENT] + 1); i += 2)
            P_Rj |= bdd_exist(m_pbdd_ATrans[0][(i - 2)/2], m_bddIVar);
        P_Rj = bdd_replace(P_Rj, pPair);
        P_Rj &= P_Ij;
        P_Rj |= m_bddIntfInit;

        for (int i = 1; i <= (unsigned short)(m_usiMaxCon[R_EVENT] + 1); i += 2)
            P_Aj |= bdd_exist(m_pbdd_RTrans[1][(i - 1)/2], m_bddIVar);
        for (int i = 2; i <= (unsigned short)(m_usiMaxUnCon[R_EVENT]); i += 2)
            P_Aj |= bdd_exist(m_pbdd_RTrans[0][(i - 2)/2], m_bddIVar);
        P_Aj = bdd_replace(P_Aj, pPair);
        P_Aj &= P_Ij;
        
        for (int i = 1; i <= (unsigned short)(m_usiMaxCon[A_EVENT] + 1); i += 2)
            P_RAj |= bdd_exist(m_pbdd_ATrans[1][(i - 1)/2] & P_Rj, m_bddIVar);
        for (int i = 2; i <= (unsigned short)(m_usiMaxUnCon[A_EVENT] + 1); i += 2)
            P_RAj |= bdd_exist(m_pbdd_ATrans[0][(i - 2)/2] & P_Rj, m_bddIVar);
        
        for (int i = 1; i <= (unsigned short)(m_usiMaxCon[R_EVENT] + 1); i += 2)
            P_ARj |= bdd_exist(m_pbdd_RTrans[1][(i - 1)/2] & P_Aj, m_bddIVar);
        for (int i = 2; i <= (unsigned short)(m_usiMaxUnCon[R_EVENT]); i += 2)
            P_ARj |= bdd_exist(m_pbdd_RTrans[0][(i - 2)/2] & P_Aj, m_bddIVar);
            
        if (P_RAj != bddfalse || P_ARj != bddfalse || P_Rj != P_Ijm)
        {
            pPrj->SetErr(sErr + "Not a command-pair interface.", HISC_BAD_INTERFACE);
            throw -1;
        }
    }
    catch(int iErr)
    {
        iRet = iErr;
    }
    bdd_freepair(pPair);
    pPair = NULL;
    return iRet;
}


/*
 * DESCR:   Compute R(GIj, P) 
 * PARA:    bddP: P (input)
            iErr: returned error code
 * RETURN:  R(GIj, P)
 * ACCESS: private
 */
bdd CLowSub::intf_r(const bdd &bddP, int &iErr)
{
    bdd bddK = bddP & m_bddIntfInit;
    bdd bddK1 = bddfalse;
    bdd bddKStep = bddfalse;
    bddPair *pPair = bdd_newpair();
    SetBddPairs(pPair, m_bddIVarPrim, m_bddIVar);
    iErr = 0;
        
    try
    {
        while (bddK != bddK1)
        {
            bddK1 = bddK;
            for (int i = 1; i <= (unsigned short)(m_usiMaxCon[R_EVENT] + 1); i += 2)
            {
                bddKStep = bdd_exist(m_pbdd_RTrans[1][(i - 1)/2] & bddK1, m_bddIVar);
                bddKStep = bdd_replace(bddKStep, pPair);
                bddK |= bddKStep;
            }
            for (int i = 2; i <= (unsigned short)(m_usiMaxUnCon[R_EVENT]); i += 2)
            {
                bddKStep = bdd_exist(m_pbdd_RTrans[0][(i - 2)/2] & bddK1, m_bddIVar);
                bddKStep = bdd_replace(bddKStep, pPair);
                bddK |= bddKStep;
            }
            for (int i = 1; i <= (unsigned short)(m_usiMaxCon[A_EVENT] + 1); i += 2)
            {
                bddKStep = bdd_exist(m_pbdd_ATrans[1][(i - 1)/2] & bddK1, m_bddIVar);
                bddKStep = bdd_replace(bddKStep, pPair);
                bddK |= bddKStep;
            }
            for (int i = 2; i <= (unsigned short)(m_usiMaxUnCon[A_EVENT] + 1); i += 2)
            {
                bddKStep = bdd_exist(m_pbdd_ATrans[0][(i - 2)/2] & bddK1, m_bddIVar);
                bddKStep = bdd_replace(bddKStep, pPair);
                bddK |= bddKStep;
            }
        }
    }
    catch(int)
    {
        string sErr = "Interface for low-level ";
        sErr += this->GetSubName();
        sErr += ": Error during computing interface reachable states.";
        pPrj->SetErr(sErr, HISC_LOWERR_REACH);
        iErr = -1;
        bdd_freepair(pPair);
        pPair = NULL;
        return bddfalse;
    }
    bdd_freepair(pPair);
    pPair = NULL;
    return bddK;
}


} //end of namespace BDDHISC


