/*	Author:		Magdin Stoica
	Supervisor: Dr. Ryan Leduc

	Project created in conformity with the requirements for the Degree of Master of Engineering in Software Engineering,
	Computing and Software Department,
	McMaster University
	2003 - 2007
*/
 
#include <QMessageBox>
#include "OutputUiPart.h"
#include "MultiNonBlockingAlgo.h"
//#include "CtrlStdAlgo.h"
#include "ProjIntegrityAlgo.h"
#include "MultiCtrlAlgo.h"
#include "DesProject.h"
#include "LdInterfaceCheckAlgo.h"
#include "IConsistAlgo.h"
#include "LwNonBlockingAlgo.h"
#include "LwCtrlAlgo.h"
#include "DesInterface.h"
#include "DesHierProject.h"
#include "IConsistAnsAcceptAlgo.h"
#include "LowSubsysIConsistAlgo.h"
#include "HighSubsysNonBlockAlgo.h"
#include "LowSubsysNonBlockAlgo.h"
#include "HighSubsysCtrlAlgo.h"
#include "LowSubsysCtrlAlgo.h"
#include "FlatProjectPropCheck.h"
#include "HierProjectPropCheck.h"
#include "MeetAlgo.h"


#include "BddHiscMain.h"

// for debug
// #include <iostream>

namespace DESpot
{

const QString OutputUiPart::cProjIntegOutputTitle = "Project Integrity";
const QString OutputUiPart::cNonBlockOutputTitle = "Nonblocking";
const QString OutputUiPart::cCtrlOutputTitle = "Controllability";
const QString OutputUiPart::cMeasureOutputTitle = "System Measurements";
const QString OutputUiPart::cInterfConsistOutputTitle = "LD Interface Consistency";
const QString OutputUiPart::cLwNonBlockOutputTitle = "LD Level-Wise Nonblocking";
const QString OutputUiPart::cLwCtrlOutputTitle = "LD Level-Wise Controllable";
const QString OutputUiPart::cSubsysInterfConsistOutputTitle = "Subsystem LD I-Consistency";
const QString OutputUiPart::cSubsysNonBlockingOutputTitle = "Subsystem LD Nonblocking";
const QString OutputUiPart::cSubsysCtrlOutputTitle = "Subsystem LD Controllability";
const QString OutputUiPart::cLdInterfCheckTitle = "Interface LD Check";
const QString OutputUiPart::cLastOutputTitle = "Last Output";

OutputUiPart::OutputUiPart(Ui::FlatProjectEditorUI* pUiContainer, DesProject* pProject) :
		m_project(pProject)
{
	initWidgets(pUiContainer);

	setupConnections();

	initOutputSelectionWidget();
}

//_________________________________________________________________________________________________

OutputUiPart::OutputUiPart(Ui::HierProjectEditorUI* pUiContainer, DesProject* pProject):
		m_project(pProject)
{
	initWidgets(pUiContainer);

	setupConnections();

	initOutputSelectionWidget();
}

//_________________________________________________________________________________________________

OutputUiPart::~OutputUiPart()
{
}

//_________________________________________________________________________________________________

template<class T>
void OutputUiPart::initWidgets(T* pUiContainer)
{
	m_showOutputForWidg = pUiContainer->m_showOutputForWidg;
	m_copyBtn = pUiContainer->m_copyBtn;
	m_clearBtn = pUiContainer->m_clearBtn;
	m_wrapBtn = pUiContainer->m_wrapBtn;
	m_outputWidget = pUiContainer->m_outputWidg;

}

//_________________________________________________________________________________________________

void OutputUiPart::setupConnections()
{
	connect(m_showOutputForWidg, SIGNAL(currentIndexChanged(int)), this, SLOT(onShowOutputFor(int)));
	connect(m_clearBtn, SIGNAL(clicked()), m_outputWidget, SLOT(clear()));
	connect(m_outputWidget, SIGNAL(copyAvailable(bool)), m_copyBtn, SLOT(setEnabled(bool)));
	connect(m_copyBtn, SIGNAL(clicked()), m_outputWidget, SLOT(copy()));
	connect(m_wrapBtn, SIGNAL(clicked()), this, SLOT(onToggleWrap()));
}

//_________________________________________________________________________________________________

void OutputUiPart::initOutputSelectionWidget()
{
	m_showOutputForWidg->addItem(cLastOutputTitle);
	m_showOutputForWidg->addItem(cProjIntegOutputTitle, QString("Project integrity has not been checked in this session"));
	m_showOutputForWidg->addItem(cNonBlockOutputTitle, QString("Nonblocking has not been checked in this session"));
	m_showOutputForWidg->addItem(cCtrlOutputTitle, QString("Controllability has not been checked in this session"));
	m_showOutputForWidg->addItem(cMeasureOutputTitle, QString("The size of the flat system is unknown and is not currently being calculated."));

	if (m_project->getType() == eHierProject)
	{
		m_showOutputForWidg->addItem(cInterfConsistOutputTitle, QString("LD-Interface consistency has not been checked in this session"));
		m_showOutputForWidg->addItem(cLwNonBlockOutputTitle, QString("LD Level-wise nonblocking has not been checked in this session"));
		m_showOutputForWidg->addItem(cLwCtrlOutputTitle, QString("LD Level-wise controllablility has not been checked in this session"));
		m_showOutputForWidg->addItem(cSubsysInterfConsistOutputTitle, QString("Selected a subsystem to check whether it is interface consistent or not."));
		m_showOutputForWidg->addItem(cLdInterfCheckTitle, QString("Select an interface to check if it is LD-interface"));
		m_showOutputForWidg->addItem(cSubsysNonBlockingOutputTitle, QString("Select an interface to check if it is nonblocking"));
		m_showOutputForWidg->addItem(cSubsysCtrlOutputTitle, QString("Select an interface to check if it is controllable"));
	}
}

//_________________________________________________________________________________________________

void OutputUiPart::setOwnerTabWidget(QTabWidget* tabWidget)
{
	m_tabWidget = tabWidget;
}

//_________________________________________________________________________________________________

void OutputUiPart::onShowOutputFor(int sourceIndex)
{
	QString output = m_showOutputForWidg->itemData(sourceIndex).toString();
	m_outputWidget->setHtml(output);
}

//_________________________________________________________________________________________________

void OutputUiPart::onToggleWrap()
{
	QTextEdit::LineWrapMode crtWrapMode = m_outputWidget->lineWrapMode();
	m_outputWidget->setLineWrapMode(crtWrapMode == QTextEdit::NoWrap ? QTextEdit::WidgetWidth : QTextEdit::NoWrap);
}

//_________________________________________________________________________________________________

void OutputUiPart::saveOutput()
{
	QString output = m_outputWidget->toHtml();
	m_showOutputForWidg->setItemData(m_showOutputForWidg->currentIndex(), output);
}

//_________________________________________________________________________________________________

void OutputUiPart::saveLastOutput(LastOutputOption copyToLastOutput)
{
	QString output = m_outputWidget->toHtml();

	int lastOutputIndex = m_showOutputForWidg->findText(cLastOutputTitle);
	QString lastOutput = m_showOutputForWidg->itemData(lastOutputIndex).toString();

	//set the output to show the Last Output section
	m_showOutputForWidg->setCurrentIndex(lastOutputIndex);

	if (copyToLastOutput == OutputUiPart::eSetLastOutput)
	{
		//we're supposed to set the last output so let's clear what's there
		clearLastOutput();

		m_outputWidget->write(output);

		m_showOutputForWidg->setItemData(lastOutputIndex, output);
	}
	else if (copyToLastOutput == OutputUiPart::eAppendLastOutput)
	{
		//the output widget already contains the "last output" so we just to write the new output
		m_outputWidget->write(output);

		//append current output to the last output if we were
		lastOutput += output;

		//save the new "last" output
		m_showOutputForWidg->setItemData(lastOutputIndex, lastOutput);
	}
}

//_________________________________________________________________________________________________

DesHierProject* OutputUiPart::hierProject()
{
	if (m_project->getType() == eHierProject)
	{
		return dynamic_cast<DesHierProject*>(m_project);
	}

	throw EX("Project not hierarhical");
}

//_________________________________________________________________________________________________

void OutputUiPart::show(ProjIntegrityAlgo& integAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cProjIntegOutputTitle));
	m_outputWidget->clear();

	if (integAlgo.isValid())
	{
		if (integAlgo.getWarningCount() > 0)
		{
			QString out("<b>Project is valid - %1 warning(s)</b>");
			out = out.arg(integAlgo.getWarningCount());

			m_outputWidget->writeLine(out);

			if (showMsgBox)
			{
				QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
			}

			m_outputWidget->writeLine("");

			//display all warnings
			const IntegrityAlgo::WarningList& warnList = integAlgo.getWarningList();
			for(IntegrityAlgo::WarnIt warnIt = warnList.begin(); warnIt != warnList.end(); ++warnIt)
			{
				m_outputWidget->writeLine(QString("Warning: ") + QString::fromStdWString(*warnIt));
			}
		}
		else
		{
			if (showMsgBox)
			{
				QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot","Project is valid");
			}

			QString out("Project is valid. Verified on %1");
			out = out.arg(QString::fromStdWString(m_project->getIntegrityStamp()));
			m_outputWidget->writeLine("<b>" + out + "</b>");
		}
	}
	else
	{
		QString out("Project is not valid - %1 error(s)");
		out = out.arg(integAlgo.getErrorCount());
		out += QString(", %1 warning(s)").arg(integAlgo.getWarningCount());

		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");


		//display all warnings
		m_outputWidget->writeLine("");const IntegrityAlgo::WarningList& warnList = integAlgo.getWarningList();
		for(IntegrityAlgo::WarnIt warnIt = warnList.begin(); warnIt != warnList.end(); ++warnIt)
		{
			m_outputWidget->writeLine(QString("Warning: ") + QString::fromStdWString(*warnIt));
		}

		//display all errors
        showErrors(integAlgo.getErrorList());
	}

	saveOutput();
}

//_________________________________________________________________________________________________

void OutputUiPart::show(ProjIntegrityAlgo& integAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox/* = true*/)
{
	show(integAlgo, showMsgBox);

	saveLastOutput(copyToLastOutput);
}
//_________________________________________________________________________________________________

void OutputUiPart::show(MultiNonBlockingAlgo& nonBlockAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cNonBlockOutputTitle));

	m_outputWidget->clear();

	if (nonBlockAlgo.isNonBlocking())
	{
		QString message("System is nonblocking. Verified on %1");
		message = message.arg(QString::fromStdWString(m_project->getNonBlockingStamp()));

		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot","System is nonblocking");
		}

		m_outputWidget->writeLine("<b>" + message + "</b>");

	}
	else
	{
		m_outputWidget->writeLine("<b>System is blocking</b>");
		m_outputWidget->writeLine("The following states are blocking:");

		//go through the blockign states and write them out in the output

     int errcount;
	
      if(nonBlockAlgo.getDesNum() >1)
	  {
		  ErrorList errlst = nonBlockAlgo.getErrorList();
		  for(ErrorListIt lt=errlst.begin(); lt!=errlst.end(); lt++)
		  {
			  m_outputWidget->writeLine(QString("&nbsp;&nbsp;&nbsp;&nbsp;%1").arg(QString::fromStdWString(*lt)));
		  }

          errcount = nonBlockAlgo.getErrorCount();
	  }
	  else
	  {
		  NonBlockingAlgo::StateIteratorPtr stateIt = nonBlockAlgo.m_non_algo.createBlockingStateIterator();
		  for(stateIt->first(); stateIt->notDone(); stateIt->next())
		  {
			const DesState* blockingState = stateIt->currentItem();
			m_outputWidget->writeLine(QString("&nbsp;&nbsp;&nbsp;&nbsp;%1").arg(QString::fromStdWString(blockingState->getName())));
		  }
		  errcount=nonBlockAlgo.m_non_algo.getBlockingStateCount();
	  }

		m_outputWidget->writeLine("");

		QString message("System has %1 blocking state(s). See the output for the list of blocking states");
		QString blockingStateCount = QVariant(errcount).toString();

		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", message.arg(blockingStateCount));
		}
	}

	//save the output in the output selection widget
	saveOutput();
}

//_________________________________________________________________________________________________

void OutputUiPart::show(MultiNonBlockingAlgo& nonBlockAlgo, LastOutputOption copyToLastOutput/* = eSetLastOutput*/,
						bool showMsgBox/* = true*/)
{
	show(nonBlockAlgo, showMsgBox);

	saveLastOutput(copyToLastOutput);
}
//_________________________________________________________________________________________________


/*
void OutputUiPart::show(CtrlStdAlgo& ctrlAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cCtrlOutputTitle));
	m_outputWidget->clear();

	if (ctrlAlgo.isControllable())
	{
		QString message("System is controllable. Verified on %1");
		message = message.arg(QString::fromStdWString(m_project->getCtrlStamp()));

		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot","The system is controllable");
		}

		m_outputWidget->writeLine("<b>" + message + "</b>");
	}
	else
	{
		m_outputWidget->writeLine("<b>The system is not controllable</b>");
		QString messageTempl = "The system is uncontrollable at state (%1, %2) for event %3.";
		QString plantStateName = QString::fromStdWString(ctrlAlgo.getUnctrlBehaviour().plantState->getName());
		QString supStateName = QString::fromStdWString(ctrlAlgo.getUnctrlBehaviour().supState->getName());
		QString blockEventName = QString::fromStdWString(ctrlAlgo.getUnctrlBehaviour().event->getName());
		QString message = messageTempl.arg(plantStateName, supStateName, blockEventName);

		m_outputWidget->writeLine(message);
		m_outputWidget->writeLine("");
		m_outputWidget->writeLine("Trace to uncontrollable behaviour:");

		const CtrlStdAlgo::CounterExample& counterExample = ctrlAlgo.getCounterExample();
		for(CtrlStdAlgo::BehavIt it = counterExample.begin(); it != counterExample.end(); it++)
		{
			const CtrlStdAlgo::CtrlBehaviour& behav = *it;

			QString plantStateName = QString::fromStdWString(behav.plantState->getName());
			QString supStateName = QString::fromStdWString(behav.supState->getName());
			QString eventName = QString::fromStdWString(behav.event->getName());

			m_outputWidget->writeLine(QString("&nbsp;&nbsp;&nbsp;(%1, %2)&nbsp;&nbsp;-->&nbsp;&nbsp;%3").arg(plantStateName, supStateName, eventName));
		}

		if (m_tabWidget->currentIndex() == 2)
		{
			message += "\nSee the output for a detailed counter example.";
			if (showMsgBox)
			{
				QMessageBox::information(m_outputWidget->topLevelWidget(), "DESPOT", message);
			}
		}
		else
		{
			message += "\nWould you like to see a detailed counter example?";
			if (showMsgBox)
			{
				if (QMessageBox::information(m_outputWidget->topLevelWidget(), "DESPOT", message, QMessageBox::Yes|QMessageBox::No) == QMessageBox::Yes)
				{
					m_tabWidget->setCurrentIndex(2);
				}
			}
		}
	}

	//save the output in the output selection widget
	saveOutput();
}

void OutputUiPart::show(CtrlStdAlgo& ctrlAlgo, LastOutputOption copyToLastOutput ,
						bool showMsgBox )
{
	show(ctrlAlgo, showMsgBox);

	saveLastOutput(copyToLastOutput);
}



*/
//_________________________________________________________________________________________________

void OutputUiPart::show(MultiCtrlAlgo& ctrlAlgo, bool showMsgBox /*= true*/)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cCtrlOutputTitle));
	m_outputWidget->clear();

	if (ctrlAlgo.isControllable())
	{
		QString message("<b>System is controllable. Verified on %1</b>");
		message = message.arg(QString::fromStdWString(m_project->getCtrlStamp()));
		m_outputWidget->writeLine(message);

		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot","The system is controllable");
		}
	}
	else
	{
		m_outputWidget->writeLine("<b>System is not controllable</b>");
		QString messageTempl = "The system is uncontrollable at state:&nbsp;&nbsp;(%1)<br>for event:&nbsp;&nbsp;'%2'.<br><br> State '%3' of DES '%4' is blocking the event.";

		// Creates list of DEs for project so can be matched
		// to state names
		MeetAlgo::DesSet&  inDesSet = ctrlAlgo.getInDesSet();

		//Create the closed state name
        QString closedStateName;
		for(unsigned int iSrcState = 0; iSrcState < ctrlAlgo.getUnctrlBehaviour().closedState.size(); iSrcState++)
		{

       		       // get DES that matches state
		       const Des*  blockingDes = inDesSet[iSrcState];

			if (iSrcState > 0)
			{
				closedStateName += ", ";
			}

			// prepend state by DES name so we can
			// distinguish which DES the state name
			// belongs to 
			closedStateName += QString::fromStdWString(blockingDes->getName()) + "/";

			closedStateName += QString::fromStdWString(ctrlAlgo.getUnctrlBehaviour().closedState[iSrcState]->getName());
		}





		QString blockEventName = QString::fromStdWString(ctrlAlgo.getUnctrlBehaviour().exitEvent->getName());
		QString blockingStateName = QString::fromStdWString(ctrlAlgo.getUnctrlBehaviour().blockingState->getName());
		QString blockingDesName = QString::fromStdWString(ctrlAlgo.getUnctrlBehaviour().blockingDes->getName());

		QString message = messageTempl.arg(closedStateName, blockEventName, blockingStateName, blockingDesName);
		m_outputWidget->writeLine(message);
		m_outputWidget->writeLine("");


		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot","The system is uncontrollable. See output for details.");
		}
	}

	//save the output in the output selection widget
	saveOutput();
}

//_________________________________________________________________________________________________

void OutputUiPart::show(MultiCtrlAlgo& ctrlAlgo, LastOutputOption copyToLastOutput /*= OutputUiPart::eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(ctrlAlgo, showMsgBox);

	saveLastOutput(copyToLastOutput);
}
//_________________________________________________________________________________________________

void OutputUiPart::showMeasurements(const Des* flatSystem)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cMeasureOutputTitle));
	m_outputWidget->clear();

	m_outputWidget->writeLine("<b>Flat System Size:</b>");
	m_outputWidget->write(QString("&nbsp;&nbsp;&nbsp;%1 state(s)").arg(flatSystem->getStateCount()));
	m_outputWidget->writeLine(QString(" with %2 marked").arg(flatSystem->getMarkedStateCount()));
	m_outputWidget->writeLine(QString("&nbsp;&nbsp;&nbsp;%1 event(s)").arg(flatSystem->getEventCount()));
	m_outputWidget->writeLine(QString("&nbsp;&nbsp;&nbsp;%1 transitions(s)").arg(flatSystem->getTransCount()));

	saveOutput();
}

//_________________________________________________________________________________________________

void OutputUiPart::show(LdInterfaceCheckAlgo& ldInterfCheckAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cLdInterfCheckTitle));

	m_outputWidget->clear();

	if (ldInterfCheckAlgo.isLdInterface())
	{
		if (showMsgBox)
		{
			QString message("'%1' is an LD-interface");
			message = message.arg(QString::fromStdWString(ldInterfCheckAlgo.getInputInterface()->getName()));
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",message);
		}

		QString out("<b>Interface '%1' is an LD-interface. Verified on %2.</b>");
		out = out.arg(QString::fromStdWString(ldInterfCheckAlgo.getInputInterface()->getName()),
							  QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine(out);

	}
	else
	{
		if (showMsgBox)
		{
			QString message("'%1' is not an LD-interface.");
			message = message.arg(QString::fromStdWString(ldInterfCheckAlgo.getInputInterface()->getName()));
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",message);
		}

		QString out("<b>Interface '%1' is not LD-interface.</b> <br><br><I>Error:</I> &nbsp;&nbsp;%2.");
		out = out.arg(QString::fromStdWString(ldInterfCheckAlgo.getInputInterface()->getName()),
					  QString::fromStdWString(ldInterfCheckAlgo.getCounterExample().errMessage));
		m_outputWidget->writeLine(out);
		m_outputWidget->writeLine("");

	}

	saveOutput();
}

//_________________________________________________________________________________________________

void OutputUiPart::show(LdInterfaceCheckAlgo& ldInterfCheckAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(ldInterfCheckAlgo, showMsgBox);

	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________

void OutputUiPart::show(IConsistAnsAcceptAlgo& ansAcceptAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cSubsysInterfConsistOutputTitle));

	m_outputWidget->clear();

	if (ansAcceptAlgo.allAnsEventsAccepted())
	{
		if (showMsgBox)
		{
			QString msg("The high-level subsystem is interface consistent.");
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",msg);
		}

		QString out("The high-level subsystem is interface consistent. Verified on %1.");
		out = out.arg(QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine("<b>" + out + "</b>");

	}
	else
	{
		QString out("The high-level subsystem fails to accept all answer events.");
		out = out.arg(QString::fromStdWString(hierProject()->getRootSubsys().getName()));

		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");
		m_outputWidget->writeLine("");

		QString errOut("<I>Error:</I> %1");
		errOut = out.arg(QString::fromStdWString(ansAcceptAlgo.getError()));
		m_outputWidget->writeLine(errOut);
		m_outputWidget->writeLine("");

	}

	saveOutput();
}

//_________________________________________________________________________________________________

void OutputUiPart::show(IConsistAnsAcceptAlgo& ansAcceptAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(ansAcceptAlgo, showMsgBox);

	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________

void OutputUiPart::show(LowSubsysIConsistAlgo& lowSubsysIConsAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cSubsysInterfConsistOutputTitle));

	m_outputWidget->clear();

	if (lowSubsysIConsAlgo.isSubsysInterfConsist())
	{
		if (showMsgBox)
		{
			QString msg("The low-level subsystem '%1' is LD interface consistent");
			msg = msg.arg(QString::fromStdWString(lowSubsysIConsAlgo.getInputSubsystem().getName()));
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",msg);
		}

		QString out("<b>The low-level subsystem '%1' is LD interface consistent. Verified on %2.</b>");
		out = out.arg(QString::fromStdWString(lowSubsysIConsAlgo.getInputSubsystem().getName()),
							  QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine(out);

	}
	else
	{
		QString out("The low-level subsystem '%1' is not LD interface consistent - %2 error(s).");
		out = out.arg(QString::fromStdWString(lowSubsysIConsAlgo.getInputSubsystem().getName()),
					  QString("%1").arg(lowSubsysIConsAlgo.getErrorCount()));

		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
        showErrors(lowSubsysIConsAlgo.getErrorList());
		m_outputWidget->writeLine("");

	}

	saveOutput();
}


//_________________________________________________________________________________________________

void OutputUiPart::show(LowSubsysIConsistAlgo& lowSubsysIConsAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(lowSubsysIConsAlgo, showMsgBox);

	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________

void OutputUiPart::show(IConsistAlgo& iConsAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cInterfConsistOutputTitle));

	m_outputWidget->clear();

	if (iConsAlgo.isInterfaceConsistent())
	{
		if (showMsgBox)
		{
			QString message("Project is LD interface consistent");
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", message);
		}

		QString out("<b>Project is LD interface consistent. Verified on %1.</b>");
		out = out.arg(QString::fromStdWString(hierProject()->getIConsistStamp()));
		m_outputWidget->writeLine(out);
	}
	else
	{

		QString out("<b>Project is not LD interface consistent - %1 error(s)</b>");
		out = out.arg(iConsAlgo.getErrorCount());

		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		showErrors(iConsAlgo.getErrorList());

		m_outputWidget->writeLine("");

	}

	saveOutput();
}


//_________________________________________________________________________________________________

void OutputUiPart::show(IConsistAlgo& iConsAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(iConsAlgo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________

void OutputUiPart::show(LwNonBlockingAlgo& lwNonBlockAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cLwNonBlockOutputTitle));

	m_outputWidget->clear();

	if (lwNonBlockAlgo.isLwNonBlocking())
	{
		if (showMsgBox)
		{
			QString message("Project is LD level-wise nonblocking.");
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", message);
		}

		QString out("<b>Project is LD level-wise nonblocking. Verified on %1.</b>");
		out = out.arg(QString::fromStdWString(hierProject()->getLwNonBlockStamp()));
		m_outputWidget->writeLine(out);
		m_outputWidget->writeLine("");
	}
	else
	{
		QString out("Project is not LD level-wise nonblocking - %1 error(s)");
		out = out.arg(lwNonBlockAlgo.getErrorCount());
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		showErrors(lwNonBlockAlgo.getErrorList());
		// remove next two  lines
		//		m_outputWidget->writeLine("Verification of low-level<br>line 2<br> line 3");
		//		m_outputWidget->writeLine("line 4.");
	}

	saveOutput();
}


//_________________________________________________________________________________________________

void OutputUiPart::show(LwNonBlockingAlgo& lwNonBlockAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(lwNonBlockAlgo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________

void OutputUiPart::show(LwCtrlAlgo& lwCtrlAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cLwCtrlOutputTitle));

	m_outputWidget->clear();

	if (lwCtrlAlgo.isLwControllable())
	{
		if (showMsgBox)
		{
			QString message("Project is LD level-wise controllable.");
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", message);
		}

		QString out("<b>Project is LD level-wise controllable. Verified on %1.<b>");
		out = out.arg(QString::fromStdWString(hierProject()->getLwCtrlStamp()));
		m_outputWidget->writeLine(out);
		m_outputWidget->writeLine("");
	}
	else
	{
		QString out("Project is not LD level-wise controllable - %1 error(s)");
		out = out.arg(lwCtrlAlgo.getErrorCount());
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		showErrors(lwCtrlAlgo.getErrorList());

		m_outputWidget->writeLine("");
	}

	saveOutput();
}


//_________________________________________________________________________________________________

void OutputUiPart::show(LwCtrlAlgo& lwCtrlAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(lwCtrlAlgo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________

void OutputUiPart::show(HighSubsysNonBlockAlgo& hSubsysNonBlockAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cSubsysNonBlockingOutputTitle));

	m_outputWidget->clear();

	if (hSubsysNonBlockAlgo.isNonBlocking())
	{
		if (showMsgBox)
		{
			QString msg("The high-level subsystem is nonblocking.");
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",msg);
		}

		QString out("<b>The high-level subsystem is nonblocking. Verified on %1.</b>");
		out = out.arg(QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine(out);

	}
	else
	{
		QString out("The high-level subsystem is blocking - %1 error(s).");

		const DesAlgo::ErrorList& errList = hSubsysNonBlockAlgo.getErrorList();
		// for some reason, getErrorCount() returns wrong value
		//		only for the subsystem nonblocking algs
		out = out.arg(errList.size());
		//		out = out.arg(hSubsysNonBlockAlgo.getErrorCount());
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		showErrors(errList);
		//		showErrors(hSubsysNonBlockAlgo.getErrorList());

		m_outputWidget->writeLine("");

	}

	saveOutput();
}


//_________________________________________________________________________________________________

void OutputUiPart::show(HighSubsysNonBlockAlgo& hSubsysNonBlockAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(hSubsysNonBlockAlgo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________

void OutputUiPart::show(LowSubsysNonBlockAlgo& lowSubsysNonBlockAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cSubsysNonBlockingOutputTitle));

	m_outputWidget->clear();

	if (lowSubsysNonBlockAlgo.isNonBlocking())
	{
		if (showMsgBox)
		{
			QString msg("The low-level subsystem '%1' is nonblocking");
			msg = msg.arg(QString::fromStdWString(lowSubsysNonBlockAlgo.getInputSubsystem().getName()));
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",msg);
		}

		QString out("<b>The low-level subsystem '%1' is nonblocking. Verified on %2.</b>");
		out = out.arg(QString::fromStdWString(lowSubsysNonBlockAlgo.getInputSubsystem().getName()),
							  QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine(out);

	}
	else
	{
		QString out("The low-level subsystem '%1' is blocking - %2 error(s).");

		const DesAlgo::ErrorList& errList = lowSubsysNonBlockAlgo.getErrorList();
		// for some reason, getErrorCount() returns wrong value
		// only for the subsystem nonblocking algs

		out = out.arg(QString::fromStdWString(lowSubsysNonBlockAlgo.getInputSubsystem().getName()),
			      QString("%1").arg(errList.size()));

		/*		out = out.arg(QString::fromStdWString(lowSubsysNonBlockAlgo.getInputSubsystem().getName()),
				QString("%1").arg(lowSubsysNonBlockAlgo.getErrorCount()));  */


		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		showErrors(errList);
		//		showErrors(lowSubsysNonBlockAlgo.getErrorList());

		m_outputWidget->writeLine("");

	}

	saveOutput();
}


//_________________________________________________________________________________________________

void OutputUiPart::show(LowSubsysNonBlockAlgo& lowSubsysNonBlockAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(lowSubsysNonBlockAlgo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________

void OutputUiPart::show(HighSubsysCtrlAlgo& hSubsysCtrlAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cSubsysCtrlOutputTitle));

	m_outputWidget->clear();

	if (hSubsysCtrlAlgo.isControllable())
	{
		if (showMsgBox)
		{
			QString msg("The high-level subsystem is controllable.");
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",msg);
		}

		QString out("<b>The high-level subsystem is controllable. Verified on %1.</b>");
		out = out.arg(QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine(out);

	}
	else
	{
		QString out("The high-level subsystem is not controllable - %1 error(s).");
		out = out.arg(hSubsysCtrlAlgo.getErrorCount());
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		showErrors(hSubsysCtrlAlgo.getErrorList());

		m_outputWidget->writeLine("");

	}

	saveOutput();
}


//_________________________________________________________________________________________________

void OutputUiPart::show(HighSubsysCtrlAlgo& hSubsysCtrlAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(hSubsysCtrlAlgo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________

void OutputUiPart::show(LowSubsysCtrlAlgo& lowSubsysCtrlAlgo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cSubsysCtrlOutputTitle));

	m_outputWidget->clear();

	if (lowSubsysCtrlAlgo.isControllable())
	{
		if (showMsgBox)
		{
			QString msg("The low-level subsystem '%1' is controllable");
			msg = msg.arg(QString::fromStdWString(lowSubsysCtrlAlgo.getInputSubsystem().getName()));
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",msg);
		}

		QString out("<b>The low-level subsystem '%1' is controllable. Verified on %2.</b>");
		out = out.arg(QString::fromStdWString(lowSubsysCtrlAlgo.getInputSubsystem().getName()),
							  QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine(out);

	}
	else
	{
		QString out("The low-level subsystem '%1' is not controllable - %2 error(s).");
		out = out.arg(QString::fromStdWString(lowSubsysCtrlAlgo.getInputSubsystem().getName()),
					  QString("%1").arg(lowSubsysCtrlAlgo.getErrorCount()));
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot",out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		showErrors(lowSubsysCtrlAlgo.getErrorList());

		m_outputWidget->writeLine("");

	}

	saveOutput();
}


//_________________________________________________________________________________________________

void OutputUiPart::show(LowSubsysCtrlAlgo& lowSubsysCtrlAlgo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/,
						bool showMsgBox /*= true*/)
{
	show(lowSubsysCtrlAlgo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________
void OutputUiPart::show(OutputSource source, std::wstring heading, std::wstring error, LastOutputOption copyToLastOutput, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(getOutputIndexForSource(source));

	m_outputWidget->clear();

	m_outputWidget->writeLine("<b>" + QString::fromStdWString(heading) + ":</b>");
	m_outputWidget->writeLine(QString::fromStdWString(error));

	if (showMsgBox)
	{
		QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", QString::fromStdWString(error));
	}

	//save the output in the output selection widget
	saveOutput();

	saveLastOutput(copyToLastOutput);
}

//_________________________________________________________________________________________________

#ifdef __ENABLE_BDD__

void OutputUiPart::show(BDDHISC::Hisc_LWNBChkInfo& checkInfo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cLwNonBlockOutputTitle));

	m_outputWidget->clear();

	if (checkInfo.m_chkPassed)
	{
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", checkInfo.m_headerStr);
		}

		QString out("<b>Project is LD level-wise nonblocking. Verified on %1.</b>");
		out = out.arg(QString::fromStdWString(hierProject()->getLwNonBlockStamp()));
		m_outputWidget->writeLine(out);
		m_outputWidget->writeLine(checkInfo.m_succStr);
		m_outputWidget->writeLine("");
	}
	else
	{
	        QString out("Project is not LD level-wise nonblocking.");
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		m_outputWidget->writeLine("");
		m_outputWidget->writeLine(checkInfo.m_errStr);
		m_outputWidget->writeLine("");
	}

	saveOutput();
}

void OutputUiPart::show(BDDHISC::Hisc_LWNBChkInfo& checkInfo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/, bool showMsgBox /*= true*/)
{
	show(checkInfo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}


//_________________________________________________________________________________________________

void OutputUiPart::show(BDDHISC::Hisc_SubLWNBChkInfo& checkInfo, bool showMsgBox)
{
        m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cSubsysNonBlockingOutputTitle));

	m_outputWidget->clear();

	if (checkInfo.m_chkPassed)
	{
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", checkInfo.m_headerStr);
		}

		QString out("<b>Subsystem '" + checkInfo.m_subsystName + "' is LD nonblocking. Verified on %1.</b>");
		out = out.arg(QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine(out);
		m_outputWidget->writeLine(checkInfo.m_succStr);
		m_outputWidget->writeLine("");
	}
	else
	{
	        QString out("Subsystem '" + checkInfo.m_subsystName + 
			    "' is not LD nonblocking.");
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		m_outputWidget->writeLine("");
		m_outputWidget->writeLine(checkInfo.m_errStr);
		m_outputWidget->writeLine("");
	}

	saveOutput();
}

void OutputUiPart::show(BDDHISC::Hisc_SubLWNBChkInfo& checkInfo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/, bool showMsgBox /*= true*/)
{
	show(checkInfo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}


//_________________________________________________________________________________________________

void OutputUiPart::show(BDDHISC::Hisc_LWContChkInfo& checkInfo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cLwCtrlOutputTitle));

	m_outputWidget->clear();

	if (checkInfo.m_chkPassed)
	{
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", checkInfo.m_headerStr);
		}

		QString out("<b>Project is LD level-wise controllable. Verified on %1.</b>");
		out = out.arg(QString::fromStdWString(hierProject()->getLwCtrlStamp()));
		m_outputWidget->writeLine(out);
		m_outputWidget->writeLine(checkInfo.m_succStr);
		m_outputWidget->writeLine("");
	}
	else
	{
	        QString out("Project is not LD level-wise controllable.");
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		m_outputWidget->writeLine("");
		m_outputWidget->writeLine(checkInfo.m_errStr);
		m_outputWidget->writeLine("");
	}

	saveOutput();
}

void OutputUiPart::show(BDDHISC::Hisc_LWContChkInfo& checkInfo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/, bool showMsgBox /*= true*/)
{
	show(checkInfo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}


//_________________________________________________________________________________________________

void OutputUiPart::show(BDDHISC::Hisc_SubLWContChkInfo& checkInfo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cSubsysCtrlOutputTitle));

	m_outputWidget->clear();

	if (checkInfo.m_chkPassed)
	{
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", checkInfo.m_headerStr);
		}

		QString out("<b>Subsystem '" + checkInfo.m_subsystName + "' is LD controllable. Verified on %1.</b>");
		out = out.arg(QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine(out);
		m_outputWidget->writeLine(checkInfo.m_succStr);
		m_outputWidget->writeLine("");
	}
	else
	{
	        QString out("Subsystem '" + checkInfo.m_subsystName + 
			    "' is not LD controllable.");
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		m_outputWidget->writeLine("");
		m_outputWidget->writeLine(checkInfo.m_errStr);
		m_outputWidget->writeLine("");
	}

	saveOutput();
}

void OutputUiPart::show(BDDHISC::Hisc_SubLWContChkInfo& checkInfo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/, bool showMsgBox /*= true*/)
{
	show(checkInfo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}


//_________________________________________________________________________________________________

void OutputUiPart::show(BDDHISC::Hisc_IConsisChkInfo& checkInfo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cInterfConsistOutputTitle));

	m_outputWidget->clear();

	if (checkInfo.m_chkPassed)
	{
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", checkInfo.m_headerStr);
		}

		QString out("<b>Project is LD interface consistent. Verified on %1.</b>");
		out = out.arg(QString::fromStdWString(hierProject()->getIConsistStamp()));
		m_outputWidget->writeLine(out);
		m_outputWidget->writeLine(checkInfo.m_succStr);
		m_outputWidget->writeLine("");
	}
	else
	{
	        QString out("Project is not LD interface consistent.");
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		m_outputWidget->writeLine("");
		m_outputWidget->writeLine(checkInfo.m_errStr);
		m_outputWidget->writeLine("");
	}

	saveOutput();	
}

void OutputUiPart::show(BDDHISC::Hisc_IConsisChkInfo& checkInfo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/, bool showMsgBox /*= true*/)
{
	show(checkInfo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}


//_________________________________________________________________________________________________

void OutputUiPart::show(BDDHISC::Hisc_SubIConsisChkInfo& checkInfo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cSubsysInterfConsistOutputTitle));

	m_outputWidget->clear();

	if (checkInfo.m_chkPassed)
	{
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", checkInfo.m_headerStr);
		}

		QString out("<b>Subsystem '" + checkInfo.m_subsystName + "' is LD interface consistent. Verified on %1.</b>");
		out = out.arg(QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine(out);
		m_outputWidget->writeLine(checkInfo.m_succStr);
		m_outputWidget->writeLine("");
	}
	else
	{
	        QString out("Subsystem '" + checkInfo.m_subsystName + 
			    "' is not LD interface consistent.");
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		m_outputWidget->writeLine("");
		m_outputWidget->writeLine(checkInfo.m_errStr);
		m_outputWidget->writeLine("");
	}

	saveOutput();
}

void OutputUiPart::show(BDDHISC::Hisc_SubIConsisChkInfo& checkInfo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/, bool showMsgBox /*= true*/)
{
	show(checkInfo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}


//_________________________________________________________________________________________________

void OutputUiPart::show(BDDHISC::Hisc_LDIntfChkInfo& checkInfo, bool showMsgBox)
{
	m_showOutputForWidg->setCurrentIndex(m_showOutputForWidg->findText(cLdInterfCheckTitle));

	m_outputWidget->clear();

	if (checkInfo.m_chkPassed)
	{
		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", checkInfo.m_headerStr);
		}

		QString out("<b>Interface '" + checkInfo.m_theIntfName
		+ "' is an LD interface. Verified on %1.</b>");
		out = out.arg(QDateTime::currentDateTime().toString());
		m_outputWidget->writeLine(out);
		//		m_outputWidget->writeLine("");
		//		m_outputWidget->writeLine(checkInfo.m_succStr);
		m_outputWidget->writeLine("");
	}
	else
	{
	        QString out("Interface '" + checkInfo.m_theIntfName
			    + "' is not an LD interface.");

		if (showMsgBox)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", out);
		}
		m_outputWidget->writeLine("<b>" + out + "</b>");

		//output all errors
		m_outputWidget->writeLine("");
		m_outputWidget->writeLine(checkInfo.m_errStr);
		m_outputWidget->writeLine("");
	}

	saveOutput();

}

void OutputUiPart::show(BDDHISC::Hisc_LDIntfChkInfo& checkInfo, LastOutputOption copyToLastOutput /*= eSetLastOutput*/, bool showMsgBox /*= true*/)
{
	show(checkInfo, showMsgBox);
	saveLastOutput(copyToLastOutput);
}

#endif

//_________________________________________________________________________________________________

int OutputUiPart::getOutputIndexForSource(OutputSource source)
{
	switch(source)
	{
		case eProjectIntegrityCheck:
			return m_showOutputForWidg->findText(cProjIntegOutputTitle);

		case eNonBlockingCheck:
			return m_showOutputForWidg->findText(cNonBlockOutputTitle);

		case eControllabilityCheck:
			return m_showOutputForWidg->findText(cCtrlOutputTitle);

		case eInterfConsistCheck:
			return m_showOutputForWidg->findText(cInterfConsistOutputTitle);

		case eLwNonblockingCheck:
			return m_showOutputForWidg->findText(cLwNonBlockOutputTitle);

		case eLwCtrlCheck:
			return m_showOutputForWidg->findText(cLwCtrlOutputTitle);

		case eSubsysInterfConsistCheck:
			return m_showOutputForWidg->findText(cSubsysInterfConsistOutputTitle);

		case eLdInterfCheck:
			return m_showOutputForWidg->findText(cLdInterfCheckTitle);

		case eSubsysNonblockingCheck:
			return m_showOutputForWidg->findText(cSubsysNonBlockingOutputTitle);

		case eSubsysCtrlCheck:
			return m_showOutputForWidg->findText(cSubsysCtrlOutputTitle);

		default:
			return m_showOutputForWidg->findText(cLastOutputTitle);
	}

}

//_________________________________________________________________________________________________

void OutputUiPart::clearLastOutput()
{
	int lastOutIndex = m_showOutputForWidg->findText(cLastOutputTitle);

	if (m_showOutputForWidg->currentIndex() == lastOutIndex)
	{
		//we're displaying the last output so we must clear the output widget
		m_outputWidget->clear();
	}

	//erase the data stored for the last output itme
	m_showOutputForWidg->setItemData(lastOutIndex, "");
}

//_________________________________________________________________________________________________

void OutputUiPart::show(FlatProjectPropCheck& projCheckAlgo)
{
	clearLastOutput();

	if (projCheckAlgo.runSuccessful())
	{
		QMessageBox::question(m_outputWidget->topLevelWidget(), "DESpot", STR_PROJ_PROP_CHECKED_OK, QMessageBox::Ok);
	}
	else
	{
		if (m_tabWidget->currentIndex() == 2)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", STR_PROJ_PROP_CHECKED_FAIL_STM);
		}
		else
		{
			bool showOutput = (QMessageBox::question(m_outputWidget->topLevelWidget(), "DESpot", STR_PROJ_PROP_CHECKED_FAIL, QMessageBox::Yes | QMessageBox::No)
													 == QMessageBox::Yes);
			if (showOutput)
			{
				m_tabWidget->setCurrentIndex(2);
			}
		}
	}

	if (projCheckAlgo.integrityChecked())
	{
		show(projCheckAlgo.integAlgo(), OutputUiPart::eSetLastOutput, false);
	}

	if (projCheckAlgo.nonBlockChecked())
	{
		showMeasurements(&m_project->getFlatSystemDes());
		show(projCheckAlgo.nonBlockAlgo(), OutputUiPart::eAppendLastOutput, false);
	}

	if (projCheckAlgo.ctrlChecked())
	{
		show(projCheckAlgo.ctrlAlgo(), OutputUiPart::eAppendLastOutput, false);
	}
}

//_________________________________________________________________________________________________

void OutputUiPart::show(HierProjectPropCheck& projCheckAlgo)
{
	clearLastOutput();

	if (projCheckAlgo.runSuccessful())
	{
		QMessageBox::question(m_outputWidget->topLevelWidget(), "DESpot", STR_PROJ_PROP_CHECKED_OK, QMessageBox::Ok);
	}
	else
	{
		if (m_tabWidget->currentIndex() == 2)
		{
			QMessageBox::information(m_outputWidget->topLevelWidget(), "DESpot", STR_PROJ_PROP_CHECKED_FAIL_STM);
		}
		else
		{
			bool showOutput = (QMessageBox::question(m_outputWidget->topLevelWidget(), "DESpot", STR_PROJ_PROP_CHECKED_FAIL, QMessageBox::Yes | QMessageBox::No)
													 == QMessageBox::Yes);
			if (showOutput)
			{
				m_tabWidget->setCurrentIndex(2);
			}
		}
	}

	if (projCheckAlgo.integrityChecked())
	{
		show(projCheckAlgo.integAlgo(), OutputUiPart::eSetLastOutput, false);
	}

	if (projCheckAlgo.iConsistChecked())
	{
		show(projCheckAlgo.interfConsistAlgo(), OutputUiPart::eAppendLastOutput, false);
	}
	if (projCheckAlgo.lwNonBlockChecked())
	{
		show(projCheckAlgo.levelWiseNonBlockAlgo(), OutputUiPart::eAppendLastOutput, false);
	}

	if (projCheckAlgo.lwCtrlChecked())
	{
		show(projCheckAlgo.levelWiseCtrlAlgo(), OutputUiPart::eAppendLastOutput, false);
	}
}

//_________________________________________________________________________________________________
//Show the errors in the algorithm, paging them in case the number of errors is too large
void OutputUiPart::showErrors(const DesAlgo::ErrorList& errList)
{
	m_outputWidget->writeLine("");

	const short errorPagingSize = 10;
	long errorsRemaining = errList.size();

	//for debug
	// std::cout << "entering showErrors. Size" << errorsRemaining << "\n";

	bool pageErrors = true;
	long crtPageErrCount = 0;
	for(DesAlgo::ErrIt errIt = errList.begin(); errIt != errList.end(); ++errIt)
	{
		m_outputWidget->writeLine(QString("<I>Error:</I> ") + QString::fromStdWString(*errIt));
		crtPageErrCount++;
		errorsRemaining--;

		if (pageErrors && (crtPageErrCount == errorPagingSize) && errorsRemaining > 0)
		{
			QString msg("Do you want to show the next %1 errors?");
			msg = msg.arg(errorPagingSize);
			int userAnswer = QMessageBox::question(m_outputWidget->topLevelWidget(), STR_DESPOT_WARNING, msg, 
															  QMessageBox::Yes | QMessageBox::YesToAll | 
															  QMessageBox::No, 
															  QMessageBox::Yes);

			if (userAnswer == QMessageBox::Yes)
			{
				//the user wants to see the next page of errors; reset the error counter for the new page
				crtPageErrCount = 0;
				continue;
			}
			else if (userAnswer == QMessageBox::YesToAll)
			{
				pageErrors = false;							
				continue;
			}
			else if (userAnswer == QMessageBox::No)
			{
				//the user doesn't want to see anymore errors
				m_outputWidget->writeLine(
					(errorsRemaining == 1) ? 
						QString("1 error has not been displayed.") : 
						QString("%1 errors have not been displayed.").arg(errorsRemaining));
				break;
			}
			else
			{
				assert(false);
				//assume the user doesn't want to see anymore errors
				m_outputWidget->writeLine(
					(errorsRemaining == 1) ? 
						QString("1 error has not been displayed.") : 
						QString("%1 errors have not been displayed.").arg(errorsRemaining));
				break;
			}
		}

	}

	m_outputWidget->writeLine("");
}


}

