#include "CoObsAlgo.h"
#include "DecEvent.h"

namespace DESpot
{

CoObsAlgo::CoObsAlgo(DesProject* project): m_project(project), m_isCoObservable(false)
{
	addDecObserver(m_project->createObserverIterator());
	addInputPlantDes(m_project->createDesIterator(ePlantDes));
	addInputSupDes(m_project->createDesIterator(eSupervisorDes));

	m_numDec=m_project->getObserverCount();
	tupleEvent=NULL;
	tupleState=NULL;
	state_pendinglist=NULL;
}

//_________________________________________________________________________________________________

CoObsAlgo::~CoObsAlgo(void)
{
	if(tupleEvent!=null)
	{
		delete tupleEvent;
		tupleEvent=null;
	}

	if(tupleState!=null)
	{
		delete tupleState;
		tupleState=null;
	}

	if(state_pendinglist!=null)
	{
		delete state_pendinglist;
		state_pendinglist=null;
	}
}

//_________________________________________________________________________________________________

void CoObsAlgo::addDecObserver(DesProject::ObserverIteratorPtr obsvIt)
{
	for(obsvIt->first(); obsvIt->notDone(); obsvIt->next())
	{
		addDecObserver(&obsvIt->currentItem());
	}
}

//_________________________________________________________________________________________________

void CoObsAlgo::addDecObserver(const DecObserver* obsv)
{
	/*
	QString obsName = QString::fromStdWString(obsv->getName());
    cout<< obsName.toStdString()<<endl;

    cout<<"ctr event:"<<endl;
    DecObserver::ObserverCtrlEventIteratorPtr obsvCtrIt = obsv->createObserverCtrlEventIterator();
    for(obsvCtrIt->first(); obsvCtrIt->notDone(); obsvCtrIt->next())
    {
        const DecEvent& decEvent = obsvCtrIt->currentItem();
        QString itemText = QString::fromStdWString(decEvent.getName());
        cout<< itemText.toStdString()<<endl;
    }

    cout<<"obs event:"<<endl;
    DecObserver::ObserverObsEventIteratorPtr obsvObsIt = obsv->createObserverObsEventIterator();
    for(obsvObsIt->first(); obsvObsIt->notDone(); obsvObsIt->next())
    {
        const DecEvent& decEvent = obsvObsIt->currentItem();
        QString itemText = QString::fromStdWString(decEvent.getName());
        cout<< itemText.toStdString()<<endl;
    }
    */

	//Make sure the same observer is not added twice
	for (unsigned int iObs = 0; iObs < m_observers.size(); ++iObs)
	{
		if (obsv == m_observers[iObs])
		{
			throw EX("A decentralized supervisor with the same name already exists. Cannot add.")
		}
	}
	
	m_observers.push_back(obsv);
}

//_________________________________________________________________________________________________

void CoObsAlgo::addInputPlantDes(DesProject::DesIteratorPtr plantDesIt)
{
	for(plantDesIt->first(); plantDesIt->notDone(); plantDesIt->next())
	{
		addInputPlantDes(&plantDesIt->currentItem());
	}

}

//_________________________________________________________________________________________________

void CoObsAlgo::addInputPlantDes(const Des* inPlantDes)
{
	//Make sure the same Des is not added twice
	for (unsigned int iDes = 0; iDes < m_inPlantDesSet.size(); ++iDes)
	{
		if (inPlantDes == m_inPlantDesSet[iDes])
		{
			throw EX("A DES with the same name already exists in the list of input plant DES. Cannot add.")
		}
	}
	
	m_inPlantDesSet.push_back(inPlantDes);
}

//_________________________________________________________________________________________________

void CoObsAlgo::addInputSupDes(DesProject::DesIteratorPtr supDesIt)
{
	for(supDesIt->first(); supDesIt->notDone(); supDesIt->next())
	{
		addInputSupDes(&supDesIt->currentItem());
	}
}

//_________________________________________________________________________________________________

void CoObsAlgo::addInputSupDes(const Des* inSupDes)
{
	//Make sure the same Des is not added twice
	for (unsigned int iDes = 0; iDes < m_inSupDesSet.size(); ++iDes)
	{
		if (inSupDes == m_inSupDesSet[iDes])
		{
			throw EX("A DES with the same name already exists in the list of input superivsor DES. Cannot add.")
		}
	}
	
	m_inSupDesSet.push_back(inSupDes);
}

//_________________________________________________________________________________________________

void CoObsAlgo::onEventBlocked(short eventId, SrcStateTuple& srcStateTuple, short iSrc)
{	
}

//_________________________________________________________________________________________________

void CoObsAlgo::loadProductDes()
{
	newSyncAlgo::loadProductDes();

	tupleState=new CoObsvStateTrie(m_inDesSet,m_numDec);
	state_pendinglist=new PendingList((m_numDec+1)*DesNum);
}

//_________________________________________________________________________________________________

void CoObsAlgo::createInitialStateTuple()
{
	newSyncAlgo::createInitialStateTuple();

	SrcStateTuple tuple((m_numDec+1)*DesNum);
	for(short i=0;i<m_numDec+1;i++)
	{
		for(short j=0;j<DesNum;j++)
		{
			const unsigned long long itmp=m_inDesSet[j]->getInitialState().getId().id();
        	unsigned long long tmp=itmp;
			tuple[i*DesNum+j]=(short)tmp;
		}
	}
   //put into finish list
    tupleState->insert(&tuple);

   //put into the pending list:
	state_pendinglist->PushPending(tuple);
}

//_________________________________________________________________________________________________

void CoObsAlgo::prepareRun()
{
	//1) first add the plantDes and supDes into the DesSet, and mark which one is the plant and which one is the sup
	if (m_inDesSet.empty())
	{
		for (unsigned int iDes = 0; iDes < m_inPlantDesSet.size(); ++iDes)
		{
			addInputDes(m_inPlantDesSet[iDes]);
		}

		m_numPlant=m_inPlantDesSet.size();

		for (unsigned int iDes = 0; iDes < m_inSupDesSet.size(); ++iDes)
		{
			addInputDes(m_inSupDesSet[iDes]);
		}
	}
	
	if (m_inDesSet.size() < 2)
	{
		throw EX("The co-observability algorithm requires at least two input DES.");
	}
	if (m_checkDesIntegrity)
	{
		for (unsigned int iDes = 0; iDes < m_inDesSet.size(); ++iDes)
		{
			const Des* crtDes = m_inDesSet[iDes];
			if (crtDes->getIntegrity() != eIntegYes)
			{			
				//throw errorMessage(cSyncIntegError, crtDes->getName());
				throw EX("Either all DES are not valid, or integrity has not yet been checked.");
			}
		}
	}	
	if (m_numDec < 1)
	{
		throw EX("The co-observability algorithm requires at least one decentralized controller.");
	}

	loadProductDes();
    createInitialStateTuple();
}

//_________________________________________________________________________________________________

bool CoObsAlgo::runAlgo()
{
	if(m_integrity == false)
	{
		throw EX("Either project is not valid, or integrity has not yet been checked.");
		return false;
	}

	//check dec event modified
	DesProject::ObserverIteratorPtr obsvIt = m_project->createObserverIterator();
	for(obsvIt->first(); obsvIt->notDone(); obsvIt->next())
	{
		const DecObserver& Obsv = obsvIt->currentItem();
        if(Obsv.isEventModified())
        {
        	throw EX("Supervisor's event is deleted or controllable property of controllable event is changed. \n Please open Edit Observable\\Controllable editor for adjustment.")
        }
	}

	prepareRun();

	while(state_pendinglist->isEmpty()==false)
	{
		SrcStateTuple crtSrcStateTuple((m_numDec+1)*DesNum);
		state_pendinglist->PopPending(crtSrcStateTuple);
		tupleEvent=new CoObsvLabelTrie(EventNum,m_numDec);

		for(short i=0;i<m_numDec+1;i++)
		{
			for(short j=1;j<EventNum;j++)
			{
				DesEvent* event=tranMap->getEvent(j);
				if(hasPlantTransition(crtSrcStateTuple,i,j))
				{
					if(hasSupTransition(crtSrcStateTuple,i,j))
					{
						SrcEventTuple label;
						if(i==0)
						{
							label=makeObsLabel(j);
						}
						else
						{
							DecObserver obsv=m_project->getObserver(i-1);
							if(obsv.findObsvObsEvent(event->getName()))
							{
								label=makeObsLabel(j);
							} 
							else
							{
								label=makeUnobsLabel(j,i);
							}
						}

						if(tupleEvent->search(&label)==false)
						{
							tupleEvent->insert(&label);
							SrcStateTuple* nextSrcStateTuple=calcNextState(crtSrcStateTuple,label);
							if(nextSrcStateTuple!=NULL && tupleState->search(nextSrcStateTuple)==false)
							{
								tupleState->insert(nextSrcStateTuple);
								state_pendinglist->PushPending(*nextSrcStateTuple);
							}
						}

					}
					else if(i==0 && tranMap->isControllable(j))
					{
						if(isIllegalConfiguration(crtSrcStateTuple,j))
						{
							m_CoObsError="Fail at (";
							for(short x=0;x<m_numDec+1;x++)
							{
								for(short y=0;y<DesNum;y++)
								{
									if(!y)
									{
										m_CoObsError+="(";
									} 

									const Des* des = m_inDesSet[y];
      								QString desName = QString::fromStdWString(des->getName());
									m_CoObsError+=desName+"/";

									short state_id=crtSrcStateTuple[x*DesNum+y];
      								const DesState& state = des->getState(state_id);
      								QString stateName = QString::fromStdWString(state.getName());
									m_CoObsError+=stateName;

									m_CoObsError+=(y<DesNum-1)?",":")";
								}
								m_CoObsError+=(x<m_numDec)?",":")";
							}
							QString eventName = QString::fromStdWString(event->getName());							
							m_CoObsError+=" with event "+eventName;

							return false;
						}
					}
				}
			}
		}
		delete tupleEvent;
		tupleEvent=null;  
	}

	delete state_pendinglist;
	state_pendinglist=null;

	m_isCoObservable=true;

	return true;
}

//_________________________________________________________________________________________________

CoObsAlgo::SrcEventTuple& CoObsAlgo::makeObsLabel(short eventId)
{
	SrcEventTuple* label = new SrcEventTuple;
	
	//the original system with full observability and controllability
	label->push_back(eventId);

	DesEvent* event=tranMap->getEvent(eventId);
    DesProject::ObserverIteratorPtr obsvIt = m_project->createObserverIterator();
	for(obsvIt->first(); obsvIt->notDone(); obsvIt->next())
	{
		const DecObserver& Obsv = obsvIt->currentItem();
        if(Obsv.findObsvObsEvent(event->getName()))
        {
        	label->push_back(eventId);
        }
        else
        {
        	label->push_back(EventNum); //epsilon
        }
	}

	return *label;
}

//_________________________________________________________________________________________________

CoObsAlgo::SrcEventTuple& CoObsAlgo::makeUnobsLabel(short eventId, short iDec)
{
	SrcEventTuple* label = new SrcEventTuple;

	for(short i=0;i<m_numDec+1;i++)
	{
		label->push_back(EventNum); //epsilon
	}

	(*label)[iDec]=eventId;

	return *label;
}

//_________________________________________________________________________________________________

newSyncAlgo::SrcStateTuple* CoObsAlgo::calcNextState(SrcStateTuple s1, SrcEventTuple label)
{
	SrcStateTuple temptuple((m_numDec+1)*DesNum); 
	
	for(short i=0;i<m_numDec+1;i++)
	{
		if(label[i]==EventNum) //epsilon
		{
			for(short j=0;j<DesNum;j++)
			{
				temptuple[i*DesNum+j]=s1[i*DesNum+j];
			}
		}
		else
		{
			SrcStateTuple* tmp=getDesTransition(s1,i,label[i]);
			if(tmp!=NULL)
			{
				for(short j=0;j<DesNum;j++)
				{
					temptuple[i*DesNum+j]=(*tmp)[j];
				}
			}
			else	
			{
				//cout<<"NULL"<<endl;
				return NULL;
			}	
		}
	}

	SrcStateTuple* s2=new SrcStateTuple;
	for(short j=0;j<temptuple.size();j++)
	{
		s2->push_back(temptuple[j]);
	}

	return s2;
}

//_________________________________________________________________________________________________

bool CoObsAlgo::isIllegalConfiguration(SrcStateTuple s, short eventId)
{
	DesEvent* event=tranMap->getEvent(eventId);
	short iDec=1;
    DesProject::ObserverIteratorPtr obsvIt = m_project->createObserverIterator();
	for(obsvIt->first(); obsvIt->notDone(); obsvIt->next())
	{
		const DecObserver& obsv = obsvIt->currentItem();
        if(obsv.findObsvCtrlEvent(event->getName()) && getDesTransition(s,iDec,eventId)==false)
        {
        	return false;
        }
        iDec++;
	}
	return true;
}

//_________________________________________________________________________________________________

newSyncAlgo::SrcStateTuple* CoObsAlgo::getDesTransition(SrcStateTuple s1, short iDec, short eventId)
{
	SrcStateTuple temptuple(DesNum); 
	for(short j=0;j<DesNum;j++)
	{
	  short from_state=s1[iDec*DesNum+j];
	  short to_state=transitionMatrix[j][eventId][from_state];
	 
	  if(to_state!=-1)
	  {
		  temptuple[j]=to_state;
	  }
	  else
	  {
		  return NULL;
	  }
	}
	
	SrcStateTuple* s2=new SrcStateTuple;
	for(short j=0;j<DesNum;j++)
	{
		s2->push_back(temptuple[j]);
	}

	return s2;
}

//_________________________________________________________________________________________________

bool CoObsAlgo::hasPlantTransition(SrcStateTuple s1, short iDec, short eventId)
{ 
	for(short j=0;j<m_numPlant;j++)
	{
	  short from_state=s1[iDec*DesNum+j];
	  short to_state=transitionMatrix[j][eventId][from_state];
	 
	  if(to_state==-1)
	  {
		  return false;
	  }
	}

	return true;
}

//_________________________________________________________________________________________________

bool CoObsAlgo::hasSupTransition(SrcStateTuple s1, short iDec, short eventId)
{ 
	for(short j=m_numPlant;j<DesNum;j++)
	{
	  short from_state=s1[iDec*DesNum+j];
	  short to_state=transitionMatrix[j][eventId][from_state];
	 
	  if(to_state==-1)
	  {
		  return false;
	  }
	}

	return true;
}

}
