/*	Author:		Magdin Stoica
	Supervisor: Dr. Ryan Leduc
	
	Project created in conformity with the requirements for the Degree of Master of Engineering in Software Engineering, 
	Computing and Software Department, 
	McMaster University
	2003 - 2007
*/

#include <QAction>
#include <QMenuBar>
#include <QToolBar>
#include <QStatusBar>
#include <qevent.h>
#include <QHeaderView>
#include <QFont>

#include "CommonDefinitions.h"
#include "DesWorkspace.h"
#include "DesEditor.h"
#include "MainForm.h"
#include "StateEditorUiPart.h"
#include "StateQuickLookUiPart.h"
#include "EventEditorUiPart.h"
#include "TransitionEditorUiPart.h"
#include "DesEditorOutputUiPart.h"
#include "Des.h"
#include "TransRecorder.h"
#include "ProgressWidget.h"

namespace DESpot
{

DesWorkspace::DesWorkspace(DesEditor* pDesEditor, Des* pDes) : 
			Workspace(pDesEditor) 
{
	//configure the form to use the DesEditor.ui
	setupUi(pDesEditor);

	//set the title of the window
	QString desName = QString::fromStdWString(pDes->getName());
	desEditor()->setWindowTitle(desName.isEmpty() ? STR_UNNAMED_DES : desName);

	//create the UI parts of the workspace
	createUiParts(pDesEditor, pDes);

	//Initialize the transition recorder with the output ui
	pDesEditor->getTransRecorder().initOutputUi(m_pOutputUiPart);

	//create menu, toolbar and status bar
	createActions();
	setupMenu();
	setupToolBar();
	setupStatusBar();

	//set the spliter position
	positionSplitters();

	//setup connections between widgets
	setupConnections();
	setMode(pDes->isInstantiation());
}

//_________________________________________________________________________________________________

DesWorkspace::~DesWorkspace(void)
{
	if (m_pStateEdUiPart)
	{
		delete m_pStateEdUiPart;
		m_pStateEdUiPart = null;
	}

	if (m_pStateQuickLookUiPart)
	{
		delete m_pStateQuickLookUiPart;
		m_pStateQuickLookUiPart = null;
	}
	
	if (m_pEventEdUiPart)
	{
		delete m_pEventEdUiPart;
		m_pEventEdUiPart = null;
	}

	if (m_pTransEdUiPart)
	{
		delete m_pTransEdUiPart;
		m_pTransEdUiPart = null;
	}

	if (m_pOutputUiPart)
	{
		delete m_pOutputUiPart;
		m_pOutputUiPart = null;
	}
}

//_________________________________________________________________________________________________

void DesWorkspace::resize(QResizeEvent* /*event*/)
{
	//rezize the main layout to fit the editor content area
	QSize centralWidgSize = centralWidget->size();
	m_mainSplitterWidg->resize(centralWidgSize - QSize(20, 16));
}

//_________________________________________________________________________________________________

StateEditorUiPart& DesWorkspace::stateEditorUiPart()
{
	return *m_pStateEdUiPart;
}

//_________________________________________________________________________________________________

StateQuickLookUiPart&	DesWorkspace::stateQuickLookUiPart()
{
	return *m_pStateQuickLookUiPart;
}

//_________________________________________________________________________________________________

EventEditorUiPart& DesWorkspace::eventEditorUiPart()
{
	return *m_pEventEdUiPart;
}

//_________________________________________________________________________________________________

TransitionEditorUiPart& DesWorkspace::transEditorUiPart()
{
	return *m_pTransEdUiPart;
}

//_________________________________________________________________________________________________

DesEditorOutputUiPart&  DesWorkspace::outputUiPart()
{
	return *m_pOutputUiPart;
}

//_________________________________________________________________________________________________

QAction* DesWorkspace::recordTransAct()
{
	return m_recordTransAct;
}

//_________________________________________________________________________________________________

ProgressWidget* DesWorkspace::progressWidget()
{
	return m_progressWidget;
}

//_________________________________________________________________________________________________

void DesWorkspace::onChangedCurrentState(const DesState* pNewCurrentState, const DesState* pOldCurrentState)
{
	if (pNewCurrentState)
	{
		//a current state was established so configure the actions related with the current state
		m_changeStateAct->setEnabled(true);
		if (pOldCurrentState == null)
		{		
			//the keyboard signal was disconnected so connect it back
			connect(m_stateViewWidg, SIGNAL(itemActivated(QTreeWidgetItem*, int)), desEditor(), SLOT(onChangeState()));
		}

		m_deleteStateAct->setEnabled(true);
		if (pOldCurrentState == null)
		{		
			//the keyboard signal was disconnected so connect it back
			connect(m_stateViewWidg, SIGNAL(onCurrentItemDeleted()), desEditor(), SLOT(onDeleteState()));
		}
		
		m_toggleInitialStateAct->setEnabled(true);
		m_toggleInitialStateAct->setChecked(pNewCurrentState->isInit());
		
		m_toggleMarkedStateAct->setEnabled(true);
		m_toggleMarkedStateAct->setChecked(pNewCurrentState->isMarked());
	}
	else
	{
		//the current state doesn't exist so disable any actions related to it
		m_changeStateAct->setEnabled(false);
		disconnect(m_stateViewWidg, SIGNAL(itemActivated(QTreeWidgetItem*, int)), desEditor(), SLOT(onChangeState()));
		
		m_deleteStateAct->setEnabled(false);
		disconnect(m_stateViewWidg, SIGNAL(onCurrentItemDeleted()), desEditor(), SLOT(onDeleteState()));
		
		m_toggleInitialStateAct->setEnabled(false);
		m_toggleInitialStateAct->setChecked(false);
		
		m_toggleMarkedStateAct->setEnabled(false);
		m_toggleMarkedStateAct->setChecked(false);
	}
}

//_________________________________________________________________________________________________

void DesWorkspace::updateCurrentStateUi(const DesState& crtState)
{
	m_toggleInitialStateAct->setChecked(crtState.isInit());
	m_toggleMarkedStateAct->setChecked(crtState.isMarked());
}

//_________________________________________________________________________________________________

void DesWorkspace::onChangedCurrentEvent(const DesEvent* pNewCurrentEvent, const DesEvent* pOldCurrentEvent)
{
	if (pNewCurrentEvent)
	{
		//a current event was established so configure the actions related with the current event
		m_changeEventAct->setEnabled(true);
		if (pOldCurrentEvent == null)
		{		
			//the keyboard signal was disconnected so connect it back
			connect(m_eventViewWidg, SIGNAL(itemActivated(QTreeWidgetItem*, int)), desEditor(), SLOT(onChangeEvent()));
		}

		m_deleteEventAct->setEnabled(true);
		if (pOldCurrentEvent == null)
		{		
			//the keyboard signal was disconnected so connect it back
			connect(m_eventViewWidg, SIGNAL(onCurrentItemDeleted()), desEditor(), SLOT(onDeleteEvent()));
		}

		m_toggleEventCtrlAct->setEnabled(true);
		m_toggleEventCtrlAct->setChecked(pNewCurrentEvent->isControllable());

		m_eventTypeSubMenu->menuAction()->setEnabled(true);
		m_pEventEdUiPart->getEventTypeContextSubMenu().menuAction()->setEnabled(true);
		getActionForEventType(pNewCurrentEvent->getType())->setChecked(true);
	}
	else
	{
		//the current event doesn't exist so disable any actions related to it
		m_changeEventAct->setEnabled(false);
		disconnect(m_eventViewWidg, SIGNAL(itemActivated(QTreeWidgetItem*, int)), desEditor(), SLOT(onChangeEvent()));
		
		m_deleteEventAct->setEnabled(false);
		disconnect(m_eventViewWidg, SIGNAL(onCurrentItemDeleted()), desEditor(), SLOT(onDeleteEvent()));		

		m_toggleEventCtrlAct->setEnabled(false);
		m_toggleEventCtrlAct->setChecked(false);

		m_eventTypeSubMenu->menuAction()->setEnabled(false);
		m_pEventEdUiPart->getEventTypeContextSubMenu().menuAction()->setEnabled(false);		
	}
}

//_________________________________________________________________________________________________

void DesWorkspace::updateCurrentEventUi(const DesEvent& crtEvent)
{
	m_toggleEventCtrlAct->setChecked(crtEvent.isControllable());
	getActionForEventType(crtEvent.getType())->setChecked(true);
}


//_________________________________________________________________________________________________

void DesWorkspace::onChangedCurrentTrans(const DesTransition* pNewCurrentTrans, const DesTransition* pOldCurrentTrans)
{
	if (pNewCurrentTrans)
	{
		//a current transition was established so configure the actions related with the current transition
		m_changeTransAct->setEnabled(true);
		if (pOldCurrentTrans == null)
		{		
			//the keyboard signal was disconnected so connect it back
			connect(m_transitionViewWidg, SIGNAL(itemActivated(QTreeWidgetItem*, int)), desEditor(), SLOT(onChangeTransition()));
		}

		m_deleteTransAct->setEnabled(true);
		if (pOldCurrentTrans == null)
		{		
			//the keyboard signal was disconnected so connect it back
			connect(m_transitionViewWidg, SIGNAL(onCurrentItemDeleted()), desEditor(), SLOT(onDeleteTransition()));
		}
	}
	else
	{
		//the current transition doesn't exist so disable any actions related to it
		m_changeTransAct->setEnabled(false);
		disconnect(m_transitionViewWidg, SIGNAL(itemActivated(QTreeWidgetItem*, int)), desEditor(), SLOT(onChangeTransition()));
		
		m_deleteTransAct->setEnabled(false);
		disconnect(m_transitionViewWidg, SIGNAL(onCurrentItemDeleted()), desEditor(), SLOT(onDeleteTransition()));		
	}
}

//_________________________________________________________________________________________________

void DesWorkspace::updateCurrentTransUi(const DesTransition& /*crtTrans*/)
{
	//the current transition doesn't have any UI associated with it.
}

//_________________________________________________________________________________________________

void DesWorkspace::onChangedCurrentSelfTrans(const DesEvent* pNewCrtSelfTransEvent, 
											 const DesEvent* pOldCrtSelfTransEvent)
{
	if (pNewCrtSelfTransEvent)
	{
		//a current transition was established so configure the actions related with the current transition
		m_deleteSelfTransAct->setEnabled(true);
		if (pOldCrtSelfTransEvent == null)
		{		
			//the keyboard signal was disconnected so connect it back
			connect(m_selfTransWidg, SIGNAL(onCurrentItemDeleted()), desEditor(), SLOT(onDeleteSelfTransition()));
		}
	}
	else
	{
		//the current transition doesn't exist so disable any actions related to it
		m_deleteSelfTransAct->setEnabled(false);
		disconnect(m_selfTransWidg, SIGNAL(onCurrentItemDeleted()), desEditor(), SLOT(onDeleteSelfTransition()));		
	}
}

//_________________________________________________________________________________________________

DesEditor* DesWorkspace::desEditor()
{
	return dynamic_cast<DesEditor*>(form());
}

//_________________________________________________________________________________________________

void DesWorkspace::setupConnections()
{
	connect(m_pTransEdUiPart, SIGNAL(onChangedCurrentTrans(const DesTransition*, const DesTransition*)), 
		    this, SLOT(onChangedCurrentTrans(const DesTransition*, const DesTransition*))) ;

	connect(m_pTransEdUiPart, SIGNAL(onChangedActiveTrans(const DesTransition*, const DesTransition*)), 
		    this, SLOT(onChangedCurrentTrans(const DesTransition*, const DesTransition*))) ;

	connect(m_pTransEdUiPart, SIGNAL(onCurrentTransChanged(const DesTransition&)), 
		    this, SLOT(updateCurrentTransUi(const DesTransition&))) ;

	connect(m_pTransEdUiPart, SIGNAL(onChangedCurrentSelfTrans(const DesEvent*, const DesEvent*)), 
		    this, SLOT(onChangedCurrentSelfTrans(const DesEvent*, const DesEvent*))) ;

	connect(m_pTransEdUiPart, SIGNAL(onChangedActiveSelfTrans(const DesEvent*, const DesEvent*)), 
		    this, SLOT(onChangedCurrentSelfTrans(const DesEvent*, const DesEvent*))) ;
	connect(this,SIGNAL(setInstMode(bool)),this,SLOT(DisableAll(bool)));


}

//_________________________________________________________________________________________________

//Arrange the position of all splitters to have the following arrangement:
//		- States and Events are shown equaly the vertical spliter being right in the middle
//		- States & Events area occupy ~25% of the horizontal spliter
//		- Transitions occupy ~40% of the horizontal splitter
//		- Quick State Look occupies the rest of the ~35 of the horizontal spliter
void DesWorkspace::positionSplitters()
{
	//Calculate the 80/20 height usage of the main splitter that divides the des splitter and the output splitter
	const float cDesAreaVRatio = 4.0f/5.0f;
	
	QSize mainSplitterSize = m_mainSplitterWidg->size();
        int desAreaHeight = (int)(mainSplitterSize.height() * cDesAreaVRatio);
        int outputAreaHeight = (int)(mainSplitterSize.height() - desAreaHeight);

	//arrange the main splitter to have the above sizes
	QList<int> mainSplitterSizeList;
	mainSplitterSizeList.push_back(desAreaHeight);
	mainSplitterSizeList.push_back(outputAreaHeight);
	m_mainSplitterWidg->setSizes(mainSplitterSizeList);

	//Calculate the 33/33/33 width sizes for the des splitter
	const float cStateHRatio = 3;
	QSize desSplitterSize   = m_desSplitterWidg->size();
        int   stateAreaWidth = (int)(desSplitterSize.width() / cStateHRatio);
	int   eventAreaWidth = stateAreaWidth;
	int   transitionAreaWidth = desSplitterSize.width() - (stateAreaWidth + eventAreaWidth);

	//arrange the des area spliter to have the above sizes
	QList<int> desSplitterSizeList;
	desSplitterSizeList.push_back(stateAreaWidth);
	desSplitterSizeList.push_back(eventAreaWidth);
	desSplitterSizeList.push_back(transitionAreaWidth);
	m_desSplitterWidg->setSizes(desSplitterSizeList);

	//States and State Quick Look are shown equaly the vertical spliter being right in the middle
	const double cStateQuickLookRatio = 1.7;
	QSize stateSplitterSize = m_stateSplitterWidg->size();
        int   stateAreaHeight = (int)(stateSplitterSize.height() / cStateQuickLookRatio);
	int   quickLookAreaHeight = stateSplitterSize.height() - stateAreaHeight; // the rest of the splitter; 
	
	//arrange the state list - state quick view splitter to have the above sizes
	QList<int> stateSpliterSizeList;
	stateSpliterSizeList.push_back(stateAreaHeight);
	stateSpliterSizeList.push_back(quickLookAreaHeight);
	m_stateSplitterWidg->setSizes(stateSpliterSizeList);

	//Regular transitions and self-transitions are shown 70% / 30%
	const float cTransRatio = 1.7f;
	QSize transSplitterSize = m_transSplitterWidg->size();
        int   transAreaHeight = (int)(transSplitterSize.height() / cTransRatio);
	int   selfTransAreaHeight = transSplitterSize.height() - transAreaHeight; //the rest of the splitter;

	//arange the state transition - self-trans splitter to have the above sizes
	QList<int> transSplitterSizeList;
	transSplitterSizeList.push_back(transAreaHeight);
	transSplitterSizeList.push_back(selfTransAreaHeight);
	m_transSplitterWidg->setSizes(transSplitterSizeList);
}

//_________________________________________________________________________________________________

QAction* DesWorkspace::getActionForEventType(EventType eventType)
{
	switch(eventType)
	{
		case eDefaultEvent:
			return m_setEventTypeDefaultAct;

		case eRequestEvent:
			return m_setEventTypeRequestAct;

		case eAnswerEvent:
			return m_setEventTypeAnswerAct;

		case eLDataEvent:
			return m_setEventTypeLDataAct;

		default:
			assert(false);
			throw EX("Unknown event type. Cannot return label.")
	}
}

//_________________________________________________________________________________________________

void DesWorkspace::createUiParts(DesEditor* pDesEditor, Des* pDes)
{
	m_pStateQuickLookUiPart = new StateQuickLookUiPart(this, pDes);
	m_pStateEdUiPart		= new StateEditorUiPart(this, this, pDes, m_pStateQuickLookUiPart, pDesEditor->getTransRecorder());
	m_pEventEdUiPart		= new EventEditorUiPart(this, this, pDes, pDesEditor->getTransRecorder());
	m_pTransEdUiPart		= new TransitionEditorUiPart(this,pDes);
	m_pOutputUiPart			= new DesEditorOutputUiPart(this);
}

//_________________________________________________________________________________________________

void DesWorkspace::createActions()
{
	//#pragma region create DES actions
	//create DES actions
	{
		m_newDesAct = new QAction(QIcon(":/resources/newRegularDes.png"), tr("New &DES..."), this);
		m_newDesAct->setShortcut(tr("Ctrl+D"));
		m_newDesAct->setToolTip(tr("New DES..."));
		m_newDesAct->setStatusTip(tr("Create a new DES with given name and type"));
		connect(m_newDesAct, SIGNAL(triggered()), mainForm(), SLOT(onNewDes()));

		m_openDesAct = new QAction(QIcon(":/resources/openDes.png"), tr("O&pen DES..."), this);
		m_openDesAct->setShortcut(tr("Ctrl+Alt+O"));
		m_openDesAct->setStatusTip(tr("Open existing DES"));
		connect(m_openDesAct, SIGNAL(triggered()), mainForm(), SLOT(onOpenDes()));

		m_saveDesAct = new QAction(QIcon(":/resources/saveDes.png"), tr("&Save"), this);
		m_saveDesAct->setShortcut(tr("Ctrl+S"));
		m_saveDesAct->setStatusTip(tr("Saves DES to a file."));
		connect(m_saveDesAct, SIGNAL(triggered()), desEditor(), SLOT(onSaveDes()));

		m_saveDesAsAct = new QAction(tr("Save &Duplicate..."), this);
		m_saveDesAsAct->setStatusTip(tr("Saves the current DES to the specified file."));
		connect(m_saveDesAsAct, SIGNAL(triggered()), desEditor(), SLOT(onSaveDesAs()));

		m_revertToFileAct = new QAction(tr("Revert to File"), this);
		m_revertToFileAct->setStatusTip(tr("Reverts the current DES to what currenty is stored in the corresponding *des file."));
		connect(m_revertToFileAct, SIGNAL(triggered()), desEditor(), SLOT(onRevertToFile()));

		m_desSetNameAct = new QAction(tr("Set &Name..."), this);
		m_desSetNameAct->setStatusTip(tr("Sets the name of DES"));
		connect(m_desSetNameAct, SIGNAL(triggered()), desEditor(), SLOT(onDesSetName()));

		m_desSetTypeAct = new QAction(tr("Set &Type..."), this);
		m_desSetTypeAct->setStatusTip(tr("Sets the type of DES"));
		connect(m_desSetTypeAct, SIGNAL(triggered()), desEditor(), SLOT(onDesSetType()));

		m_printDesAct = new QAction(tr("&Print to file..."), this);
		m_printDesAct->setShortcut(tr("Ctrl+P"));
		m_printDesAct->setStatusTip(tr("Prints the contents of the DES to text file."));
		connect(m_printDesAct, SIGNAL(triggered()), desEditor(), SLOT(onPrintDes()));

		m_closeDesEditorAct = new QAction(tr("&Close"), this);
		m_closeDesEditorAct->setStatusTip(tr("Closes the DES editor"));
		connect(m_closeDesEditorAct, SIGNAL(triggered()), desEditor(), SLOT(onClose()));
	}
	//#pragma endregion 

	//#pragma region create state actions
	//create state actions
	{
		m_addStateAct = new QAction(QIcon(":/resources/addState.png"), tr("&Add State"), this);
		m_addStateAct->setShortcut(tr("Ctrl+A, Ctrl+S"));
		m_addStateAct->setStatusTip(tr("Adds a new state to DES"));
		connect(m_addStateAct, SIGNAL(triggered()), desEditor(), SLOT(onAddState()));

		m_changeStateAct = new QAction(QIcon(":/resources/renameState.png"), tr("&Change State"), this);
		m_changeStateAct->setStatusTip(tr("Change currently selected state in DES"));
		m_changeStateAct->setEnabled(false); //initially disabled until a current state is established
		connect(m_changeStateAct, SIGNAL(triggered()), desEditor(), SLOT(onChangeState()));

		m_deleteStateAct = new QAction(QIcon(":/resources/deleteState.png"), tr("&Delete State"), this);
		m_deleteStateAct->setStatusTip(tr("Deletes currently selected  from DES"));
		m_deleteStateAct->setEnabled(false); //initially disabled until a current state is established
		connect(m_deleteStateAct, SIGNAL(triggered()), desEditor(), SLOT(onDeleteState()));

		m_toggleInitialStateAct = new QAction(tr("&Initial State"), this);
		m_toggleInitialStateAct->setStatusTip(tr("Toggles the initial state status of the currently active state"));
		m_toggleInitialStateAct->setCheckable(true);
		m_toggleInitialStateAct->setEnabled(false); //initially disabled until a current state is established
		connect(m_toggleInitialStateAct, SIGNAL(triggered()), desEditor(), SLOT(onToggleInitialState()));

		m_toggleMarkedStateAct = new QAction(tr("&Marked State"), this);
		m_toggleMarkedStateAct->setStatusTip(tr("Toggles the initial marked status of the currently active state"));
		m_toggleMarkedStateAct->setCheckable(true);
		m_toggleMarkedStateAct->setEnabled(false); //initially disabled until a current state is established
		connect(m_toggleMarkedStateAct, SIGNAL(triggered()), desEditor(), SLOT(onToggleMarkedState()));

		m_markAllStatesAct = new QAction(tr("Mar&k All States"), this);
		m_markAllStatesAct->setStatusTip(tr("Makes all existing states marked states"));
		connect(m_markAllStatesAct, SIGNAL(triggered()), desEditor(), SLOT(onMarkAllStates()));

		m_unmarkAllStatesAct = new QAction(tr("&Unmark All States"), this);
		m_unmarkAllStatesAct->setStatusTip(tr("Makes all existing states unmarked states"));
		connect(m_unmarkAllStatesAct, SIGNAL(triggered()), desEditor(), SLOT(onUnmarkAllStates()));

		#ifdef __ENABLE_NEXT_VERSION__		
			m_sortStatesByNameAct = new QAction(tr("&Name"), this);
			m_sortStatesByNameAct->setCheckable(true);
			m_sortStatesByNameAct->setStatusTip(tr("Sorts the list of states by the state's name"));
			connect(m_sortStatesByNameAct, SIGNAL(triggered()), desEditor(), SLOT(onSortStatesByName()));

			m_sortStatesByCreationAct = new QAction(tr("&Creation"), this);
			m_sortStatesByCreationAct->setCheckable(true);
			m_sortStatesByCreationAct->setStatusTip(tr("Sorts the list of states by the order of their creation"));
			connect(m_sortStatesByCreationAct, SIGNAL(triggered()), desEditor(), SLOT(onSortStatesByCreation()));
			
			m_sortStatesByUserDefAct = new QAction(tr("&User Defined"), this);
			m_sortStatesByUserDefAct->setCheckable(true);
			m_sortStatesByUserDefAct->setStatusTip(tr("Sorts the list of states by the custom order setup by user"));
			connect(m_sortStatesByUserDefAct, SIGNAL(triggered()), desEditor(), SLOT(onSortStatesByUserDefined()));

			m_sortStatesActGroup = new QActionGroup(this);
			m_sortStatesActGroup->addAction(m_sortStatesByNameAct);
			m_sortStatesActGroup->addAction(m_sortStatesByCreationAct);
			m_sortStatesActGroup->addAction(m_sortStatesByUserDefAct);
			m_sortStatesByCreationAct->setChecked(true);
		#endif
	}
	//#pragma endregion
	
	//#pragma region create event actions
	//create event actions
	{
		m_addEventAct = new QAction(QIcon(":/resources/addEvent.png"), tr("&Add Event"), this);
		m_addEventAct->setShortcut(tr("Ctrl+A, Ctrl+E"));
		m_addEventAct->setStatusTip(tr("Add new event to DES"));
		connect(m_addEventAct, SIGNAL(triggered()), desEditor(), SLOT(onAddEvent()));

		m_changeEventAct = new QAction(QIcon(":/resources/renameEvent.png"), tr("&Change Event"), this);
		m_changeEventAct->setStatusTip(tr("Edits the currently selected event in DES"));
		m_changeEventAct->setEnabled(false);
		connect(m_changeEventAct, SIGNAL(triggered()), desEditor(), SLOT(onChangeEvent()));
		
		m_deleteEventAct = new QAction(QIcon(":/resources/deleteEvent.png"), tr("&Delete Event"), this);
		m_deleteEventAct->setStatusTip(tr("Deletes the currently selected event from DES"));
		m_deleteEventAct->setEnabled(false);
		connect(m_deleteEventAct, SIGNAL(triggered()), desEditor(), SLOT(onDeleteEvent()));

		m_toggleEventCtrlAct = new QAction(tr("&Controllable Event"), this);
		m_toggleEventCtrlAct->setStatusTip(tr("Toggles the controllability of the currently selected event"));
		m_toggleEventCtrlAct->setCheckable(true);
		m_toggleEventCtrlAct->setEnabled(false);
		connect(m_toggleEventCtrlAct, SIGNAL(triggered()), desEditor(), SLOT(onToggleEventCtrl()));

		m_setEventTypeDefaultAct = new QAction(tr("&Default"), this);
		m_setEventTypeDefaultAct->setStatusTip(tr("Makes the currently selected event a default event"));
		m_setEventTypeDefaultAct->setCheckable(true);
		connect(m_setEventTypeDefaultAct, SIGNAL(triggered()), desEditor(), SLOT(onSetEventTypeDefault()));

		m_setEventTypeAnswerAct = new QAction(tr("&Answer"), this);
		m_setEventTypeAnswerAct->setStatusTip(tr("Makes the currently selected event an answer event"));
		m_setEventTypeAnswerAct->setCheckable(true);
		connect(m_setEventTypeAnswerAct, SIGNAL(triggered()), desEditor(), SLOT(onSetEventTypeAnswer()));
		
		m_setEventTypeRequestAct = new QAction(tr("&Request"), this);
		m_setEventTypeRequestAct->setStatusTip(tr("Makes the currently selected event a request event"));
		m_setEventTypeRequestAct->setCheckable(true);
		connect(m_setEventTypeRequestAct, SIGNAL(triggered()), desEditor(), SLOT(onSetEventTypeRequest()));

		m_setEventTypeLDataAct = new QAction(tr("&Low Data"), this);
		m_setEventTypeLDataAct->setStatusTip(tr("Makes the currently selected event a low-level data event"));
		m_setEventTypeLDataAct->setCheckable(true);
		connect(m_setEventTypeLDataAct, SIGNAL(triggered()), desEditor(), SLOT(onSetEventTypeLData()));

		m_clearEventTypeAct = new QAction(tr("&Clear"), this);
		m_clearEventTypeAct->setStatusTip(tr("Clears the type of the currently selected event"));
		m_clearEventTypeAct->setEnabled(false); //TODO figure out if we still need this command
		connect(m_clearEventTypeAct, SIGNAL(triggered()), desEditor(), SLOT(onClearEventType()));

		m_setEventTypeActGroup = new QActionGroup(this);
		m_setEventTypeActGroup->addAction(m_setEventTypeDefaultAct);
		m_setEventTypeActGroup->addAction(m_setEventTypeAnswerAct);
		m_setEventTypeActGroup->addAction(m_setEventTypeRequestAct);
		m_setEventTypeActGroup->addAction(m_setEventTypeLDataAct);
		m_setEventTypeActGroup->addAction(m_clearEventTypeAct);

		#ifdef __ENABLE_NEXT_VERSION__				
			m_sortEventsByNameAct = new QAction(tr("&Name"), this);
			m_sortEventsByNameAct->setCheckable(true);
			m_sortEventsByNameAct->setStatusTip(tr("Sorts the list of events by the event names"));
			connect(m_sortEventsByNameAct, SIGNAL(triggered()), desEditor(), SLOT(onSortEventsByName()));

			m_sortEventsByCtrlAct = new QAction(tr("&Controllability"), this);
			m_sortEventsByCtrlAct->setCheckable(true);
			m_sortEventsByCtrlAct->setStatusTip(tr("Sorts the list of events by the event's controllability"));
			connect(m_sortEventsByCtrlAct, SIGNAL(triggered()), desEditor(), SLOT(onSortEventsByCtrl()));

			m_sortEventsByTypeAct = new QAction(tr("&Type"), this);
			m_sortEventsByTypeAct->setCheckable(true);
			m_sortEventsByTypeAct->setStatusTip(tr("Sorts the list of events by the event's type"));
			connect(m_sortEventsByTypeAct, SIGNAL(triggered()), desEditor(), SLOT(onSortEventsByType()));

			m_sortEventsByCreationAct = new QAction(tr("&Creation"), this);
			m_sortEventsByCreationAct->setCheckable(true);
			m_sortEventsByCreationAct->setStatusTip(tr("Sorts the list of events by the event's type"));
			connect(m_sortEventsByCreationAct, SIGNAL(triggered()), desEditor(), SLOT(onSortEventsByCreation()));

			m_sortEventsByUserDefAct = new QAction(tr("&User Defined"), this);
			m_sortEventsByUserDefAct->setCheckable(true);
			m_sortEventsByUserDefAct->setStatusTip(tr("Sorts the list of events by the order manually defined by the user"));
			connect(m_sortEventsByUserDefAct, SIGNAL(triggered()), desEditor(), SLOT(onSortEventsByUserDefined()));

			m_sortEventsActGroup = new QActionGroup(this);
			m_sortEventsActGroup->addAction(m_sortEventsByNameAct);
			m_sortEventsActGroup->addAction(m_sortEventsByCtrlAct);
			m_sortEventsActGroup->addAction(m_sortEventsByTypeAct);
			m_sortEventsActGroup->addAction(m_sortEventsByCreationAct);
			m_sortEventsActGroup->addAction(m_sortEventsByUserDefAct);
			m_sortEventsByCreationAct->setChecked(true);
		#endif
	}
	//#pragma endregion

	//#pragma region create transition actions
	//create transition actions
	{
		m_recordTransAct = new QAction(QIcon(":/resources/recordTransitions.png"), tr("&Record Transitions"), this);
		m_recordTransAct->setCheckable(true);
		m_recordTransAct->setStatusTip(tr("Starts / Stops the process of recording transitions by clicking states and events in their respective lists"));
		connect(m_recordTransAct, SIGNAL(triggered()), desEditor(), SLOT(onRecordTransitions()));

		m_addTransAct = new QAction(QIcon(":/resources/addTransition.png"), tr("&Add Transition"), this);
		m_addTransAct->setStatusTip(tr("Adds a new transition to DES"));
		connect(m_addTransAct, SIGNAL(triggered()), desEditor(), SLOT(onAddTransition()));

		m_addSelfTransAct = new QAction(QIcon(":/resources/addSelfTransition.png"), tr("&Add Global Self-Loop"), this);
		m_addSelfTransAct->setStatusTip(tr("Adds a new transition to DES"));
		connect(m_addSelfTransAct, SIGNAL(triggered()), desEditor(), SLOT(onAddSelfTransition()));

		m_changeTransAct = new QAction(QIcon(":/resources/changeTransition.png"), tr("&Change Transition"), this);
		m_changeTransAct->setStatusTip(tr("Renames the currently selected transition in DES"));
		m_changeTransAct->setEnabled(false);
		connect(m_changeTransAct, SIGNAL(triggered()), desEditor(), SLOT(onChangeTransition()));

		m_deleteTransAct = new QAction(QIcon(":/resources/deleteTransition.png"), tr("&Delete Transition"), this);
		m_deleteTransAct->setStatusTip(tr("Deletes the currently selected transition from DES"));
		m_deleteTransAct->setEnabled(false);
		connect(m_deleteTransAct, SIGNAL(triggered()), desEditor(), SLOT(onDeleteTransition()));

		m_deleteSelfTransAct = new QAction(QIcon(":/resources/deleteSelfTransition.png"), tr("&Delete Global Self-Loop"), this);
		m_deleteSelfTransAct->setStatusTip(tr("Deletes the currently selected global self-loop from DES"));
		m_deleteSelfTransAct->setEnabled(false);
		connect(m_deleteSelfTransAct, SIGNAL(triggered()), desEditor(), SLOT(onDeleteSelfTransition()));

		#ifdef __ENABLE_NEXT_VERSION__				
			m_sortTransByFromStateAct = new QAction(tr("&From State"), this);
			m_sortTransByFromStateAct->setCheckable(true);
			m_sortTransByFromStateAct->setStatusTip(tr("Sorts the transitions by the name of their from-state"));
			connect(m_sortTransByFromStateAct, SIGNAL(triggered()), desEditor(), SLOT(onSortTransByFromState()));

			m_sortTransByToStateAct = new QAction(tr("&To State"), this);
			m_sortTransByToStateAct->setCheckable(true);
			m_sortTransByToStateAct->setStatusTip(tr("Sorts the transitions by the name of their to-state"));
			connect(m_sortTransByToStateAct, SIGNAL(triggered()), desEditor(), SLOT(onSortTransByToState()));

			m_sortTransByEventNameAct = new QAction(tr("Event &Name"), this);
			m_sortTransByEventNameAct->setCheckable(true);
			m_sortTransByEventNameAct->setStatusTip(tr("Sorts the transitions by the name of their event"));
			connect(m_sortTransByEventNameAct, SIGNAL(triggered()), desEditor(), SLOT(onSortTransByEventName()));

			m_sortTransByEventTypeAct = new QAction(tr("Event &Type"), this);
			m_sortTransByEventTypeAct->setCheckable(true);
			m_sortTransByEventTypeAct->setStatusTip(tr("Sorts the transitions by the type of their event"));
			connect(m_sortTransByEventTypeAct, SIGNAL(triggered()), desEditor(), SLOT(onSortTransByEventType()));

			m_sortTransByEventCtrlAct = new QAction(tr("Event &Controllability"), this);
			m_sortTransByEventCtrlAct->setCheckable(true);
			m_sortTransByEventCtrlAct->setStatusTip(tr("Sorts the transitions by their controllability property"));
			connect(m_sortTransByEventCtrlAct, SIGNAL(triggered()), desEditor(), SLOT(onSortTransByEventType()));

			m_sortTransByCreationAct = new QAction(tr("&Creation"), this);
			m_sortTransByCreationAct->setCheckable(true);
			m_sortTransByCreationAct->setStatusTip(tr("Sorts the transitions by their controllability property"));
			connect(m_sortTransByCreationAct, SIGNAL(triggered()), desEditor(), SLOT(onSortTransByCreation()));
			 
			m_sortTransByUserDefAct = new QAction(tr("&User Defined"), this);
			m_sortTransByUserDefAct->setCheckable(true);
			m_sortTransByUserDefAct->setStatusTip(tr("Sorts the transitions in the order of their creation"));
			connect(m_sortTransByUserDefAct, SIGNAL(triggered()), desEditor(), SLOT(onSortTransByCreation()));

			m_sortTransActGroup = new QActionGroup(this);
			m_sortTransActGroup->addAction(m_sortTransByFromStateAct);
			m_sortTransActGroup->addAction(m_sortTransByToStateAct);
			m_sortTransActGroup->addAction(m_sortTransByEventNameAct);
			m_sortTransActGroup->addAction(m_sortTransByEventTypeAct);
			m_sortTransActGroup->addAction(m_sortTransByEventCtrlAct);
			m_sortTransActGroup->addAction(m_sortTransByCreationAct);
			m_sortTransActGroup->addAction(m_sortTransByUserDefAct);
			m_sortTransByCreationAct->setChecked(true);
		#endif
	}
	//#pragma endregion

	//#pragma region create tools actions
	//create tools actions
	{
		m_checkDesIntegAct = new QAction(QIcon(":/resources/checkDesConsist.png"), tr("&Verify DES Integrity"), this);
		m_checkDesIntegAct->setStatusTip(tr("Verifies the integrity of the DES"));
		connect(m_checkDesIntegAct, SIGNAL(triggered()), desEditor(), SLOT(onCheckDesIntegrity()));

		m_runReachToolAct = new QAction(QIcon(":/resources/reachDes.png"), tr("&Check Reachability"), this);
		m_runReachToolAct->setStatusTip(tr("Checks that DES is reachable - every state is reachable from the initial state"));
		connect(m_runReachToolAct, SIGNAL(triggered()), desEditor(), SLOT(onRunReachableTool()));

		/* Ticket # 52 - remove coreachability check (remove visibility only)
		m_runCoreachToolAct = new QAction(QIcon(":/resources/coReachDes.png"), tr("&Check Coreachability"), this);
		m_runCoreachToolAct->setStatusTip(tr("Checks that DES is coreachable - from any state DES can get to a marked state"));
		connect(m_runCoreachToolAct, SIGNAL(triggered()), desEditor(), SLOT(onRunCoreachableTool()));
		*/

		m_runNonblockToolAct = new QAction(QIcon(":/resources/checkNonBlocking.png"), tr("&Check Nonblocking"), this);
		m_runNonblockToolAct->setStatusTip(tr("Checks that every reachable state is coreachable"));
		connect(m_runNonblockToolAct, SIGNAL(triggered()), desEditor(), SLOT(onRunNonBlockTool()));
		
		#ifdef __ENABLE_NEXT_VERSION__				
			m_runIsomorphToolAct = new QAction(QIcon(":/resources/isomorph.png"), tr("&Isomorph..."), this);
			m_runIsomorphToolAct->setStatusTip(tr("Checks whether DES is identical to another up to relabling"));
			connect(m_runIsomorphToolAct, SIGNAL(triggered()), desEditor(), SLOT(onRunIsomorphTool()));

			m_runMultiTools = new QAction(tr("&Run Tools..."), this);
			m_runMultiTools->setToolTip(tr("Run Multiple Tools"));
			m_runMultiTools->setStatusTip(tr("Allows user to select multiple tools to run."));
			connect(m_runMultiTools, SIGNAL(triggered()), desEditor(), SLOT(onRunMultiTools()));		
		#endif
	}
	//#pragma endregion

	//#pragma region create "Help" actions
	//create "Help" actions
	{
		m_openHelpFileAct = new QAction(tr("&Help Contents"), this);
		m_openHelpFileAct->setStatusTip("Opens the application help file");
		connect(m_openHelpFileAct, SIGNAL(triggered()), mainForm(), SLOT(onOpenHelpFile()));
		
		m_aboutDespotAct = new QAction(tr("&About"), this);
		m_aboutDespotAct->setStatusTip("Displays information about the DESpot application");
		connect(m_aboutDespotAct, SIGNAL(triggered()), mainForm(), SLOT(onAboutDespot()));
	}
	//#pragma endregion

	#ifdef __ENABLE_NEXT_VERSION__			
		//create the "User Defined Order" actions used in toolbar only
		{
			m_moveFirstAct = new QAction(QIcon(":/resources/moveFirst.png"), tr("Move &First"), this);
			m_moveFirstAct->setStatusTip("Moves the currently selected item to be first in its list");
			connect(m_moveFirstAct, SIGNAL(triggered()), desEditor(), SLOT(onMoveItemFirst()));

			m_moveUpAct = new QAction(QIcon(":/resources/moveUp.png"), tr("Move &Up"), this);
			m_moveUpAct->setStatusTip("Moves the currently selected item up one position in its list");
			connect(m_moveUpAct, SIGNAL(triggered()), desEditor(), SLOT(onMoveItemUp()));
		
			m_moveDownAct = new QAction(QIcon(":/resources/moveDown.png"), tr("Move &Down"), this);
			m_moveDownAct->setStatusTip("Moves the currently selected item down one position in its list");
			connect(m_moveDownAct, SIGNAL(triggered()), desEditor(), SLOT(onMoveItemDown()));
			
			m_moveLastAct = new QAction(QIcon(":/resources/moveLast.png"), tr("Move &Last"), this);
			m_moveLastAct->setStatusTip("Moves the currently selected item to be last in its list");
			connect(m_moveLastAct, SIGNAL(triggered()), desEditor(), SLOT(onMoveItemLast()));		
		}
	#endif

}

void DesWorkspace::setupMenu()
{
	//#pragma region setup the DES menu
	//setup the DES menu
	{
		m_desMenu = desEditor()->menuBar()->addMenu(tr("&Des"));
		m_desMenu->addAction(m_newDesAct);
		m_desMenu->addAction(m_openDesAct);
		m_desMenu->addAction(m_saveDesAct);
		m_desMenu->addAction(m_saveDesAsAct);
		m_desMenu->addAction(m_revertToFileAct);
		m_desMenu->addSeparator();
		m_desMenu->addAction(m_desSetNameAct);
		m_desMenu->addAction(m_desSetTypeAct);
		m_desMenu->addSeparator();
		m_desMenu->addAction(m_printDesAct);
		m_desMenu->addSeparator();
		m_desMenu->addAction(m_closeDesEditorAct);
	}
	//#pragma endregion

	//#pragma region setup the States menu
	//setup the States menu
	{
		m_statesMenu = desEditor()->menuBar()->addMenu(tr("&States"));
		m_statesMenu->addAction(m_addStateAct);
		m_statesMenu->addAction(m_changeStateAct);
		m_statesMenu->addAction(m_deleteStateAct);
		m_statesMenu->addSeparator();
		m_statesMenu->addAction(m_toggleInitialStateAct);
		m_statesMenu->addAction(m_toggleMarkedStateAct);
		m_statesMenu->addSeparator();
		m_statesMenu->addAction(m_markAllStatesAct);
		m_statesMenu->addAction(m_unmarkAllStatesAct);
		
		#ifdef __ENABLE_NEXT_VERSION__		
			m_statesMenu->addSeparator();
			{
				m_sortStatesBySubMenu = m_statesMenu->addMenu(tr("&Sort By"));
				m_sortStatesBySubMenu->addAction(m_sortStatesByNameAct);
				m_sortStatesBySubMenu->addSeparator();
				m_sortStatesBySubMenu->addAction(m_sortStatesByCreationAct);
				m_sortStatesBySubMenu->addAction(m_sortStatesByUserDefAct);
			}
		#endif
	}
	//#pragma endregion

	//#pragma region setup the Events menu
	//setup the Events menu
	{
		m_eventsMenu = desEditor()->menuBar()->addMenu(tr("&Events"));
		m_eventsMenu->addAction(m_addEventAct);
		m_eventsMenu->addAction(m_changeEventAct);
		m_eventsMenu->addAction(m_deleteEventAct);
		m_eventsMenu->addSeparator();
		m_eventsMenu->addAction(m_toggleEventCtrlAct);
		{
			m_eventTypeSubMenu = m_eventsMenu->addMenu(tr("Set &Type"));
			m_eventTypeSubMenu->addAction(m_setEventTypeDefaultAct);
			m_eventTypeSubMenu->addSeparator();
			m_eventTypeSubMenu->addAction(m_setEventTypeRequestAct);
			m_eventTypeSubMenu->addAction(m_setEventTypeAnswerAct);
			m_eventTypeSubMenu->addAction(m_setEventTypeLDataAct);
			m_eventTypeSubMenu->addSeparator();
			m_eventTypeSubMenu->addAction(m_clearEventTypeAct);

			//disable the menu to start with
			m_eventTypeSubMenu->menuAction()->setEnabled(false);
		}
		
		#ifdef __ENABLE_NEXT_VERSION__		
			m_eventsMenu->addSeparator();
			{
				m_sortEventsBySubMenu = m_eventsMenu->addMenu(tr("&Sort By"));
				m_sortEventsBySubMenu->addAction(m_sortEventsByNameAct);
				m_sortEventsBySubMenu->addAction(m_sortEventsByCtrlAct);
				m_sortEventsBySubMenu->addAction(m_sortEventsByTypeAct);
				m_sortEventsBySubMenu->addSeparator();
				m_sortEventsBySubMenu->addAction(m_sortEventsByCreationAct);
				m_sortEventsBySubMenu->addAction(m_sortEventsByUserDefAct);
			}
		#endif
	}
	//#pragma endregion

	//#pragma region setup the Transitions menu
	//setup the Transitions menu
	{
		m_transMenu = desEditor()->menuBar()->addMenu(tr("&Transitions"));
		m_transMenu->addAction(m_recordTransAct);
		m_transMenu->addSeparator();
		m_transMenu->addAction(m_addTransAct);
		m_transMenu->addAction(m_addSelfTransAct);
		m_transMenu->addAction(m_changeTransAct);
		m_transMenu->addAction(m_deleteTransAct);
		m_transMenu->addAction(m_deleteSelfTransAct);
		
		#ifdef __ENABLE_NEXT_VERSION__		
			m_transMenu->addSeparator();
			{
				m_sortTransBySubMenu = m_transMenu->addMenu(tr("Sort By"));
				m_sortTransBySubMenu->addAction(m_sortTransByFromStateAct);
				m_sortTransBySubMenu->addAction(m_sortTransByToStateAct);
				m_sortTransBySubMenu->addSeparator();
				m_sortTransBySubMenu->addAction(m_sortTransByEventNameAct);
				m_sortTransBySubMenu->addAction(m_sortTransByEventTypeAct);
				m_sortTransBySubMenu->addAction(m_sortTransByEventCtrlAct);
				m_sortTransBySubMenu->addSeparator();
				m_sortTransBySubMenu->addAction(m_sortTransByCreationAct);
				m_sortTransBySubMenu->addAction(m_sortTransByUserDefAct);
			}
		#endif
	}
	//#pragma endregion

	//#pragma region setup the Tools menu
	//setup the Tools menu
	{
		m_toolsMenu = desEditor()->menuBar()->addMenu(tr("&Tools"));
		m_toolsMenu->addAction(m_checkDesIntegAct);
		m_toolsMenu->addSeparator();
		m_toolsMenu->addAction(m_runReachToolAct);
		/* Ticket # 52 remove coreachability check (remove visibility only)
		m_toolsMenu->addAction(m_runCoreachToolAct);
		*/
		m_toolsMenu->addAction(m_runNonblockToolAct);
		
		#ifdef __ENABLE_NEXT_VERSION__		
			m_toolsMenu->addAction(m_runIsomorphToolAct);
			m_toolsMenu->addSeparator();
			m_toolsMenu->addAction(m_runMultiTools);		
		#endif
	}

	//#pragma endregion
	
	//#pragma region setup the Help menu
	desEditor()->menuBar()->addSeparator();
	//setup the Help menu
	{
		m_helpMenu = desEditor()->menuBar()->addMenu(tr("&Help"));
		m_helpMenu->addAction(m_openHelpFileAct);
		m_helpMenu->addSeparator();
		m_helpMenu->addAction(m_aboutDespotAct);
	}
	//#pragma endregion

	//#pragma region setup the state editor context menu
	//setup the state editor context menu
	{
		QMenu& stateEditorCtxMenu = m_pStateEdUiPart->getContextMenu();
		stateEditorCtxMenu.addAction(m_addStateAct);
		stateEditorCtxMenu.addAction(m_changeStateAct);
		stateEditorCtxMenu.addAction(m_deleteStateAct);
		
		stateEditorCtxMenu.addSeparator();

		stateEditorCtxMenu.addAction(m_toggleInitialStateAct);
		stateEditorCtxMenu.addAction(m_toggleMarkedStateAct);		
	}
	//#pragma endregion

	//#pragma region setup the event editor context menu
	{
		QMenu& eventEditorContextMenu = m_pEventEdUiPart->getContextMenu();
		eventEditorContextMenu.addAction(m_addEventAct);
		eventEditorContextMenu.addAction(m_changeEventAct);
		eventEditorContextMenu.addAction(m_deleteEventAct);

		eventEditorContextMenu.addSeparator();

		eventEditorContextMenu.addAction(m_toggleEventCtrlAct);
		{
			QMenu& eventTypeSubContextMenu = m_pEventEdUiPart->getEventTypeContextSubMenu();
			eventTypeSubContextMenu.addAction(m_setEventTypeDefaultAct);
			eventTypeSubContextMenu.addSeparator();
			eventTypeSubContextMenu.addAction(m_setEventTypeRequestAct);
			eventTypeSubContextMenu.addAction(m_setEventTypeAnswerAct);
			eventTypeSubContextMenu.addAction(m_setEventTypeLDataAct);

			//disable the menu to start with
			eventTypeSubContextMenu.menuAction()->setEnabled(false);
		}
	}
	//#pragma endregion

	//#pragma region setup the transition editor context menu
	{
		QMenu& transEditorContextMenu = m_pTransEdUiPart->getTransContextMenu();
		transEditorContextMenu.addAction(m_recordTransAct);
		transEditorContextMenu.addSeparator();
		transEditorContextMenu.addAction(m_addTransAct);
		transEditorContextMenu.addAction(m_addSelfTransAct);
		transEditorContextMenu.addAction(m_changeTransAct);
		transEditorContextMenu.addAction(m_deleteTransAct);
	}

	//#pragma region setup the self-transition editor context menu
	{
		QMenu& selfTransEditorContextMenu = m_pTransEdUiPart->getSelfTransContextMenu();
		selfTransEditorContextMenu.addAction(m_addSelfTransAct);
		selfTransEditorContextMenu.addAction(m_deleteSelfTransAct);
	}
	//#pragma endregion
}

void DesWorkspace::setupToolBar()
{
	//#pragma region Create DES toolbar
	{
		m_desToolBar = desEditor()->addToolBar(tr("DES"));
		m_desToolBar->setIconSize(cToolBarIconSize);

		m_desToolBar->addAction(m_newDesAct);
		m_desToolBar->addAction(m_openDesAct);
		m_desToolBar->addAction(m_saveDesAct);
		m_desToolBar->addSeparator();
		m_desToolBar->addAction(m_addStateAct);
		m_desToolBar->addAction(m_changeStateAct);
		m_desToolBar->addAction(m_deleteStateAct);
		m_desToolBar->addSeparator();
		m_desToolBar->addAction(m_addEventAct);
		m_desToolBar->addAction(m_changeEventAct);
		m_desToolBar->addAction(m_deleteEventAct);
		m_desToolBar->addSeparator();
		m_desToolBar->addAction(m_recordTransAct);
		m_desToolBar->addAction(m_addTransAct);
		m_desToolBar->addAction(m_addSelfTransAct);
		m_desToolBar->addAction(m_changeTransAct);
		m_desToolBar->addAction(m_deleteTransAct);
		m_desToolBar->addAction(m_deleteSelfTransAct);
	}
	//#pragma endregion

	#ifdef __ENABLE_NEXT_VERSION__			
	{
		m_userDefOrdToolBar = desEditor()->addToolBar(tr("User Defined Order"));
		m_userDefOrdToolBar->setIconSize(cToolBarIconSize);

		m_userDefOrdToolBar->addAction(m_moveFirstAct);
		m_userDefOrdToolBar->addAction(m_moveUpAct);
		m_userDefOrdToolBar->addAction(m_moveDownAct);
		m_userDefOrdToolBar->addAction(m_moveLastAct);
	}
	#endif

	//#pragma region Create DES Tools toolbar
	{
		m_desToolsToolBar = desEditor()->addToolBar(tr("Tools"));
		m_desToolsToolBar->setIconSize(cToolBarIconSize);

		m_desToolsToolBar->addAction(m_checkDesIntegAct);
		m_desToolsToolBar->addSeparator();
		m_desToolsToolBar->addAction(m_runReachToolAct);
		/* Ticket # 52 - remove coreachability check (remove visibility only)
		m_desToolsToolBar->addAction(m_runCoreachToolAct);
		*/
		m_desToolsToolBar->addAction(m_runNonblockToolAct);
		
		#ifdef __ENABLE_NEXT_VERSION__		
			m_desToolsToolBar->addAction(m_runIsomorphToolAct);
		#endif
	}
	//#pragma endregion

}

void DesWorkspace::setupStatusBar()
{
	QStatusBar* statusBar = desEditor()->statusBar();	
	statusBar->addPermanentWidget(m_progressWidget);
	statusBar->showMessage(tr("Use the States, Events and Transitions to create DES"));
}
void DesWorkspace::DisableAll(bool)
{
	m_newDesAct->setEnabled(false);

	m_openDesAct->setEnabled(false);

	m_saveDesAct->setEnabled(false);

	m_saveDesAsAct->setEnabled(false);

	m_desSetNameAct->setEnabled(false);

	m_desSetTypeAct->setEnabled(false);

	m_addStateAct->setEnabled(false);

	m_changeStateAct->setEnabled(false);

	m_deleteStateAct->setEnabled(false);

	m_toggleInitialStateAct->setEnabled(false);
	
	m_toggleMarkedStateAct->setEnabled(false);

	m_markAllStatesAct->setEnabled(false);

	m_unmarkAllStatesAct->setEnabled(false);

	m_addEventAct->setEnabled(false);
	
	m_changeEventAct->setEnabled(false);
	
	m_deleteEventAct->setEnabled(false);
	
	m_toggleEventCtrlAct->setEnabled(false);
	
	m_setEventTypeDefaultAct->setEnabled(false);
	
	m_setEventTypeAnswerAct->setEnabled(false);
	
	m_setEventTypeRequestAct->setEnabled(false);
	
	m_setEventTypeLDataAct->setEnabled(false);
	
	m_clearEventTypeAct->setEnabled(false);

	m_recordTransAct->setEnabled(false);

	m_addTransAct->setEnabled(false);

	m_addSelfTransAct->setEnabled(false);

	m_changeTransAct->setEnabled(false);

	m_deleteTransAct->setEnabled(false);

	m_deleteSelfTransAct->setEnabled(false);

	m_checkDesIntegAct->setEnabled(false);

	m_runReachToolAct->setEnabled(false);

	//m_runCoreachToolAct->setEnabled(false);

        m_runNonblockToolAct->setEnabled(false);

	m_openHelpFileAct->setEnabled(false);

	m_aboutDespotAct->setEnabled(false);

	//saveAsImageAction->setEnabled(false);

	//restoreDefaultShapeAction->setEnabled(false);

       // deleteAction->setEnabled(false);
  
	//editAction->setEnabled(false);
      
	//editStateAction->setEnabled(false);
   
	//editTransAction->setEnabled(false);

   	//toggleBackgroundGrid->setEnabled(false);
	

	//m_stateQuickLookWidg->setEnabled(false);

	m_eventViewWidg->setEnabled(false);

	//m_outputWidg->setEnabled(false);

	m_mainSplitterWidg->setEnabled(false);
	
	m_progressWidget->setEnabled(false);
}

/**
 * If this is an instantiation of a template,
 * send a signal to disable the UI.
 */
void DesWorkspace::setMode(bool isInstantiation)
{
	// Check if it is an instantiation befor emitting.
	// The emitted signal disables the UI regardless of what
	// the passed value is.
	if (isInstantiation) {
		emit setInstMode(isInstantiation);
	}
}

} //end of namespace DESpot
