/*	Author:		Magdin Stoica
	Supervisor: Dr. Ryan Leduc
	
	Project created in conformity with the requirements for the Degree of Master of Engineering in Software Engineering, 
	Computing and Software Department, 
	McMaster University
	2003 - 2007
*/

#include <QHeaderView>
#include <QMessageBox>
#include <QResizeEvent>

#include "EventViewerWidget.h"
#include "Des.h"


namespace DESpot
{

const int EventViewerWidget::cColCount = 4;

const int EventViewerWidget::cEventNameColIdx = 0;
const int EventViewerWidget::cEventAliasColIdx = 1;
const int EventViewerWidget::cEventCtrlColIdx = 2;
const int EventViewerWidget::cEventTypeColIdx = 3;

const QString EventViewerWidget::cEventNameColName = "Name";
const QString EventViewerWidget::cEventAliasColName = "Alias";
const QString EventViewerWidget::cEventCtrlColName = "Ctrl";
const QString EventViewerWidget::cEventTypeColName = "Type";

const QString EventViewerWidget::cYes = "Yes";
const QString EventViewerWidget::cNo = "No";

const QString EventViewerWidget::cEventCtrlProp = "Controllable";
const QString EventViewerWidget::cEventTypeProp = "Event Type";
const QString EventViewerWidget::cEventUsedProp = "Used";

const QString EventViewerWidget::cDefaultTypeLabel		= tr("d");
const QString EventViewerWidget::cDefaultTypeAltLabel	= tr("D");
const QString EventViewerWidget::cDefaultTypeName		= tr("Default");

const QString EventViewerWidget::cHighLevelTypeLabel	= tr("h");
const QString EventViewerWidget::cHighLevelTypeAltLabel	= tr("H");
const QString EventViewerWidget::cHighLevelTypeName		= tr("High");

const QString EventViewerWidget::cLowLevelTypeLabel		= tr("l");
const QString EventViewerWidget::cLowLevelTypeAltLabel	= tr("L");
const QString EventViewerWidget::cLowLevelTypeName		= tr("Low");

const QString EventViewerWidget::cRequestTypeLabel		= tr("r");
const QString EventViewerWidget::cRequestTypeAltLabel	= tr("R");
const QString EventViewerWidget::cRequestTypeName		= tr("Request");

const QString EventViewerWidget::cAnswerTypeLabel		= tr("a");
const QString EventViewerWidget::cAnswerTypeAltLabel	= tr("A");
const QString EventViewerWidget::cAnswerTypeName		= tr("Answer");

const QString EventViewerWidget::cLDataTypeLabel		= tr("ld");
const QString EventViewerWidget::cLDataTypeAltLabel		= tr("LD");
const QString EventViewerWidget::cLDataTypeName			= tr("Low Data");

const QString EventViewerWidget::cPropCheckSymbol		= tr("x");


EventViewerWidget::EventViewerWidget(QWidget* parent) : DespotTreeWidget(parent)
{
	setupConnections();
}

//_________________________________________________________________________________________________

EventViewerWidget::~EventViewerWidget(void)
{
}

//_________________________________________________________________________________________________
			
void EventViewerWidget::setAllowEdit(bool allowEdit /*= true*/)
{
	m_allowEdit = allowEdit;
}

//_________________________________________________________________________________________________

void EventViewerWidget::setupConnections()
{
	connect(this, SIGNAL(currentItemChanged(QTreeWidgetItem*, QTreeWidgetItem*)), 
		    this, SLOT(onChangedCurrentEventItem(QTreeWidgetItem*, QTreeWidgetItem*)));
	
	connect(this, SIGNAL(onActiveItemChanged(QTreeWidgetItem*, QTreeWidgetItem*)), 
		    this, SLOT(onChangedActiveEventItem(QTreeWidgetItem*, QTreeWidgetItem*)));

	connect(this, SIGNAL(itemChanged(QTreeWidgetItem*, int)), 
		    this, SLOT(onEventItemChanged(QTreeWidgetItem*, int)));

	connect(this, SIGNAL(itemClicked(QTreeWidgetItem*, int)), 
		    this, SLOT(onEventItemClicked(QTreeWidgetItem*, int)));	
}

//_________________________________________________________________________________________________

void EventViewerWidget::initWidget()
{
	//setup header: Name | Alias | Ctrl | Type
	setColumnCount(getColumnCount());
	
	QStringList headers;
	getHeaders(headers);
	setHeaderLabels(headers);
	resizeHeaders(geometry().width());
	
	headerItem()->setTextAlignment(cEventCtrlColIdx, Qt::AlignHCenter);	
	headerItem()->setTextAlignment(cEventTypeColIdx, Qt::AlignHCenter);
	
	header()->setSortIndicatorShown(false);
}

//_________________________________________________________________________________________________

int EventViewerWidget::getColumnCount()
{
	return cColCount;
}

//_________________________________________________________________________________________________

void EventViewerWidget::getHeaders(QStringList& out_headerList)
{
	out_headerList << cEventNameColName << cEventAliasColName << cEventCtrlColName << cEventTypeColName;
}

//_________________________________________________________________________________________________

void EventViewerWidget::loadEvents(Des* des)
{
	Des::EventIteratorPtr eventIt = des->createEventIterator();
	for(eventIt->first(); eventIt->isDone() == false; eventIt->next())
	{
		const DesEvent& event = eventIt->currentItem();
		QTreeWidgetItem* pUiItem = createUiItemFromEvent(event);	
		addTopLevelItem(pUiItem);
	}
}

//_________________________________________________________________________________________________

bool EventViewerWidget::findEventItem(const QString& eventName, QTreeWidgetItem*& out_foundItem)
{
	QList<QTreeWidgetItem*> eventItems = findItems(eventName, Qt::MatchStartsWith | Qt::MatchRecursive);
	for(int iItem = 0; iItem < eventItems.size(); iItem++)
	{
		if (isEventUiItem(*(eventItems.at(iItem))))
		{
			out_foundItem = eventItems.at(iItem);
			return true;
		}
	}

	return false;
}

//_________________________________________________________________________________________________

QTreeWidgetItem* EventViewerWidget::addEvent(const DesEvent& event)
{
	QTreeWidgetItem* pEventUiItem = createUiItemFromEvent(event);
	addTopLevelItem(pEventUiItem);
	return pEventUiItem;
}

//_________________________________________________________________________________________________

QTreeWidgetItem*  EventViewerWidget::updateEvent(const DesEvent& event)
{
	QTreeWidgetItem* pEventUiItem = getUiItemFromEvent(event);

	//found UI item corresponding to the changed event. update it
	fillEventUiItem(*pEventUiItem, event);

	return pEventUiItem;
}

//_________________________________________________________________________________________________

void EventViewerWidget::removeEvent(const DesEvent& event)
{
	QTreeWidgetItem* pEventItem = getUiItemFromEvent(event);
	if (pEventItem == null)
	{
		assert(false);
		return; //nothing to delete; there is no item for this event
	}

	if (pEventItem == currentItem())
	{
		setCurrentItem(getNextSibling(currentItem()));
	}


	//Remove the item from the event viewer which corresponds to the event to be deleted
	takeTopLevelItem(indexOfTopLevelItem(pEventItem));
	delete pEventItem;
	pEventItem = null;
}

//_________________________________________________________________________________________________

void EventViewerWidget::onChangedCurrentEventItem(QTreeWidgetItem* current, QTreeWidgetItem* previous)
{
	if (previous == current)
		return; //nothing really changed

	//previously there might not have been a selected event so we need to check
	const DesEvent* pOldCurrentEvent = null;
	if (previous && isEventUiItem(*previous))
	{
		const DesEvent& prevCrtEvent = getEventFromUiItem(*previous);
		pOldCurrentEvent = &prevCrtEvent;
	}

	const DesEvent* pNewCurrentEvent = null;
	if (current && isEventUiItem(*current))
	{
		//obtain the currently selected event
		const DesEvent& crtEvent  = getEventFromUiItem(*current);
		pNewCurrentEvent = &crtEvent;
	}

	emit onChangedCurrentEvent(pNewCurrentEvent, pOldCurrentEvent);
}

//_________________________________________________________________________________________________

void EventViewerWidget::onChangedActiveEventItem(QTreeWidgetItem* current, QTreeWidgetItem* previous)
{
	if (previous == current)
		return; //nothing really changed

	//previously there might not have been a selected event so we need to check
	const DesEvent* pOldCurrentEvent = null;
	if (previous)
	{
		const DesEvent& prevCrtEvent = getEventFromUiItem(*previous);
		pOldCurrentEvent = &prevCrtEvent;
	}

	const DesEvent* pNewCurrentEvent = null;
	if (current)
	{
		//obtain the currently selected event
		const DesEvent& crtEvent  = getEventFromUiItem(*current);
		pNewCurrentEvent = &crtEvent;
	}

	emit onChangedActiveEvent(pNewCurrentEvent, pOldCurrentEvent);
}

//_________________________________________________________________________________________________

void EventViewerWidget::onEventItemClicked ( QTreeWidgetItem * item, int /*column*/ )
{
	//check if the user selected "nothing" and deselect the curent item
	if (item == null)
	{
		setItemSelected(currentItem(), false);
		setCurrentItem(null);
	}
	else
	{
		if (isEventUiItem(*item))
		{
			const DesEvent& event = getEventFromUiItem(*item);
			emit onEventClicked(event);
		}
	}
}

//_________________________________________________________________________________________________
//This signal is received whenever the user changes the item in place (within the tree widget)
//The method ensures the data is validated and the corresponding event is changed.   Once the event
//is changed in the DES, the DES itself will send a changed event notfication that this UI part 
//subscribes too
void EventViewerWidget::onEventItemChanged ( QTreeWidgetItem * item, int column )
{
	if (isEventUiItem(*item) == false)
		return; //this is not an event item. NOTE: this event seems to be sent in Linux when a new event is added to its parent

	//Change the event that corresponds to this item
	switch(column)
	{
		case cEventNameColIdx:
			onEventItemNameChanged(item, column);
			break;

		case cEventAliasColIdx:
			onEventItemAliasChanged(item, column);
			break;

		case cEventCtrlColIdx:
			onEventItemCtrlChanged(item, column);
			break;

		case cEventTypeColIdx:
			onEventItemTypeChanged(item, column);
			break;

		default:
			assert(false);
	}
}

//_________________________________________________________________________________________________

void EventViewerWidget::onEventItemNameChanged(QTreeWidgetItem * item, int /*column*/ )
{
	if (item == null)
		return;

	const DesEvent& changedEvent  = getEventFromUiItem(*item);

	QString nameInItem = item->text(cEventNameColIdx);
	if (nameInItem.size() == 0)
	{
		//the user has erased the name making the event illegal so let the user know
		//and put back the old name
		QMessageBox::critical(parentWidget(), STR_DESPOT_ERROR, STR_EVENT_NAME_MANDATORY);
		item->setText(cEventNameColIdx, QString::fromStdWString(changedEvent.getName()));
		return;
	}
	
	std::wstring newName = item->text(cEventNameColIdx).toStdWString();
	if (newName != changedEvent.getName())
	{
		try
		{
			emit onUserChangedEventName(changedEvent, newName);
		}
		catch(EXO)
		{
			display_ex_in(parentWidget());
			item->setText(cEventNameColIdx, QString::fromStdWString(changedEvent.getName()));
		}
	}
}

//_________________________________________________________________________________________________

void EventViewerWidget::onEventItemAliasChanged(QTreeWidgetItem * item, int /*column*/ )
{
	if (item == null)
		return;

	const DesEvent& changedEvent  = getEventFromUiItem(*item);
	QString aliasInItem = item->text(cEventAliasColIdx);
	std::wstring newAlias = aliasInItem.size() > 0 ? aliasInItem.toStdWString() : L"";
	
	if (newAlias != changedEvent.getAlias())
	{
		try
		{
			emit onUserChangedEventAlias(changedEvent, newAlias);
		}
		catch(EXO)
		{
			display_ex_in(parentWidget());
			item->setText(cEventAliasColIdx, QString::fromStdWString(changedEvent.getAlias()));
		}
	}

}

//_________________________________________________________________________________________________

void EventViewerWidget::onEventItemCtrlChanged( QTreeWidgetItem * item, int /*column*/ )
{
	if (item == null)
		return;

	const DesEvent& changedEvent  = getEventFromUiItem(*item);
	QString ctrlLabel = item->text(cEventCtrlColIdx);
	if ((ctrlLabel.size() > 0) && (ctrlLabel != cPropCheckSymbol))
	{
		item->setText(cEventCtrlColIdx, cPropCheckSymbol);
		return; // do not change the DES event because a new notification will come due to the text correction
	}

	if ((ctrlLabel.size() > 0) !=  (changedEvent.isControllable() == true))
	{
		emit onUserChangedEventCtrl(changedEvent, (ctrlLabel.size() > 0));
	}
}

//_________________________________________________________________________________________________

void EventViewerWidget::onEventItemTypeChanged( QTreeWidgetItem * item, int /*column*/ )
{
	if (item == null)
		return;

	const DesEvent& changedEvent  = getEventFromUiItem(*item);

	QString newTypeLabel = item->text(cEventTypeColIdx);
	QString oldTypeLabel = getTypeLabel(changedEvent.getType());
		
	QString typeLabelCorrected;
	switch (isEventTypeLabelValid(newTypeLabel, &typeLabelCorrected))
	{
		case eInvalidTypeLabel:
			QMessageBox::critical(parentWidget(), STR_DESPOT_ERROR, EVENT_TYPE_INVALID_LABEL);
			item->setText(cEventTypeColIdx, oldTypeLabel); //set back the old label
			break;

		case eTypeLabelCorrected:
			//The label is valid but it needs correction
			item->setText(cEventTypeColIdx, typeLabelCorrected);
			break; // do not change the DES event because a new notification will come due to the text correction

		case eValidTypeLabel:
			if (newTypeLabel != oldTypeLabel)
			{
				emit onUserChangedEventType(changedEvent, getTypeForLabel(newTypeLabel));
			}
			break;
		
		default:
			assert(false);

	}				
}

//_________________________________________________________________________________________________

void EventViewerWidget::resizeEvent(QResizeEvent* event)
{
	DespotTreeWidget::resizeEvent(event);

	resizeHeaders(event->size().width());
}

//_________________________________________________________________________________________________

void EventViewerWidget::resizeHeaders(int eventViewWidgWidth)
{
	int headerWidth = eventViewWidgWidth;
	
	int nameWidth = headerWidth / 3;	 //(40%)
	int nameWidthOpt = header()->sectionSizeHint(0);
	
	int aliasWidth = headerWidth / 4;  //25%
	int aliasWidthOpt = header()->sectionSizeHint(1);
	
	int ctrlWidth       = (headerWidth - (nameWidth + aliasWidth)) / 2;
	int ctrlWidthOpt = header()->sectionSizeHint(2);
	
	int typeWidth = (headerWidth - (nameWidth + aliasWidth + ctrlWidth));
	int typeWidthOpt = header()->sectionSizeHint(3);
	
	header()->resizeSection(cEventNameColIdx, nameWidth < nameWidthOpt ? nameWidthOpt : nameWidth);
	header()->resizeSection(cEventAliasColIdx, aliasWidth < aliasWidthOpt ? aliasWidthOpt : aliasWidth);
	header()->resizeSection(cEventCtrlColIdx, ctrlWidth < ctrlWidthOpt ? ctrlWidthOpt : ctrlWidth);		
	header()->resizeSection(cEventTypeColIdx, typeWidth < typeWidthOpt ? typeWidthOpt : typeWidth);
}

//_________________________________________________________________________________________________

QTreeWidgetItem* EventViewerWidget::createUiItemFromEvent(const DesEvent& event)
{
	//add the event information to the event editor
	QTreeWidgetItem* pEventUiItem = new QTreeWidgetItem();

	//save the event pointer in the item itsel for easy retrieval.
	//this pointer cannot be used to modify the events but only to read things from it
	pEventUiItem->setData(0, Qt::UserRole, QVariant(reinterpret_cast<unsigned long long>(&event)));

	//set general UI item not related to the event itself
	pEventUiItem->setTextAlignment(cEventCtrlColIdx, Qt::AlignHCenter);
	pEventUiItem->setTextAlignment(cEventTypeColIdx, Qt::AlignHCenter);

	//make the item editable
	if (m_allowEdit)
	{
		pEventUiItem->setFlags(pEventUiItem->flags() | Qt::ItemIsEditable);
	}

	//fill the UI item with the event attributes (name, alias, etc...)
	fillEventUiItem(*pEventUiItem, event);

	return pEventUiItem;
}

//_________________________________________________________________________________________________

void EventViewerWidget::fillEventUiItem(QTreeWidgetItem& eventUiItem, const DesEvent& desEvent)
{
	QString eventName = QString::fromStdWString(desEvent.getName());
	if (eventName != eventUiItem.text(cEventNameColIdx))
	{
		eventUiItem.setText(cEventNameColIdx, eventName);
	}
	
	QString eventAlias = QString::fromStdWString(desEvent.getAlias());
	if (eventAlias != eventUiItem.text(cEventAliasColIdx))
	{
		eventUiItem.setText(cEventAliasColIdx, eventAlias);
	}
	
	if ((eventUiItem.text(cEventCtrlColIdx).size() > 0) != desEvent.isControllable())
	{
		eventUiItem.setText(cEventCtrlColIdx, desEvent.isControllable() ? cPropCheckSymbol : "");
	}

	if ((eventUiItem.text(cEventTypeColIdx).size() == 0) || getTypeForLabel(eventUiItem.text(cEventTypeColIdx)) != desEvent.getType())
	{
		eventUiItem.setText(cEventTypeColIdx, getTypeLabel(desEvent.getType()));		
	}
	
	//make the whole row bold if the event is marked or unbold if it is not marked anymore
	if (desEvent.isControllable() != (eventUiItem.textColor(0) == Qt::blue))
	{
		for(int i = 0; i < getColumnCount(); i++)
		{
			eventUiItem.setTextColor(i, desEvent.isControllable() ? Qt::blue : Qt::black);
		}
	}
	
	//set the tooltip for all columns if it changed
	QString eventToolTip = composeTooltip(eventUiItem);

	//check if the computed tooltip is different then the one actually present in the UI item
	if (eventToolTip != eventUiItem.toolTip(0))
	{
		//current item has a different tooltip => update it
		for(int i = 0; i < getColumnCount(); i++)
		{
			eventUiItem.setToolTip(i, eventToolTip);
		}
	}

}

//_________________________________________________________________________________________________

QString EventViewerWidget::composeTooltip(QTreeWidgetItem& eventUiItem)
{
	QString ctrlLabel   = (eventUiItem.text(cEventCtrlColIdx).size() > 0) ? cYes : cNo;
	QString typeName   = getTypeName(eventUiItem.text(cEventTypeColIdx));
	QString eventToolTip = eventUiItem.text(cEventNameColIdx);
	if (eventUiItem.text(cEventAliasColIdx).size() > 0)
	{
		eventToolTip += " (" + eventUiItem.text(cEventAliasColIdx) + ")";
	}

	const DesEvent& desEvent = getEventFromUiItem(eventUiItem);
	eventToolTip += tr("\n\n") + cEventCtrlProp + ":\t" + ctrlLabel + "\n" +
					cEventTypeProp + ":\t" + typeName + "\n" + 
					cEventUsedProp + ":\t" + (desEvent.isUsed() ? cYes : cNo) ;

	return eventToolTip;
}

//_________________________________________________________________________________________________

bool EventViewerWidget::isEventUiItem(QTreeWidgetItem& item)
{
	DesEvent* pEvent = reinterpret_cast<DesEvent*>(item.data(0, Qt::UserRole).toULongLong());
	return (pEvent != null);
}

//_________________________________________________________________________________________________

const DesEvent& EventViewerWidget::getEventFromUiItem(QTreeWidgetItem& eventUiItem)
{
	DesEvent* pEvent = reinterpret_cast<DesEvent*>(eventUiItem.data(0, Qt::UserRole).toULongLong());
	assert(pEvent != null);
	return *pEvent;
}

//_________________________________________________________________________________________________

QTreeWidgetItem* EventViewerWidget::getUiItemFromEvent(const DesEvent& event)
{
	for (int iRow = 0; iRow < topLevelItemCount(); iRow++)
	{
		QTreeWidgetItem* pEventUiItem = topLevelItem(iRow);
		assert(pEventUiItem != null);

		const DesEvent& crtEvent  = getEventFromUiItem(*pEventUiItem);
		if (&crtEvent == &event)
		{
			return pEventUiItem;
		}
	}

	//if we got here this event was not displayed. This should not happen
	throw EX("Event not found in the event view widget")
}

//_________________________________________________________________________________________________

QTreeWidgetItem* EventViewerWidget::getUiItemFromEvent(const DesEvent::ID& eventID)
{
	for (int iRow = 0; iRow < topLevelItemCount(); iRow++)
	{
		QTreeWidgetItem* pEventUiItem = topLevelItem(iRow);
		assert(pEventUiItem != null);

		const DesEvent& crtEvent  = getEventFromUiItem(*pEventUiItem);
		if (crtEvent.getId() == eventID)
		{
			return pEventUiItem;
		}
	}

	//if we got here this event was not displayed. This should not happen
	throw EX("Event not found in the event view widget")
}

//_________________________________________________________________________________________________

QString EventViewerWidget::getTypeName(const QString& typeLabel)
{
	if (typeLabel == cDefaultTypeLabel)
	{
		return cDefaultTypeName;
	}
	else if (typeLabel == cHighLevelTypeLabel)
	{
		return cHighLevelTypeName;
	}
	else if (typeLabel == cLowLevelTypeLabel)
	{
		return cLowLevelTypeName;
	}
	else if (typeLabel == cRequestTypeLabel)
	{
		return cRequestTypeName;
	}
	else if (typeLabel == cAnswerTypeLabel)
	{
		return cAnswerTypeName;
	}
	else if (typeLabel == cLDataTypeLabel)
	{
		return cLDataTypeName;
	}
	else
	{
		assert(false);
		throw EX("Unknown event type. Cannot return name.")
	}
}

//_________________________________________________________________________________________________

EventViewerWidget::ETypeLabelCheck EventViewerWidget::isEventTypeLabelValid(const QString& typeLabel, QString* pTypeLabelCorrected/* = null*/)
{
	if (typeLabel == cDefaultTypeLabel)
	{
		return eValidTypeLabel;
	}
	else if (typeLabel == cDefaultTypeAltLabel)
	{
		if (pTypeLabelCorrected)
		{
			*pTypeLabelCorrected = cDefaultTypeLabel;
		}
		
		return eTypeLabelCorrected;
	}
	else if (typeLabel == cHighLevelTypeLabel)
	{
		return eValidTypeLabel;
	}
	else if (typeLabel == cHighLevelTypeAltLabel)
	{
		if (pTypeLabelCorrected)
		{
			*pTypeLabelCorrected = cHighLevelTypeLabel;
		}
		
		return eTypeLabelCorrected;
	}
	else if (typeLabel == cLowLevelTypeLabel)
	{
		return eValidTypeLabel;
	}
	else if (typeLabel == cLowLevelTypeAltLabel)
	{
		if (pTypeLabelCorrected)
		{
			*pTypeLabelCorrected = cLowLevelTypeLabel;
		}
		
		return eTypeLabelCorrected;
	}
	else if (typeLabel == cRequestTypeLabel)
	{
		return eValidTypeLabel;
	}
	else if (typeLabel == cRequestTypeAltLabel)
	{
		if (pTypeLabelCorrected)
		{
			*pTypeLabelCorrected = cRequestTypeLabel;
		}
		
		return eTypeLabelCorrected;
	}
	else if (typeLabel == cAnswerTypeLabel)
	{
		return eValidTypeLabel;
	}
	else if (typeLabel == cAnswerTypeAltLabel)
	{
		if (pTypeLabelCorrected)
		{
			*pTypeLabelCorrected = cAnswerTypeLabel;
		}
		
		return eTypeLabelCorrected;
	}
	else if (typeLabel == cLDataTypeLabel)
	{
		return eValidTypeLabel;
	}
	else if (typeLabel == cLDataTypeAltLabel)
	{
		if (pTypeLabelCorrected)
		{
			*pTypeLabelCorrected = cLDataTypeLabel;
		}
		
		return eTypeLabelCorrected;
	}
	else
	{
		return eInvalidTypeLabel;
	}
}

//_________________________________________________________________________________________________

QString EventViewerWidget::getTypeLabel(EventType eventType)
{
	switch(eventType)
	{
		case eDefaultEvent:
			return cDefaultTypeLabel;

		case eHighLevelEvent:
			return cHighLevelTypeLabel;

		case eLowLevelEvent:
			return cLowLevelTypeLabel;

		case eRequestEvent:
			return cRequestTypeLabel;

		case eAnswerEvent:
			return cAnswerTypeLabel;

		case eLDataEvent:
			return cLDataTypeLabel;

		default:
			assert(false);
			throw EX("Unknown event type. Cannot return label.")
	}
}

//_________________________________________________________________________________________________

EventType EventViewerWidget::getTypeForLabel(const QString& typeLabel)
{
	if ((typeLabel == cDefaultTypeLabel) || (typeLabel == cDefaultTypeAltLabel))
	{
		return eDefaultEvent;
	}
	if ((typeLabel == cHighLevelTypeLabel) || (typeLabel == cHighLevelTypeAltLabel))
	{
		return eHighLevelEvent;
	}
	if ((typeLabel == cLowLevelTypeLabel) || (typeLabel == cLowLevelTypeAltLabel))
	{
		return eLowLevelEvent;
	}
	else if ((typeLabel == cRequestTypeLabel) || (typeLabel == cRequestTypeAltLabel))
	{
		return eRequestEvent;
	}
	else if ((typeLabel == cAnswerTypeLabel) || (typeLabel == cAnswerTypeAltLabel))
	{
		return eAnswerEvent;
	}
	else if ((typeLabel == cLDataTypeLabel) || (typeLabel == cLDataTypeAltLabel))
	{
		return eLDataEvent;
	}
	else
	{
		assert(false);
		throw  "Invalid event type label. Cannot return event type.";
	}
}

} //end of namespace DESpot
