/* trap.c -- Parallel Trapezoidal Rule
 *
 * Input: None.
 * Output:  Estimate of the integral from a to b of f(x)
 *    using the trapezoidal rule and n trapezoids.
 *
 * Algorithm:
 *    1.  Each process calculates "its" interval of
 *        integration.
 *    2.  Each process estimates the integral of f(x)
 *        over its interval using the trapezoidal rule.
 *    3a. Each process != 0 sends its integral to 0.
 *    3b. Process 0 sums the calculations received from
 *        the individual processes and prints the result.
 *
 * The number of processes (p) should evenly divide the 
 * number of trapezoids (n = 1024)
 */


#include <stdio.h>
#include "mpi.h"

extern double Trap(double a, double b, int n, double h);

int main(int argc, char** argv) 
{
  int       my_rank;   /* My process rank           */
  int       p;         /* The number of processes   */
  double    a = 0.0;   /* Left endpoint             */
  double    b = 1.0;   /* Right endpoint            */
  int       n = 1024;  /* Number of trapezoids      */
  double    h;         /* Trapezoid base length     */
  double    local_a;   /* Left endpoint my process  */
  double    local_b;   /* Right endpoint my process */
  int       local_n;   /* Number of trapezoids for  */
                       /* my calculation            */
  double    integral;  /* Integral over my interval */
  double    total=-1;  /* Total integral            */
  int       source;    /* Process sending integral  */
  int       dest = 0;  /* All messages go to 0      */
  int       tag = 0;
  MPI_Status  status;
  
  MPI_Init(&argc, &argv);
  
  MPI_Comm_rank(MPI_COMM_WORLD, &my_rank);
  MPI_Comm_size(MPI_COMM_WORLD, &p);
  
  h = (b-a)/n;    /* h is the same for all processes */
  local_n = n/p;  /* So is the number of trapezoids */
  
  /* Length of each process' interval of 
     integration = local_n*h. */
  local_a = a + my_rank*local_n*h;
  local_b = local_a + local_n*h;
  integral = Trap(local_a, local_b, local_n, h);
  
  if (my_rank == 0) 
    {
      total = integral;
      for (source = 1; source < p; source++) 
	{
	  /* Receive values from each process. */
	  MPI_Recv(&integral,1,MPI_DOUBLE, source, tag,MPI_COMM_WORLD, &status);
	  total = total + integral;
	}
    } 
  else /* Send from my_rank to 0. */
    MPI_Send(&integral, 1, MPI_DOUBLE, dest, tag, MPI_COMM_WORLD);
  
  /* Print the result */
  if (my_rank == 0) 
    printf("Value of the integral from %f to %f = %f\n", a, b, total);
  
  MPI_Finalize();
  
  return 0;
} 


