/* trap.c -- Parallel Trapezoidal Rule
 *
 * Input: None.
 * Output:  Estimate of the integral from a to b of f(x)
 *    using the trapezoidal rule and n trapezoids.
 *
 * Algorithm:
 *    1.  Each process calculates "its" interval of
 *        integration.
 *    2.  Each process estimates the integral of f(x)
 *        over its interval using the trapezoidal rule.
 *    3a. Each process != 0 sends its integral to 0.
 *    3b. Process 0 sums the calculations received from
 *        the individual processes and prints the result.
 *
 *        The number of processes (p) should evenly divide
 *        the number of trapezoids (n = 1024)
 */


#include <stdio.h>
#include "mpi.h"

extern float Trap(float a, float b, int n, float h);



int main(int argc, char** argv) 
{
  int         my_rank;   /* My process rank           */
  int         p;         /* The number of processes   */
  float       a = 0.0;   /* Left endpoint             */
  float       b = 1.0;   /* Right endpoint            */
  int         n = 1024;  /* Number of trapezoids      */
  float       h;         /* Trapezoid base length     */
  float       local_a;   /* Left endpoint my process  */
  float       local_b;   /* Right endpoint my process */
  int         local_n;   /* Number of trapezoids for  */
                         /* my calculation            */
  float       integral;  /* Integral over my interval */
  float       total=-1;  /* Total integral            */
  int         source;    /* Process sending integral  */
  int         dest = 0;  /* All messages go to 0      */
  int         tag = 0;
  MPI_Status  status;
  
  MPI_Init(&argc, &argv);

  MPI_Comm_rank(MPI_COMM_WORLD, &my_rank);
  MPI_Comm_size(MPI_COMM_WORLD, &p);
  
  h = (b-a)/n;    /* h is the same for all processes */
  local_n = n/p;  /* So is the number of trapezoids */
  
  /* Length of each process' interval of 
     integration = local_n*h. */
  local_a = a + my_rank*local_n*h;
  local_b = local_a + local_n*h;
  integral = Trap(local_a, local_b, local_n, h);
  
  /* Add up the integrals calculated by each process */
  if (my_rank == 0) 
    {
      total = integral;
      for (source = 1; source < p; source++) 
	{
	  MPI_Recv(&integral, 1, MPI_FLOAT, source, tag,
		   MPI_COMM_WORLD, &status);
	  printf("PE %d <- %d,   %f\n", my_rank,source,
		 integral);
	  total = total + integral;
	}
    } 
  else 
    {
      printf("PE %d -> %d,   %f\n", my_rank, dest, integral);
      MPI_Send(&integral, 1, MPI_FLOAT, dest,
	       tag, MPI_COMM_WORLD);
    }


  

  /* Print the result */
  if (my_rank == 0) 
    {
      printf("With n = %d trapezoids, our estimate\n",
	      n);
      printf("of the integral from %f to %f = %f\n",
	     a, b, total);
    }
  
  MPI_Finalize();

  return 0;
} 


