
// FILE initencl.h

#ifndef INCLUDED_INITENCL_H
#define INCLUDED_INITENCL_H

#include "odesolv.h"


/** Abstract class for validating existence and uniqueness of the
    solution and computing a priori bounds on the solution.
    
    @author Ned Nedialkov 
    @date 28 April 2001
*/
class INIT_ENCL : protected NAME
{ 
  
public: 
  
  /** Validates existence and uniqueness and computes a priori bounds.
      
      This function tries to validate existence and uniqueness of the
      solution.  If successful, it returns an enclosure of the
      solution, an enclosure of the local error, and a stepsize.

      @param Y (output) an enclosure of the solution
      @param LocErr (output) an enclosure of the local error
      @param h (input/output) stepsize 
      @param order order of the method
      @param ODE  pointer to a problem
      @param Solver pointer to a solver
      
      @pre h != 0 && order>0 && Order<=GetMaxOrder() && NotNull(ODE) && NotNull(Solver)
      
      @return \c true if existence and uniqueness have been
      established, and \c false otherwise.  */
  virtual bool Validate( INTERVAL_VECTOR & Y,
			 INTERVAL_VECTOR & LocErr, 
			 double & h, 
			 int order,
			 PtrODENumeric ODE, 
			 PtrODESolver Solver ) = 0; 

  /** Initializes the method.
      
      @param ODE pointer to an object representing the numerical problem
      @pre NotNull(ODE)
  */
  virtual void Init( PtrODENumeric ODE ) = 0;
  
  /// Returns the maximum order allowed.
  int GetMaxOrder() const;  
  
  /// Destructor.
  virtual ~INIT_ENCL() {;} 
  
  
protected:

  /** Constructor. 
      @param max_order maximum order allowed
      @param name name of the method
      
      @pre max_order>0 && !name.empty()
  */
  INIT_ENCL( int max_order, const string & name );
  
  
private: 
  
  /// Maximum order allowed.
  const int MaxK;

};


/// Pointer to INIT_ENCL object
typedef INIT_ENCL* PtrInitEncl;


inline INIT_ENCL :: INIT_ENCL( int max_order, const string & name ) 
  : NAME(name), MaxK(max_order)
{
  assert(max_order>0);
}


inline int INIT_ENCL :: GetMaxOrder() const
{
  return MaxK;
}


#endif
