
// FILE itsbase.h 


#ifndef INCLUDED_ITSBASE_H
#define INCLUDED_ITSBASE_H

#include "paral.h"
#include "tgtencl.h"


/** Base class for implementing a Taylor series method for computing
  tight enclosures on the solution.
  
  This class encapsulates the common part of the direct and QR-factorization methods.
  
  @author Ned Nedialkov 
  @date   19 July 2001 */
class ITS_BASE : public TIGHT_ENCL
{	
  
public:									
  
  /** Computes a tight enclosure. */
  virtual void CompEncl( INTERVAL_VECTOR & Ytight, 
			 INTERVAL_VECTOR & LocErr,
			 const INTERVAL & h, 
			 int order,
			 PtrODENumeric ODE, 
			 PtrODESolver Solver );
  
  /** Initializes the method.
   */
  virtual void Init( const PtrODENumeric ODE );  
  
  ///  Destructor 
  ~ITS_BASE();
  
protected:
  
  /** Constructor.  
      
      @param order maximum order allowed.       
      @param name name of the method
    
      @pre order>0 && order <= TAYLOR_CTRL::MaxOrder && !name.empty() 
      */
  ITS_BASE( int order, const string & name );  
  
  /** Prepares to compute a tight enclosure. 

    This function encapsulates the common part of the direct and QR-factorization methods.

    @param LocErr  (output) enclosure of the local error
    @param h       enclosure of the stepsize
    @param order   order of the method
    @param ODE     pointer to a problem 
    @param Solver  pointer to a solver
    
    @pre NonZero(h) && order>0 && order<=GetMaxOrder() && NotNull(ODE) && NotNull(Solver) && 
    ODE->GetSize()==Dimension(LocErr)
    */
  virtual void Prepare(  INTERVAL_VECTOR & LocErr,
			 const INTERVAL & h, 
			 int order,
			 PtrODENumeric ODE, 
			 PtrODESolver Solver );
    
  /** Computes a tight enclosure. This function is called after \ref Prepare.
    
    @param Ytight (output)  enclosure of the solution 
    @param LocErr (output)  enclosure of the local error 
    @param h  enclosure of the stepsize 
    @param order order of the enclosure method
    @param ODE pointer to  a problem
    @param Solver pointer to a solver 
    
    @pre NonZero(h) && order>0 && order<=GetMaxOrder() && NotNull(ODE) && NotNull(Solver) && 
    ODE->GetSize()==Dimension(LocErr) && ODE->GetSize==Dimension(Ytight);
    
    */
  virtual void Compute ( INTERVAL_VECTOR & Ytight, 
			 INTERVAL_VECTOR & LocErr,
			 const INTERVAL & h, 
			 int order,
			 PtrODENumeric ODE, 
			 PtrODESolver Solver ) = 0;
  

  /** Allocates space. 

    This function allocates the necessary matrices and vectors.
    @param n size of the problem
    @pre n>0
    */
  void ResizeSpace ( int n );    
  
  /** Pointer to a parallelepiped. It is used to store the solution
    in the form of a parallelepiped.  */
  PtrParall  PrevSol;
  
  /// Enclosure of the point solution.
  INTERVAL_VECTOR  YpointEncl;   
  
  /// Solution to the variational equation.
  INTERVAL_MATRIX  Sj;


};


inline void ITS_BASE :: CompEncl( INTERVAL_VECTOR & Ytight, 
				  INTERVAL_VECTOR & LocErr,
				  const INTERVAL & h, 
				  int order,
				  PtrODENumeric ODE, 
				  PtrODESolver Solver )
{
  Prepare(  LocErr,  h, order, ODE, Solver );
  Compute(  Ytight,  LocErr,  h, order, ODE, Solver );
}


inline ITS_BASE:: ITS_BASE( int order, const string & name  )
  
  : TIGHT_ENCL( order, name ), PrevSol( NULL )
{
  ;
}


inline ITS_BASE :: ~ITS_BASE() 
{ 
  DELETE(PrevSol); 
}


#endif 
