
// FILE itsqr.h 


#ifndef INCLUDED_ITSQR_H
#define INCLUDED_ITSQR_H


#include "itsbase.h"
#include "taylctrl.h"
#include "messgs.h"


/** Pointer to a function for computing the Q factor from the QR
  factorization of a point matrix A.  */
typedef void (*QRFactor)( MATRIX & Q, const MATRIX & A );


/** QR-factorization method for computing tight enclosures on the solution.
  
  The ITS_QR class implements the QR factorization method.
  
  
  @author Ned Nedialkov 
  @date 1 May 2001 
  */
class ITS_QR : public ITS_BASE
{	
  
public:									
  
  /** Constructor.  
    
    @param order maximum order allowed. The default value is 
    TAYLOR_CTRL::GetMaxOrder()
    
    @pre order>0 && order<=TAYLOR_CTRL::GetMaxOrder()
    */
  ITS_QR( int order = TAYLOR_CTRL::GetMaxOrder() );
  
  
  /** Initializes the method.
   */
  void Init( const PtrODENumeric ODE );  
  
  
  /** Computes a tight enclosure.
   */
  virtual void Compute( INTERVAL_VECTOR & Ytight, 
			INTERVAL_VECTOR & LocErr,
			const INTERVAL & h, int order,
			PtrODENumeric ODE, 
			PtrODESolver Solver );
  
  
  /** Sets a function for computing the Q factor from a QR
    factorization of a matrix.
    
    @param f pointer to a function 
    @pre NotNull(f) */
  void SetQRFunction( QRFactor f ); 
  
  
  ///  Destructor 
  ~ITS_QR();
  

private:
  
  /** A pointer to a function for computing the Q factor from the QR
      factorization of a matrix.  */
  QRFactor QRFactorFunction;
  
  /// Enclosure of the inverse of the transformation matrix.
  INTERVAL_MATRIX  AjInv;  
  
  /// Temporary space.
  MATRIX TmpMat1;
  
  /** Allocate space.
    @param n size of the problem
    @pre n > 0
    */
  void ResizeSpace( int n );
  
};



inline ITS_QR:: ITS_QR( int order ) : 
  ITS_BASE( order, MSG_ITS_QR_METHOD ) 
{ 
  assert( order>0 && order<=TAYLOR_CTRL::GetMaxOrder() );
}


inline void ITS_QR :: SetQRFunction( QRFactor f )
{
  assert( f!=NULL );
  QRFactorFunction = f;
}


inline ITS_QR :: ~ITS_QR() 
{ 
  ;
}


#endif 
