
// FILE matinv.cc 


#include "matinv.h"
#include "vndinl.h"
#include "vnderror.h"


INTERVAL_VECTOR LSS2 (const MATRIX & A, const VECTOR & b, 
		      const MATRIX & R, INT & info);

void Inverse( INTERVAL_MATRIX & A, const MATRIX & B )
  // This function computes an enclosure of the inverse of B and
  // stores the result in A.
{
  assert( RowDimension(A)==ColDimension(A) &&
	  RowDimension(B)==ColDimension(B) &&
	  RowDimension(A)==RowDimension(B) &&
	  RowDimension(A)>0 );
  
  
  INT N = RowDimension(A);
  int OK = 1;
  
  if ( N == 1 ) 
    if ( B(1,1) == 0 ) 
      OK = 0;
    else
      A(1,1) = DivBounds(1.0,B(1,1));
  else
    {
      VECTOR UnitVector(N);
      INTERVAL_VECTOR Encl(N);
      
      MATRIX R = Inverse(B);
      
      Clear(UnitVector);
      
      for ( int j = 1; j <= N; j++)
	{
	  UnitVector(j) = 1.0; 
	  Encl = LSS2( B, UnitVector, R, OK ); 
	  
	  if ( !OK )
	    break;
	  
	  SetCol( A,j, Encl );
	  UnitVector(j) = 0.0;
	}
    }
  
  if ( !OK )
    VnodeError("Cannot enclose the inverse of a matrix");
}


void OrthogonalInverse( INTERVAL_MATRIX & A, const MATRIX & B )
  // B is a floating-point approximation to an orthogonal matrix.
{
  assert( RowDimension(A)==ColDimension(A) &&
	  RowDimension(B)==ColDimension(B) &&
	  RowDimension(A)==RowDimension(B) &&
	  RowDimension(A)>0 );
  
  INT N = RowDimension(A);
  int OK = 1;
  
  if ( N == 1 ) 
      A(1,1) = DivBounds(1.0,B(1,1));
  else
    {
      VECTOR UnitVector(N);
      INTERVAL_VECTOR Encl(N);
      
      MATRIX R = Transpose(B);
      
      Clear(UnitVector);
      
      for ( int j = 1; j <= N; j++)
	{
	  UnitVector(j) = 1.0; 
	  Encl = LSS2( B, UnitVector, R, OK ); 
	  
	  if ( !OK )
	    break;
	  
	  SetCol( A,j, Encl );
	  UnitVector(j) = 0.0;
	}
    }
  assert(OK);
}


INTERVAL_VECTOR LSS2 (const MATRIX & A, 
		      const VECTOR & b, const MATRIX & R, INT & info)
{
  // This code is borrowed from Profil.
  INT dim = Dimension (b);
  VECTOR xs (dim);
  INTERVAL_VECTOR x (dim), y (dim), z (dim);
  INTERVAL_VECTOR Inflat (dim);
  INTERVAL_MATRIX C(dim,dim);
  INT k, done;
  INTERVAL eps (0.9,1.1);
  
  Initialize(Inflat, SymHull (Machine::MinPositive));
  
  xs = R * b;
  
  z = R * (b - MulBounds (A, xs));
  
  x = z;
  
  C = Id(dim) - MulBounds (R, A);
  
  k = 0;
  do {
    y = eps * x + Inflat; // expansion
    x = z + C * y;
    done = (x < y);
    k++;
  } while (!done && k < 10); 
  
  info = done;              // 1, if inclusion was possible
  return (xs + x);
}

