
// FILE odenum.h


#ifndef  INCLUDED_ODENUM_H
#define  INCLUDED_ODENUM_H


#include "odeprobl.h"

#define DEFAULT_ABS_TOL 1e-12
#define DEFAULT_REL_TOL 1e-12


/** Specification of the numerical IVP.
  
  This class inherits from ODE_PROBLEM the specification of the IVP
  problem and provides the end integration point, absolute and
  relative error tolerances, and an object for storing the solution.
  
  @author Ned Nedialkov 
  @date 28 April 2001 */
class ODE_NUMERIC : public ODE_PROBLEM
{

public:
  
  /** Constructor. 

      It sets the size of the problem, the function for computing the
      right side, a name for the problem, and default values for the
      absolute and relative error tolerances.
    
      @param n size of the problem
      @param fcn pointer to a function for computing the right side
      @param name name for the ODE problem
      
      @pre n>0 && NotNull(f) && !name.empty() */
  ODE_NUMERIC( int n, FCN fcn, const string & name );

  /** Sets the end point.

    @pre tEnd != GetT0() || NotNull(Solution) && tEnd != Solution->GetTcur() 
    */
  void SetTend( double tEnd );
  
  /** Sets absolute error tolerance.
    
    @pre abstol >= 0
    */
  void SetAtol( double abstol );
  
  /** Sets relative error tolerance.
    
    @pre reltol >= 0
    */
  void SetRtol( double reltol );
  
  /** Sets the current integration point.

      @pre GetTcur() <= t && t <= GetTend() || GetTend() <= t && t <= GetTcur()

   */
  void SetTcur( double t );  
  
  /** Sets an priori enclosure.
    
    @pre GetSize() == Dimension(Y)
    */
  void SetInitEncl( const INTERVAL_VECTOR & Y );
  
  /** Sets a tight enclosure.

    @pre GetSize() == Dimension(Y)
    */
  void SetTightEncl( const INTERVAL_VECTOR & Y );
  
  /** Sets a solution object.
    
    @param Sol pointer to a solution object
    @pre NotNull(Sol ) && GetSize() == Sol->GetDim()
    */
  void SetSolution( const PtrSolution Sol );
  
  /// Returns the end point.
  double GetTend() const; 
  
  /// Returns the absolute tolerance.
  double GetAtol() const;
  
  /// Returns the relative tolerance.
  double GetRtol() const; 
  
  /// Returns the current integration point.
  double  GetTcur() const;

  /// Returns the previous integration point.
  double  GetTprev() const;
  
  /// Returns the initial enclosure.
  const INTERVAL_VECTOR & GetInitEncl() const;
  
  /// Returns the tight enclosure.
  const INTERVAL_VECTOR & GetTightEncl() const;
  
  /// Returns a pointer to the solution object.
  const PtrSolution GetSolution() const ; 
  
  /** Set problem parameters.  Depending on the value of the
    parameter, different sets of parameters, e.g. initial condition,
    integration interval, and tolerances, can be set.
    
    @param ParamSet encodes the parameter set.  */
  virtual void  LoadProblemParam( int ParamSet );
  
  /** Plot a solution component.  This function plots a solution
      component versus the time and displays the plot on the screen
      (default).  If a file name is specified, the plot is saved in an
      encapsulated postscript file under this file name.
    
    @param comp   component to be plotted
    
    @param file  file name 
    
    @pre comp <= GetSize() && comp >= 1 */
  void DisplaySolution( int comp, const char *file = 0 ) const ;
    
  /** Produce a phase plot.  This function plots \c comp2 versus \c
    comp1. By default, the result is displayed on the screen. If a
    file name is given, the plot is stored into an encapsulated
    postscript file under this file name.
    
    @param comp1 first component
    @param comp2 second component

    @param file file name 

    @pre comp1 <= GetSize() && comp1 >= 1 &&
    @pre comp2 <= GetSize() && comp2 >= 1 && comp1 != comp2 */
  void DisplayPhase( int comp1, int comp2, const char * file = 0 ) const ;
  
  /// Destructor
  virtual ~ODE_NUMERIC();
  
private:

  /// absolute tolerance
  double atol;

  /// relative tolerance
  double rtol;

  /// end point
  double Tend;

  /// pointer to a solution
  PtrSolution  Solution;  

};


/// Pointer to an ODE_NUMERIC object.
typedef ODE_NUMERIC* PtrODENumeric;

/** Template function for converting a solution pointer.

    This function takes a pointer to an ODE_NUMERIC object and returns
    a pointer to a solution object of type \c T.
    
    @pre NotNull(ODE) && NotNull(ODE->GetSolution())
*/
template <class T>  
T * GetSolution( PtrODENumeric ODE )
{
  assert( NotNull(ODE) );
  return static_cast<T*>(ODE->GetSolution());
}


/** Template function for initializing the solution of an ODE.
    
    This function copies the initial condition representation to the
    representation of the solution.
    
    @pre NotNull(ODE) && NotNull(ODE->GetSolution())
*/
template <class T>  
void InitSolution( PtrODENumeric ODE )
{
  assert( NotNull(ODE) );
  assert( NotNull(ODE->GetSolution()) );

  *GetSolution<T>(ODE) = *ODE->GetPtrInitCond();
}


inline ODE_NUMERIC :: ODE_NUMERIC( int n, FCN fcn, const string & name )
  : ODE_PROBLEM( n, fcn, name ),
  atol(DEFAULT_ABS_TOL), 
  rtol(DEFAULT_REL_TOL)
{
  ;
}  


inline void ODE_NUMERIC :: SetTend( double tEnd )   
{ 
  assert( tEnd != GetT0() ||
	  NotNull(Solution) && tEnd != Solution->GetTcur() );
  Tend = tEnd;
}


inline void ODE_NUMERIC :: SetAtol( double abstol )    
{ 
  assert(abstol>=0); 
  atol = abstol; 
}


inline void ODE_NUMERIC :: SetRtol( double reltol ) 
{ 
  assert(reltol>=0); 
  rtol = reltol; 
}


inline void ODE_NUMERIC :: SetTcur( double t ) 
{
  assert( NotNull(Solution) );
  assert( (GetTcur() <= t && t <= GetTend()) ||
  	  (GetTend() <= t && t <= GetTcur()) );
  
  Solution->SetTcur(t); 
}


inline void ODE_NUMERIC :: SetInitEncl( const INTERVAL_VECTOR & Y )
{
  assert( NotNull(Solution) );  
  assert( GetSize() == Dimension(Y) );
  Solution->SetInitEncl(Y); 
}


inline void ODE_NUMERIC :: SetTightEncl( const INTERVAL_VECTOR & Y )
{
  assert( NotNull(Solution) ); 
  assert( GetSize() == Dimension(Y) );
  Solution->SetTightEncl(Y); 
}   
  

inline void ODE_NUMERIC :: SetSolution( const PtrSolution Sol )
{
  assert( NotNull(Sol) && GetSize() == Sol->GetDim() );
  
  DELETE(Solution); 
  Solution = Sol; 
}


inline double ODE_NUMERIC :: GetTend() const 
{ 
  return Tend; 
}  


inline double ODE_NUMERIC :: GetRtol() const 
{ 
  return rtol;
}  


inline double ODE_NUMERIC :: GetAtol() const 
{ 
  return atol;
}  

inline double ODE_NUMERIC :: GetTcur() const
{
  assert( NotNull(Solution) );
  return Solution->GetTcur();
}


inline double ODE_NUMERIC :: GetTprev() const
{
  assert( NotNull(Solution) );
  return Solution->GetTprev();
}


inline const INTERVAL_VECTOR & ODE_NUMERIC :: GetTightEncl() const 
{
  assert( NotNull(Solution) );
  return Solution->GetTightEncl(); 
}


inline const INTERVAL_VECTOR & ODE_NUMERIC :: GetInitEncl() const 
{
  assert( NotNull(Solution) );
  return Solution->GetInitEncl(); 
}


inline const PtrSolution ODE_NUMERIC :: GetSolution() const 
{
  assert( NotNull(Solution) );
  return Solution; 
}


inline void ODE_NUMERIC :: LoadProblemParam( int )
{
  assert(0);
}


inline ODE_NUMERIC :: ~ODE_NUMERIC() 
{ 
  DELETE(Solution); 
}

#endif 
