
// FILE qr.cc 


#include <assert.h>
#include "qr.h"
#include "vndinl.h"
#include "vndutils.h"

// Compute the QR factorization of an M x N matrix A.  The function
// overrides A; Q is stored below and including the diagonal of A.
// The diagonal of R is stored in d and the rest of it is stored
// above the diagonal of A.

void QR( MATRIX & A, VECTOR & d )
{
  const int M = RowDimension(A);
  const int N = ColDimension(A);
  
  assert(Dimension(d)==M);
  
  int i,j,k,l;
  
  k = 0;
  double s,t,x,r;
  
  for ( l = 1; l <= N; l++ )
    {
      k++;
      
      if ( k == M )
	{
	  d(l) = A(k,l);
	  break;
	}
      s = 0;
      for ( i = k; i <= M; i++ )
	{
	  x = A(i,l);
	  s += x*x;
	} // for
      s = sqrt(s);
      
      if ( s == 0 )
	{
	  d(l) = 0;
	  continue;
	}
      
      t = A(k,l);
      r = 1/sqrt( s*(s+fabs(t)) );
      if ( t < 0 )
	s = -s;
      
      d(l) = -s;
      A(k,k) = r*(t+s);
      
      for ( i = k+1; i <= M; i++ )
	A(i,k) = A(i,l)*r;
      
      for ( j = l+1; j <= N; j++ )
	{
	  t = 0;
	  for ( i = k; i <= M; i++ )
	    t += A(i,k)*A(i,j);
	  
	  for ( i = k; i <= M; i++ )
	    A(i,j) -= t*A(i,k);
	}
    }
} // QR


void QRfactor( MATRIX & Q, const MATRIX & A )
{
  assert(ColDimension(A)==ColDimension(Q));
  assert(RowDimension(A)==RowDimension(Q));

  int n = ColDimension(A);

  if ( n == 1 )
    {
      Q(1,1) = 1.0;
      return;
    }
  Q = A;
  VECTOR d(n);
  QR( Q, d );
  
  VECTOR v(n);
  
  v(n-1) = Q(n-1,n-1);
  v(n)   = Q(n, n-1);
  
  // Q = H_{k-1}
  Q(n-1,n) = Q(n,n-1) = -v(n-1)*v(n);
  Q(n,n) = 1.0 - v(n)*v(n);
  Q(n-1,n-1) = 1.0 - v(n-1)*v(n-1);
  
  int k, row, col, i;
  double a;
  for ( k = n-2; k>=1; k-- )
    {
      // Extract the v_k vector.
      for ( row = k; row <= n; row++ )
	v(row) = Q(row,k);
      
      for ( col = k+1; col <= n; col++ )
	{
	  a = 0;
	  for ( i = k+1; i <= n; i++ )
	    a += v(i)*Q(i,col);
	  
	  for ( row = k+1; row <= n; row++ )
	    Q(row,col) -= a*v(row);
	  Q(k,col) = -a*v(k);
	}
      for ( i = k+1; i <= n; i++ )
	Q(i,k) = -v(k)*v(i);
      Q(k,k) = 1.0-v(k)*v(k);
      
    }

#ifdef DEBUG_LEVEL2

  MATRIX I(n,n);
  Identity(I);
  I = Transpose(Q)*Q - I;
  assert( MaxNorm(I) < 1e-10 );
  
#endif
}



