
// FILE taylode.h


#ifndef INCLUDED_TAYLODE_H
#define INCLUDED_TAYLODE_H 


#include "taylctrl.h"

/** Abstract class for generating Taylor coefficients. 

    This class provides the interface for a Taylor series generator
    for the solution to the IVP 
    \f$ y^\prime = f(y) \f$, 
    \f$ y(t_{j}) \in [y_{j}] \f$.
    
    @author Ned Nedialkov 
    @date   8 June 2001
 */
class TAYLOR_ODE : protected TAYLOR_CTRL
{
  
public:
  
  /** Generates Taylor terms. This function generates \c k Taylor
      terms for the solution of an ODE with an interval initial
      condition \c Y at a point of expansion enclosed by \c h. 
      
      @param Y interval vector enclosing the initial condition
      @param h  interval enclosing the point of expansion
      @param k  order of the Taylor series
      
      @pre k>0 && k<=GetMaxOrder() && Dimension(Y) == GetDim() &&NonZero(h)
  */
  virtual void GenerateTerms( const INTERVAL_VECTOR & Y, 
			      const INTERVAL & h, int k ) = 0;  
  
  /** Sums Taylor series. This function computes the sum of the first
      \c k Taylor terms that are generated in \ref GenerateTerms.

      @param Sum (output) the sum of the first \c k Taylor series
      terms

      @param k number of terms
      
      @pre Dimension(Sum) == GetDim() && k > 0 && k <= GetOrder() 
  */
  virtual void SumTerms( INTERVAL_VECTOR & Sum, int k ) = 0;
  
  /** Obtains the \c k-th Taylor series term. Obtain the \c k-th term
      from a Taylor series expansion.

      @param Yk (output) the \c k-th term      
      @param k number of the term
      
      @pre Dimension(Yk) == GetDim() && k>=0 && k<=GetOrder() */
  virtual void GetOneTerm( INTERVAL_VECTOR & Yk, int k ) = 0;
  
  /** Generates Taylor terms. This function generates \c k Taylor
      coefficients for the solution of an ODE with a point initial
      condition \c Y and a point of expansion enclosed by \c h.

      @see GenerateTerms 
      
      @param Y point initial condition
      @param h interval enclosing the point of expansion
      @param k order of the Taylor series
      
      @pre k>0 && k<=GetMaxOrder() && Dimension(Y) == GetDim() && NonZero(h)
      */
  void GenerateTerms( const VECTOR & Y, const INTERVAL & h, int k );
  
  /// Destructor
  virtual ~TAYLOR_ODE();

protected:
  
  /** Constructor.
      @param n size of the problem 
      @pre n>0
  */
  TAYLOR_ODE( int n );

};


/// Pointer to a TAYLOR_ODE object.
typedef TAYLOR_ODE*  PtrTaylGenODE;


inline TAYLOR_ODE :: TAYLOR_ODE( int n ) : TAYLOR_CTRL( n ) 
{
  ;
}


inline TAYLOR_ODE :: ~TAYLOR_ODE() 
{
  ;
}


inline void TAYLOR_ODE :: GenerateTerms( const VECTOR & Y, 
					 const INTERVAL & h, 
					 int k ) 
{
  INTERVAL_VECTOR Ytmp(Dimension(Y));
  Ytmp = Y;
  GenerateTerms( Ytmp,h,k ); 
}


#endif
