
// FILE taylvar.h


#ifndef INCLUDED_TAYLVAR_H
#define INCLUDED_TAYLVAR_H


#include "taylctrl.h"


/** Abstract class for generating Jacobians of Taylor coefficients.
    
    Given the IVP \f$y^\prime = f(y)\f$, \f$y(t_{j}) \in [y_{j}]\f$, this
    class provides the interface for a Taylor series generator for the
    solution to the variational equation \f$Y^\prime = \frac{\partial
    f}{\partial y}Y\f$, \f$Y(t_{j}) = I\f$.
    
    @author Ned Nedialkov
    @date   8 June 2001 
*/
class TAYLOR_VAR : protected TAYLOR_CTRL
{

public:
  
  /** Generates Taylor terms. 
      
      This function generates \c k Taylor terms for the solution of
      the variational equation with an interval initial condition \c Y
      and a point of expansion enclosed by \c h.
      
      @param Y interval vector enclosing the initial condition
      @param h interval enclosing the point of expansion
      @param k order of the Taylor series
      
      @pre k>0 && k<=GetMaxOrder() && Dimension(Y) == GetDim() && NonZero(h) 
  */
  virtual void GenerateTerms( const INTERVAL_VECTOR & Y, 
			      const INTERVAL & h, int k ) = 0;  
    
  /** Sums Taylor series. 
    
    This function computes the sum of the first \c k Taylor series
    terms that are generated in \ref GenerateTerms.
    
    @param Sum (output) the sum of the first \c k Taylor series terms
    @param k number of terms
    
    @pre Square(Sum,GetDim()) && k > 0 && k <= GetOrder(); 
    */
  virtual void SumTerms( INTERVAL_MATRIX & Sum, int k ) = 0;
    
  /** Obtains the \c k-th Taylor series term. 
    
    This function obtains the \c k-th term from the Taylor terms
    generated in \c GenerateTerms.
    
    @param Yk (output) the \c k-th term.
    @param k number of the term
    
    @pre Square(Yk,GetDim()) && k>=0 && k<=GetOrder() 
    */
  virtual void GetOneTerm( INTERVAL_MATRIX & Yk, int k ) = 0;
  
  /// Destructor.
  virtual ~TAYLOR_VAR();
  
  
protected:
  
  /** Constructor.

      @param n size of the problem 
      @pre  n > 0
  */
  TAYLOR_VAR( int n );

};


/// Pointer to a TAYLOR_VAR object.
typedef TAYLOR_VAR*  PtrTaylGenVar;


inline TAYLOR_VAR :: TAYLOR_VAR( int n ) : TAYLOR_CTRL(n) 
{
  ;
}


inline TAYLOR_VAR :: ~TAYLOR_VAR()
{
  ;
}

  
#endif
