
// FILE vnddisp.cc

#include <stdlib.h>
#include <unistd.h>
#include "plot.h"
  
#include "odesolv.h" 
#include "odenum.h"
#include "vndutils.h"
   
static char buf[80];
static char comp[20];


void GetApriori( FILE *out, FILE *in, int k, int n );
void GetSolut( FILE *, FILE *, int , int );
void GetSolut( FILE *, FILE *, int , int, int );
char * MakePlotCommand( const char * PlotCommand, const char * PlotFile );

static char xlabel[] = "t";
static char ylabel[30];
static char title [30];
static char comp_name[30] = {""};

bool APriori = true;

void SetPlotParam( const char *xlab, const char *ylab, const char *tit, 
		   const char *comp )
{
  strcpy( xlabel, xlab );
  strcpy( ylabel, ylab );
  strcpy( title,  tit );
  strcpy( comp_name, comp );
}

void ODE_NUMERIC :: 
DisplaySolution( int component, const char *filename ) const
{
  /* Get the display command */
  assert(component<=GetSize());
  assert(component>0);
  
  /* Create the plot command */
  char buf[30];
  strcpy(buf, PLOT_COMMAND);
  
  char *d = buf + strlen(buf);
  strcpy(d, "  ");
  
  d = buf + strlen(buf);
  /* PLOT_FILE is the file that will be loaded with gnuplot, i.e., 
     the command that will be executed is 
     
     gnuplot PLOT_FILE
  */
  
  strcpy(d, PLOT_FILE);
  
  /* Get the solution component from SOLUTION_DATA_FILE, whcih contains 
     all solutions */
  
  FILE *in  = fopen(SOLUTION_DATA_FILE, "r");
  assert(in!=NULL);
  
  FILE *out = fopen(PLOT_SOLUTION, "w");
  assert(out!=NULL);
    
  GetSolut(out, in, component,GetSize());
  fclose(in);
  fclose(out);
  
  /* Now, get the appriori enclosure from APRIORI_DATA_FILE */
  in  = fopen(APRIORI_DATA_FILE, "r");
  assert(in!=NULL);
  
  out = fopen(PLOT_APRIORI, "w");
  assert(out!=NULL);
  
  GetApriori(out, in, component,GetSize());
  
  fclose(in); 
  fclose(out); 
  
  /* Create a label for the graph. The label is of the form
     Y(component), for example Y(1), Y(2),..  */
  
  strcpy(comp, "Y(");
  sprintf(comp+2, "%d", component);
  strcpy(comp + strlen(comp), ")");
  
  //  if ( comp_name[0] != "" )
      strcpy( comp, comp_name );

  /* Create the file that will be loaded with gnuplot */
  out = fopen(PLOT_FILE, "w");
  assert(out!=NULL);

  fprintf(out, " set xlabel '%s'; " , xlabel );
  fprintf(out, " set ylabel '%s'; " , ylabel );
  fprintf(out, " set title  '%s'; " , title  );
      
  
  if (filename)
    if ( APriori )
      {
	fprintf(out, "set term postscript eps;"
		"set size 5.04/5, 4.14/5;"
		"set output '%s' ; plot '%s' title  'Tight Bounds' with lines 1,"
		"'%s' title 'A Priori Bounds' with lines 3 \n", 
		filename, PLOT_SOLUTION,  PLOT_APRIORI );
      }
    else
      fprintf(out, "set term postscript eps; "
	      "set size 5.04/5, 4.14/5;" 
	      "set output '%s' ; plot '%s'  title  '%s' with lines 1\n",
	      filename, PLOT_SOLUTION, comp );
  else
    {
      if ( APriori )
	fprintf(out, " plot '%s'  title  'Tight Bounds' with lines 1,"
		"'%s' title 'A Priori Bounds' with lines 3 \n", 
		PLOT_SOLUTION,  PLOT_APRIORI );
      else
	fprintf(out, " plot '%s'  title  '%s' with lines 1\n",
		PLOT_SOLUTION,  comp );
      
      fprintf(out," %s", "pause -1 'Press return key to continue...'");
    }
  
  fclose(out); 
  system (buf);
}


void ODE_SOLVER  :: 
DisplayStepSize( const char *filename  )  const
{
  int Ok;
  char * plot = MakePlotCommand(PLOT_COMMAND, PLOT_STEP);
  
  /* Create the file that will be loaded with gnuplot */ 
  FILE *out =  fopen(PLOT_STEP, "w"); 
  assert(out!=NULL); 
  
  if (filename)
    fprintf(out,  "set term postscript eps; set nozeroaxis;"
	    "set ylabel 'Stepsize' 1; set xlabel 'Step number' 0, -1;"
	    "set output '%s' ;"
	    "plot '%s' notitle with lines 1 \n", 
	    filename, STEPSIZE_DATA_FILE);
  else
    {
      Ok = fprintf(out,   "plot '%s' title 'Stepsize vs. t' with lines 1 \n", 
		   STEPSIZE_DATA_FILE);
      assert(Ok!=EOF);
      Ok = fprintf(out," %s", "pause -1 'Press return key to continue...'");
      assert(Ok!=EOF);
    }
  Ok = fclose(out); 
  assert(Ok!=EOF);
  
  system (plot);
}


void ODE_SOLVER :: DisplayOrder( const char * file_name ) const 
{
  char * plot = MakePlotCommand(PLOT_COMMAND, PLOT_ORDER);
  
  /* Create the file that will be loaded with gnuplot */ 
  FILE *out =  fopen(PLOT_ORDER, "w"); 
  assert(out!=NULL); 
  
  fprintf(out, "plot '%s' title 'Order vs. Step Number' with lines 1 \n", 
	  ORDER_DATA_FILE);
  fprintf(out," %s", "pause -1 'Press return key to continue...'");
  fclose(out); 
  
  system (plot);
}


void ODE_NUMERIC :: 
DisplayPhase( const int Comp1, const int Comp2, const char *filename )  const 
{
  /* Display the phase portrait of Comp2 vs. Comp1. */  
  
  assert(Comp1 <= GetSize());
  assert(Comp2 <= GetSize());
  
  // Open the solution and phase data files.
  FILE *SolutionFile = fopen(SOLUTION_DATA_FILE, "r");
  assert(SolutionFile != NULL);
  FILE *PhaseFile = fopen(PHASE_DATA_FILE, "w");
  assert(PhaseFile != NULL);
  
  /* Get the Comp1 and Comp2 from the SOLUTION_DATA_FILE and write
     them into the PHASE_DATE_FILE. */
  GetSolut(PhaseFile, SolutionFile, Comp1, Comp2, GetSize());
  
  fclose(SolutionFile);
  fclose(PhaseFile);
  
  // Create the plot command.
  char * plot = MakePlotCommand(PLOT_COMMAND, PLOT_PHASE); 
  
  // Create the file that will be loaded with gnuplot.
  FILE *out =  fopen(PLOT_PHASE, "w"); 
  assert(out!=NULL); 
  
  // Create the title string.
  sprintf(comp, "%s%d%s%d%s", "Y(", Comp2, ") vs. Y(", Comp1, ")");
  
  if (filename)
    {
      
      fprintf(out, "set term postscript eps; set output '%s' ;"
	      "plot '%s' title '%s' with lines 1 \n", 
	      filename, PHASE_DATA_FILE, comp);
    }
  else
    {
      fprintf(out, "plot '%s' title '%s' with lines 1 \n", 
	      PHASE_DATA_FILE, comp);
      fprintf(out," %s", "pause -1 'Press return key to continue...'");
    }
  
  fclose(out);  
  
  system (plot); 
}


FILE * OpenFile( const char * name, const char * mode )
{
  FILE *f;
  f = fopen( name, mode );
  if ( f == NULL )
    perror( "VNODE Error");
  
  return f;
}


void CloseFile( FILE * f )
{
  
  if ( fclose(f) == EOF )
    perror( "VNODE Error" );
}


char * MakePlotCommand(const char * PlotCommand, const char * PlotFile)
{
  assert(PlotCommand!=NULL);
  assert(PlotFile!=NULL);
  
  strcpy(buf, PlotCommand);
  char *d = buf + strlen(buf);
  strcpy(d, "  ");
  d = buf + strlen(buf)-1;
  strcpy(d, PlotFile);
  
  return buf;
}


void WriteStepSize(FILE *f, double t, double step)
{
  int Ok;
  assert(f!=NULL); 
  Ok = fprintf( f,"%f %e\n", t, step );
  assert(Ok!=EOF);
}


void WriteOrder(FILE *f, int t, int order)
{
  assert(f!=NULL);
  int Ok;
  Ok = fprintf( f,"%d %d\n", t, order );
  assert(Ok!=EOF);
}


void GetSolut( FILE *out, FILE *in, int k, int n)
{
  /*
    Reads the k-component of the solution from a file *in
    and writes it into file *out
  */
  
  float t, left,right;
  char  c;
  int cc;
  
  // In the first while, we get the left component. 
  while (1)
    {
      cc = fscanf(in, "%f", &t);
      if (cc == EOF)
	break;
      
      for ( int i = 1; i <= n; i++)
	{
	  cc =  fscanf( in," %f %f", &left, &right );
	  if ( i == k)
	    fprintf ( out , "%f %f\n", t, left);
	}
      fscanf(in, "%c", &c);
    }
  fprintf(out, "\n\n");
  
  /* Now, position the file pointer to the beginning of the file and 
     get the second component. */
  fseek(in,0,0);
  
  while (1)
    {
      cc = fscanf(in, "%f", &t);
      if (cc == EOF)
	break;
      
      for ( int i = 1; i <= n; i++)
	{
	  cc =  fscanf( in," %f %f", &left, &right );
	  if ( i == k)
	    fprintf ( out , "%f %f\n", t,  right);
	}
      fscanf(in, "%c", &c);
    }
  fprintf(out, "\n\n");
}


void GetSolut( FILE *out, FILE *in, int Comp1, int Comp2, int n )
{
  /* Reads the Comp1 and Comp2 components of the solution from a file
   * in and writes them into file *out */
  
  assert( Comp1 != Comp2 );
  assert( Comp1 <= n );
  assert( Comp2 <= n );
  
  /* We use floats to store the temporary variables since we cannot expect
     high precision from the plotting program. */
  
  // The time variable.
  float t; 
  // Left and right components of an interval vector.
  float left, right; 
  INTERVAL_VECTOR TmpVec(2);
  
  char c;
  int cc;
  int Two; 
  
  while (1)
    {
      cc = fscanf(in, "%f", &t);
      if (cc == EOF)
	break;
      
      Two = 0;
      for ( int i = 1; i <= n; i++)
	{
	  cc =  fscanf(in, " %f %f", &left, &right );
	  if ( i == Comp1 )
	    {
	      TmpVec(1) = INTERVAL(left, right);
	      Two++;
	    }
	  if ( i == Comp2)
	    {
	      TmpVec(2) = INTERVAL(left, right);
	      Two++;
	    }
	  
	  if ( Two==2 )
	    WritePhase( out, TmpVec );
	  
	} /* for */
      fscanf(in, "%c", &c);
    }
}


void GetApriori( FILE *out, FILE *in, int k, int n)
{
  fprintf ( out , "# Plot A PRIORI Enclosure \n\n");
  /*
    Reads the k-component of the solution from a file *in
    and writes it into file *out
  */
  
  float t, left,right;
  char  c;
  int   cc;
  
  /* Get the LOWER bound. */
  
  int first = 0;
  float Left = 0.0;
  
  while (1)
    {
      cc = fscanf(in, "%f", &t); /* Get t */
      if (cc == EOF)          /* Here we should reach the EOF */
	break;
      
      if ( first ) 
	fprintf ( out , "%f %f\n\n", t, Left);
      
      for ( int i = 1; i <= n; i++)
	{
	  cc =  fscanf( in," %f %f", &left, &right );/* Get lower and upper
							bounds */
	  if ( i == k)
	    {
	      fprintf ( out , "%f %f\n", t, left);
	      Left = left;
	    }
	}

      fscanf(in, "%c", &c);
      first = 1;
    }
    
  fprintf(out, "\n");
  fseek(in,0,0);
  
  /* Get the UPPER bound */
  first = 0; 
  float Right = 0;
  while (1)
    {
      cc = fscanf(in, "%f", &t);
      if (cc == EOF)
	break;
      
      if (first) 
	fprintf ( out , "%f %f\n\n", t, Right);
      
      for ( int i = 1; i <= n; i++)
	{
	  cc =  fscanf( in," %f %f", &left, &right );
	  if ( i == k)
	    {
	      fprintf ( out , "%f %f\n", t,  right);
	      Right = right;
	    }
	}
      fscanf(in, "%c", &c);
      first = 1;
    }
  
  fseek(in, 0,0);
  fprintf ( out , "# Plot BOXES \n\n");
  
  first = 0;
  while (1)
    {
      cc = fscanf(in, "%f", &t);
      if (cc == EOF)
	break;
      
      if (first) 
	{
	  fprintf ( out , "%f %f\n", t,  Left);
	  fprintf ( out , "%f %f\n\n", t,  Right);
	}
      for ( int i = 1; i <= n; i++)
	{
	  cc =  fscanf( in," %f %f", &left, &right );
	  if ( i == k)
	    {
	      fprintf ( out , "%f %f\n", t,  left);
	      fprintf ( out , "%f %f\n\n", t,  right);
	      Left = left;
	      Right = right;
	    }
	}
      fscanf(in, "%c", &c);
      first = 1;
    }
}


void WriteSolution( FILE *f, double t, const INTERVAL_VECTOR *y )
{
  assert(f!=NULL);
  assert(y!=NULL);
  
  int n = Dimension(*y);
  assert(n>0);
  
  fprintf( f, "%f", t);
  
  for (int i = 1; i <= n; i++)
    fprintf( f," %f %f", Inf( (*y)(i) ), Sup( (*y)(i)) );
  fprintf( f,"%c", '\n');
}


void WritePhase( FILE *f, const INTERVAL_VECTOR &y )
{
  assert(f!=NULL);
  
  int n;
  n = Dimension(y);
  assert(n==2);
  
  if ( MaxNorm(Diam(y)) < 1e-3*MaxNorm(y) )
    {
      fprintf( f,"\n%f %f", Mid(y(1)), Mid(y(2)) );
    }
  else
    {
      fprintf( f,"\n%f %f\n", Inf( y(1) ), Inf(y(2)) );
      fprintf( f,"%f %f\n",   Sup( y(1) ), Inf(y(2)) );
      fprintf( f,"%f %f\n",   Sup( y(1) ), Sup(y(2)) );
      fprintf( f,"%f %f\n",   Inf( y(1) ), Sup(y(2)) );
      fprintf( f,"%f %f\n",   Inf( y(1) ), Inf(y(2)) );
      fprintf( f,"%c", '\n');
    }
  
}











