
// FILE vodesolv.cc 

#include <stdio.h>
#include <iostream>
using namespace std;
#include "paral.h"
#include "vnderror.h"
#include "vndutils.h"
#include "vodesolv.h"
#include "plot.h"
#include "vndout.h"
#include "vndtime.h"

char * OutputString = "    %.4e         %.4e ";
FILE * f_solution, *f_apriori;

static void InitOutput ( const PtrVODESolver & Solver, 
			 bool Continue, const PtrODENumeric & ODE );

static void Output ( const PtrVODESolver & Solver, int Order, double t, 
		     double h, int Steps, 
		     const INTERVAL_VECTOR & Yinit, 
		     const INTERVAL_VECTOR & Ytight );

static void FinishOutput ( const PtrVODESolver & Solver, 
			   const PtrODENumeric & ODE );

static void PrintError( double t, double Error );

extern void CompareSolution( const PtrODENumeric &, const INTERVAL_VECTOR &, 
			     const INTERVAL &, char *, int );




void VODE_SOLVER :: IntegrateTo  ( double  Tend, bool Continue )
{
  /* If Continue == false, integrate from 
     ODE->GetT0() to Tend
     
     If Continue == true, integrate from 
     ODE->GetTcur() to Tend
     */
  
  // Start timing
  double StartTime = GetTime();

  SolverFlags->ContinueIntegration(Continue);

  // Initialize the solver
  InitSolver();
  
  // Initialize the output
  InitOutput ( this, SolverFlags->ContinueOutput(), ODE );
  
  double t = ODE->GetTcur(); // current point
  assert( t!=Tend );  
  
  bool Forward = Tend > t ? true : false; 
  // true if the integration is forward
  
  int  Order;  // order of the method
  double h;      // current stepsize 
  double h_old;  // previous stepsize
  
  SolverFlags->FirstStep( !Continue );
  
  while ( t != Tend )   
    {
      /* Select order. In a future release, a variable order strategy
	 should be implemented. */
      Order = OrderControl->CompOrder( this );
      
      // Compute the minimum stepsize allowed between t and Tend.
      double h_min = StepControl->CompMinStepSize( t, Tend );
      
      if ( SolverFlags->FirstStep() )
	{
	  // Select initial stepsize
	  h = StepControl->PredictFirstStep( ODE, DataRepr, Order );
	  if ( h == 0 ) 
	    h = h_min;

	  /* if  ( !Forward )
	     h = -h;
	     */
	}
      else
	/* Obtain the stepsize suggested by the stepsize controller on
	   the previous step. */
	h = StepControl->GetStepSize();

      if  ( !Forward )
	h = -h;
	
      /* If we a close to the endpoint in the sense that  h > |Tend - t|,
	 then, reduce the stepsize.
	 */
      h = StepControl->IfLastStep( t, h, Tend );
      
      // Try to validate existence and uniqueness with h, or a stepsize
      // smaller than |h|. 
      if ( !InitEncl->Validate( Yinit, LocalError, h, Order, ODE, this ) )
	break;

      if ( fabs(h) < h_min )
	break;            

      // Set the initial enclosure.
      ODE->SetInitEncl( Yinit );
      
      bool accepted = false;
      
      while ( !accepted )      
	{	
	  // Compute the next intergration point.
	  double Tnext = StepControl->CompNextPoint( h, t, Tend );
	  
	  // Compute an interval containing Tnext - t.
	  INTERVAL Step = SubBounds( Tnext, t );

	  // Compute tight bounds.
	  TightEncl->CompEncl( Ytight, LocalError, Step, Order, ODE, this );

	  h_old = ( Forward == true ) ? Inf(Step) : Sup(Step);
	  
	  // Predict a stepsize. This function returns the new
	  // stepsize in h and true, if the step is accepted, and
	  // false if it is rejected.
	  accepted = StepControl->PredictStep( h, h_old, ODE, 
					       LocalError, Ytight, 
					       Order );

	  if ( fabs(h) < h_min )
	    break;
	  
	  SolverFlags->AcceptedStep(accepted); 
	  SolverStats->AccRejStep( accepted );
	  
	  if ( accepted )
	    t = Tnext;
	} // while ( !accepted )
      
      if ( fabs(h) < h_min )
	break;

      // The step has been accepted.
      
      // Set the tight enclosure.
      ODE->SetTightEncl( Ytight );

      // Set the point of the enclosure.
      ODE->SetTcur( t );              
      
      // Set the stepsize.
      StepControl->SetStepSize( h );
      
      // Set the errors. 
      SetErrors ( t, Ytight );
      
      // Output data
      Output ( this, Order, t, h_old, SolverStats->GetAcceptedSteps(), 
	       Yinit, Ytight );
      
      if ( SolverFlags->FirstStep() )
	SolverFlags->FirstStep(false);
    }
  
  
  FinishOutput( this, ODE );
  
  // Add the user time
  SolverStats->AddUserTime( TotalTime(StartTime, GetTime()) );
  
  PrintData();  
}  


void PrintError( double t, double Error )
{
  fprintf( stderr, OutputString, t,  Error );
  DeleteOutput( stderr, t );
}


void InitOutput( const PtrVODESolver & Solver, bool Continue, const PtrODENumeric & ODE )
{
  if ( Solver->GraphOutput() )
    {
      if ( Continue )
	OpenFiles("a");
      else
	OpenFiles("w");
      
      WriteSolution( f_solution, ODE->GetTcur(), &(ODE->GetTightEncl()));
    }
  
#ifdef DEBUG
  if ( ODE->ExactSol() )
    VnodeMessage("Comparing with the exact solution...");
#endif   
  
  if ( Solver->VerboseOutput() )
    fprintf( stderr, "\n %s \n", " Integrated at       Global Error");
}


void Output( const PtrVODESolver & Solver, int Order, double t, double h, 
	     int Steps, const INTERVAL_VECTOR & Yinit, 
	     const INTERVAL_VECTOR & Ytight )
{
  if ( Solver->VerboseOutput() )
    PrintError( t, MaxNorm (Diam(Ytight)) );
  
  if ( Solver->GraphOutput() )
    WriteData( Order, t, h, Steps, Yinit, Ytight );
}


void FinishOutput( const PtrVODESolver & Solver, const PtrODENumeric & ODE )
{
  if ( Solver->GraphOutput() )
    {
      WriteSolution( f_apriori, ODE->GetTcur(), &(ODE->GetInitEncl()) );
      CloseFiles();
    }
  
  if ( Solver->VerboseOutput() )
    fprintf( stderr, "\n");
}




  

VODE_SOLVER :: ~VODE_SOLVER()
{   
  DELETE(OrderControl);
  DELETE(StepControl);
  DELETE(InitEncl);
  DELETE(TightEncl);
}


void VODE_SOLVER :: InitSolver()
{
  if ( SolverFlags->ContinueIntegration() == false )  
    // The integration is from ODE->GetT0(). We have to initialize the
    // solver.
    {
      // Set a solution object. When an ODE_NUMERIC object is created,
      // a solution object is not created since we do not know what
      // solution object a solver may require. In this solver, we need
      // a parallelepiped representation for the solution.
      ODE->SetSolution( new PARALL( ODE->GetSize() ) );
      
      // Initialize the ODE
      InitSolution<PARALL>( ODE );
      
      // Initialize the submethods
      TightEncl->Init(ODE); 
      InitEncl->Init(ODE);
      StepControl->Init(ODE);
      
      SolverStats->Reset();

      // We do not have to reset the flags and statistics since they
      // are reset in the creation of the solver.
      SolverFlags->AcceptedStep(true);
      
      // Allocate space for the this class's variables.
      ResizeSpace( ODE->GetSize());
    }
  else 
    {
      // We continue the integration from the current point and the
      // current solution representation.


      // The solver continues from a new initial condition.
      // This initial condition must be set.  
      //    *GetSolution<PARALL>(ODE) = *ODE->GetPtrInitCond();
    }
      
}


void CompareSolution( const PtrODENumeric & ODE, 
		      const INTERVAL_VECTOR & Ytight, 
		      const INTERVAL & t, char *file, int line )
{
  const int N = ODE->GetSize();
  INTERVAL_VECTOR ExactSolution(N), Ytmp(N);
  
  ODE->CompExactSol( ExactSolution, t );
  if ( !Intersection(Ytmp, Ytight, ExactSolution) )
    {
      cerr.precision(16);
      cerr << " t =          " << t << endl;
      cerr << " Tight Encl = " << Ytight  << endl;
      cerr << " Exact Sol  = " << ExactSolution << endl;
      VnodeError( "%s:%d : Cannot compute an enclosure of the solution.",
		  file, line );
    }
}


void VODE_SOLVER :: ResizeSpace( int N )
{
  Resize( Ytight, N );
  Resize( Yinit, N );
  Resize( LocalError, N );
}







