
// FILE vodesolv.h 


#ifndef  INCLUDED_VODESOLV_H
#define  INCLUDED_VODESOLV_H


#include "orderctrl.h"
#include "initencl.h"
#include "tgtencl.h"


class   VODE_SOLVER;

/// Pointer to a VODE_SOLVER object.
typedef VODE_SOLVER* PtrVODESolver;


/** Solver class.
    
    @author  Ned Nedialkov
    @date    8 June 2001 
    */
class VODE_SOLVER : public ODE_SOLVER
{
public:
  
  /** Constructor.
    
    @param ODE    pointer to a problem
    @param Data   pointer to a data representation object
    @param Order  pointer to an object that implements order control
    @param Step   pointer to an object that implements stepsize control
    
    @param Init pointer to an object for validating existence and
    uniqueness of the solution and computing a priori bounds on the
    solution
    
    @param Tight pointer to an object for computing tight bounds on
    the solution
    
    @pre NotNull(ODE) && NotNull(Data) && NotNull(Order) && 
    @pre NotNull(Step) && NotNull(Init) && NotNull(Tight) */
  VODE_SOLVER( const PtrODENumeric  ODE , 
	       const PtrDataRepr    Data, 
	       const PtrOrderCtrl   Order,
	       const PtrStepCtrl    Step,
	       const PtrInitEncl    Init,
	       const PtrTightEncl   Tight );
  
  
  /** Integrator function. 
   */
  void IntegrateTo( double tEnd, bool Continue = false );
  
  /** Sets an order control method.
    
    @param Order pointer to an object that implements order control
    @pre NotNull(Order)
    */
  void SetOrderCtrl ( const PtrOrderCtrl Order );  
  
  /** Sets an initial enclosure method. 
    
    @param Init pointer to an object for validating existence and
    uniqueness of the solution and computing a priori bounds on the
    solution.
    
    @pre NotNull(Init) 
    */
  void SetInitEncl  ( const PtrInitEncl  Init  );  
  
  /** Sets a tight enclosure method.
    
    @param Tight pointer to an object for computing tight bounds on
    the solution
    
    @pre NotNull(Tight)
    */
  void SetTightEncl ( const PtrTightEncl Tight );  

  /** Returns the tight enclosure.
    
    This function returns a pointer to the method for computing tight bounds
    on the solution.
    */
  PtrTightEncl GetTightEncl () const;
      
  /// Destructor
  virtual ~VODE_SOLVER();   

  
protected:
  
  /**  Initializes the solver. */
  void InitSolver ();


protected:
  
  /// Pointer to an ORDER_CTRL object 
  PtrOrderCtrl    OrderControl; 
  
  /** Pointer to an object for validating existence and uniqueness of
    the solution and computing an initial enclosure on the solution.
    */
  PtrInitEncl     InitEncl;   
  
  /** Pointer to an object for computing a tight enclosure on the
    solution. */
  PtrTightEncl    TightEncl;    
  
  /// Enclosure of the local error.
  INTERVAL_VECTOR LocalError;
  
  /// Tight enclosure of the solution.
  INTERVAL_VECTOR Ytight;
  
  /// Initial enclosure of the solution.
  INTERVAL_VECTOR Yinit;
  
private:
  
  /** Allocates space. 
    @param n size of the problem 
    @pre n > 0 
    */
  void ResizeSpace( int n );
  
};   



inline VODE_SOLVER :: VODE_SOLVER( const PtrODENumeric  ODE, 
				   const PtrDataRepr    Data, 
				   const PtrOrderCtrl   Order, 
				   const PtrStepCtrl    Step, 
				   const PtrInitEncl    Init, 
				   const PtrTightEncl   Tight )
  
  : ODE_SOLVER(ODE,Data,Step),  OrderControl(Order),  
    InitEncl(Init), TightEncl(Tight)
{
  assert( NotNull(Order) );
  assert( NotNull(Init)  );
  assert( NotNull(Tight) );
}


inline void VODE_SOLVER :: SetOrderCtrl ( const PtrOrderCtrl  Order )
{ 
  assert(Order);
  DELETE(OrderControl);
  OrderControl = Order; 
}


inline void VODE_SOLVER :: SetInitEncl  ( const PtrInitEncl Init )
{ 
  assert(Init);
  DELETE(InitEncl);
  InitEncl = Init; 
}


inline void VODE_SOLVER :: SetTightEncl ( const PtrTightEncl Tight ) 
{ 
  assert(Tight);
  DELETE(TightEncl);
  TightEncl = Tight; 
}

inline PtrTightEncl VODE_SOLVER :: GetTightEncl () const
{
  assert(NotNull(TightEncl));
  return TightEncl;
}


#endif
