/*
 *  Copyright (c) 2014, Rong Zheng <rzheng@mcmaster.ca>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <pcap.h>
#include "ieee802_11.h"
#include "radiotap-parser.h"

#include "frame_handler.h"

void print_mac_address(MAC mac, char *label) {
	printf("%s: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x ", label,
			(unsigned int) (mac >> 40) & 0xff,
			(unsigned int) (mac >> 32) & 0xff,
			(unsigned int) (mac >> 24) & 0xff,
			(unsigned int) (mac >> 16) & 0xff, (unsigned int) (mac >> 8) & 0xff,
			(unsigned int) mac & 0xff);
}

int decode_mgmt_frame(const u_char * ptr, uint16_t fc, uint8_t len) {
	MAC bmac; // BSSID
	MAC smac; // source MAC
	MAC dmac; // dest MAC

	u_char *mac;

	uint16_t seq_ctl;
	uint16_t seq;

	printf("M ");
	printf("%9s ", mgmt_subtype_text[FC_SUBTYPE(fc)]);

	dmac = ether2MAC(ptr + 4);
	smac = ether2MAC(ptr + 10);
	bmac = ether2MAC(ptr + 16);

	print_mac_address(bmac, "bssid");
	print_mac_address(smac, "s");
	print_mac_address(dmac, "d");

	seq_ctl = pletohs(ptr + 22);

	seq = COOK_SEQUENCE_NUMBER(seq_ctl);
	printf("seq: %u ", seq);

	printf("\n");

#ifdef VERBOSE
	printf
	("BSSID %012llx\tSRC %012llx\tDST %012llx\tSEQ %u\t%s\n",
			bmac, smac, dmac, seq, mgmt_subtype_text[FC_SUBTYPE(fc)]);
#endif
	struct mgmt_body_t pbody;
	int offset = 0;
	const u_char *p = ptr + MGMT_HDRLEN;

	memset(&pbody, 0, sizeof(pbody));

	if (!TTEST2(*p,
			IEEE802_11_CAPINFO_LEN + IEEE802_11_STATUS_LEN + IEEE802_11_AID_LEN))
		return 0;

	switch (FC_SUBTYPE(fc)) {
	case ST_ASSOC_REQUEST:
		break;
	case ST_ASSOC_RESPONSE:
	case ST_REASSOC_RESPONSE:
		//------ Your code here -------
		parse_elements(&pbody, p, offset, len);
		break;

	case ST_PROBE_RESPONSE:
                //------ Your code here ------- 
		parse_elements(&pbody, p, offset, len);
		break;

	case ST_BEACON:
		//------ Your code here -------

		parse_elements(&pbody, p, offset, len);
		break;
	case ST_REASSOC_REQUEST:
		break;
	case ST_PROBE_REQUEST:
		break;
	case ST_ATIM:
		break;
	case ST_DISASSOC:
		break;
	case ST_AUTH:
		break;
	case ST_DEAUTH:
		break;
	}
	return 0;
}

int decode_data_frame(const u_char * ptr, uint16_t fc) {
	MAC bmac; //BSSID
	MAC smac; // source MAC
	MAC dmac; // dest MAC
	MAC ramac, tamac; //RA mac, and TA mac

	uint16_t seq_ctl;
	uint16_t frag;
	uint16_t seq;

	printf("D ");
	printf("%9s ", data_subtype_text[FC_SUBTYPE(fc)]);

	if (!FC_TO_DS(fc) && !FC_FROM_DS(fc)) {
		/* ad hoc IBSS */
		//        return -1;
	} else if (!FC_TO_DS(fc) && FC_FROM_DS(fc)) {
		/* frame from AP to STA */
		smac = ether2MAC(ptr + 16);
		dmac = ether2MAC(ptr + 4);
		bmac = ether2MAC(ptr + 10);
		print_mac_address(bmac, "bssid");
		print_mac_address(smac, "s");
		print_mac_address(dmac, "d");
	} else if (FC_TO_DS(fc) && !FC_FROM_DS(fc)) {
		/* frame from STA to AP */
		smac = ether2MAC(ptr + 10);
		dmac = ether2MAC(ptr + 16);
		bmac = ether2MAC(ptr + 4);
		print_mac_address(bmac, "bssid");
		print_mac_address(smac, "s");
		print_mac_address(dmac, "d");
	} else if (FC_TO_DS(fc) && FC_FROM_DS(fc)) {
		/* WDS */
		ramac = ether2MAC(ptr + 4);
		tamac = ether2MAC(ptr + 10);
		print_mac_address(ramac, "d");
		print_mac_address(tamac, "s");
	}
	seq_ctl = pletohs(ptr + 22);
	seq = COOK_SEQUENCE_NUMBER(seq_ctl);
	printf("seq: %u ", seq);

	printf("\n");

#ifdef VERBOSE
	printf
	("BSSID %012llx\tSRC %012llx\tDST %012llx\tSEQ %u\t%s\tDUR %u\n",
			bmac, smac, dmac, seq, data_subtype_text[FC_SUBTYPE(fc)],du);
#endif
	return 0;
}

int decode_ctrl_frame(const u_char * ptr, uint16_t fc) {
	MAC bmac; //BSSID
	MAC ramac, tamac; //RA mac, and TA mac

	printf("C ");
	printf("%9s ", ctrl_subtype_text[FC_SUBTYPE(fc)]);

	switch (FC_SUBTYPE(fc)) {
	case CTRL_PS_POLL:
		bmac = ether2MAC(ptr + 4);
		printf("bssid: %012llx ", bmac);
		break;
	case CTRL_RTS:
		ramac = ether2MAC(ptr + 4);
		tamac = ether2MAC(ptr + 10);
		printf("d: %012llx ", ramac);
		printf("s: %012llx ", tamac);
		break;
	case CTRL_CTS:
		ramac = ether2MAC(ptr + 4);
		printf("d: %012llx ", ramac);
		break;
	case CTRL_ACK:
		ramac = ether2MAC(ptr + 4);
		printf("d: %012llx ", ramac);
		break;
	case CTRL_CF_END:
		bmac = ether2MAC(ptr + 10);
		printf("bssid: %012llx ", bmac);
		break;
	case CTRL_END_ACK:
		bmac = ether2MAC(ptr + 10);
		printf("bssid: %012llx ", bmac);
		break;
	default:
		return -1;
		//add the case statements for QoS control frames once ieee802_11.h is updated
	}

	printf("\n");

#ifdef	VERBOSE
	if ( FC_SUBTYPE(fc)==CTRL_CTS || FC_SUBTYPE(fc)==CTRL_ACK ) {
		printf("DST %012llx\t%s\tDUR %u\n",
				ramac, ctrl_subtype_text[FC_SUBTYPE(fc)],du);
	} else {
		printf("BSSID %012llx\tSRC %012llx\tDST %012llx\t%s\tDUR %u\n",
				bmac, ramac, tamac, ctrl_subtype_text[FC_SUBTYPE(fc)],du);
	}
#endif

	return 0;
}

static void parse_elements(struct mgmt_body_t *pbody, const u_char *p,
		int offset, int len) {
	/*
	 * We haven't seen any elements yet.
	 */
	for (;;) {
		if (!TTEST2(*(p + offset), 1)) {
			return;
		}
		switch (*(p + offset)) {
		case E_VENDOR_SPECIFIC:
			if (!TTEST2(*(p + offset), sizeof(VENDOR_SPEC_HEADER)))
				return;
			MAC oui = 0;
			//Format of the vendor specific info:
			//Vendor specific tag (1 byte),
			//tag length (1 byte): the length indicates the # of bits for the remaining of the vendor specific info
			//OUI (3 bytes),
			//information field { info_tag (1 byte), info_length (1 byte), info_content}
			//info_tag: AP ID (0), Mac Address (1), Radio Index (4)

			//------ Your code here -------
			if (oui == 0x000ce6) { //Meru network
			//------ Your code here -------
			} else {
				// other vendors: ignore
			//------ Your code here -------
			}
			break;
		default:
#if 0
			printf("(1) unhandled element_id (%d), tag length (%d) \n ",
					*(p + offset),  *(p + offset + 1));
#endif
			if (!TTEST2(*(p + offset), 2)) {
				return;
			}
			if (!TTEST2(*(p + offset + 2), *(p + offset + 1))) {
				return;
			}
			offset += *(p + offset + 1) + 2;
			break;
		}
	}
}
