/*
 *  Copyright (c) 2014, Rong Zheng <rzheng@mcmaster.ca>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pcap.h>
#include "ieee802_11.h"
#include "radiotap-parser.h"
#include "frame_handler.h"

void handle_packet(u_char *user, const struct pcap_pkthdr *header0,
		const u_char * packet0) {
	struct pcap_pkthdr localhdr;
	const struct pcap_pkthdr *header = &localhdr;
	localhdr = *header0;
	int radiotap_offset = 0;
	uint32_t radiotap_bm; //bitmap in radio tap header

	struct ieee80211_radiotap_header *radiotap_header;

	radiotap_header = (struct ieee80211_radiotap_header *) packet0;
	radiotap_offset = radiotap_header->it_len;
	radiotap_bm = radiotap_header->it_present;
	localhdr.caplen -= radiotap_offset;
	localhdr.len -= radiotap_offset;
	process_ratiotap_header((uint8_t *)radiotap_header, radiotap_offset);

	const u_char *packet = packet0 + radiotap_offset;

	uint32_t framelength = localhdr.len; //why use bpf_u_int32?
	struct timeval ts = localhdr.ts;

	uint16_t fc = pletohs(packet); //frame control
	u_int hdrlen = extract_header_length(fc);

	if (header->caplen < IEEE802_11_FC_LEN || header->caplen < hdrlen) {
		//fprintf(stderr, "short 802.11 packet %d bytes\n",header->caplen);
		return;
	}

	printf("%i_%i_%i_%i_%i_%i_%i_%i ", FC_TO_DS(fc), FC_FROM_DS(fc),
			FC_MORE_FLAG(fc), FC_RETRY(fc), FC_POWER_MGMT(fc), FC_MORE_DATA(fc),
			FC_WEP(fc), FC_ORDER(fc));

// fill in current_frame: type, sn
	switch (FC_TYPE(fc)) {
	case T_MGMT:
		if (decode_mgmt_frame(packet, fc, framelength) < 0)
			return;
		break;
	case T_DATA:
		if (decode_data_frame(packet, fc) < 0)
			return;
		break;
	case T_CTRL:
		if (decode_ctrl_frame(packet, fc) < 0)
			return;
		break;
	default:
		fprintf(stderr, "Unknown frame type 0x%x\n", FC_TYPE(fc));
		return;
	}

}


void process_ratiotap_header(uint8_t *buf, int buflen) {
	int pkt_rate_100kHz = 0, antenna = 0, pwr = 0, channel = 0, rssi = 0,
			noise = 0;
	struct ieee80211_radiotap_iterator iterator;
	int ret = ieee80211_radiotap_iterator_init(&iterator,
			(struct ieee80211_radiotap_header*) buf, buflen);

	while (!ret) {

		ret = ieee80211_radiotap_iterator_next(&iterator);

		if (ret)
			continue;

		/* see if this argument is something we can use */

		switch (iterator.this_arg_index) {
		/*
		 * You must take care when dereferencing iterator.this_arg
		 * for multibyte types... the pointer is not aligned.  Use
		 * get_unaligned((type *)iterator.this_arg) to dereference
		 * iterator.this_arg for type "type" safely on all arches.
		 */
		case IEEE80211_RADIOTAP_CHANNEL:
			channel = *((uint16_t *) iterator.this_arg);
			break;

		case IEEE80211_RADIOTAP_DBM_ANTSIGNAL:
		case IEEE80211_RADIOTAP_DB_ANTSIGNAL:
			rssi = (char) (*iterator.this_arg);
			break;

		case IEEE80211_RADIOTAP_RATE:
			/* radiotap "rate" u8 is in
			 * 500kbps units, eg, 0x02=1Mbps
			 */
			pkt_rate_100kHz = (*iterator.this_arg) * 5;
			break;
		case IEEE80211_RADIOTAP_DB_ANTNOISE:
		case IEEE80211_RADIOTAP_DBM_ANTNOISE:
			noise = (char) (*iterator.this_arg);
			break;

		case IEEE80211_RADIOTAP_DBM_TX_POWER:
			pwr = *iterator.this_arg;
			break;

		default:
			break;
		}

	} /* while more rt headers */
	printf("(Radiotap) rate = %4.1f Mb/s, freq = %d, rssi = %ddBm, noise = %ddBm\n",
			pkt_rate_100kHz / 10.0, channel, rssi, noise);
	/* discard the radiotap header part */
	buf += iterator.max_length;
	buflen -= iterator.max_length;
}



int main(int argc, char **argv) {
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_t* descr;
	u_char* args = NULL;
	int count = -1;

	descr = pcap_open_offline(argv[1], errbuf);

	if (descr == NULL) {
		printf("pcap_open_offline(): file %s -- %s\n", argv[1], errbuf);
		exit(1);
	}

	/* ... and loop */
	printf("WiFi frame parser starting...\n");
	pcap_loop(descr, count, handle_packet, args);
}

