/*
 *  Copyright (c) 2014, Rong Zheng <rzheng@mcmaster.ca>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.CAS765.android.sensordatacollector;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.List;

import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.net.wifi.ScanResult;
import android.net.wifi.WifiManager;
import android.os.Bundle;
import android.os.Environment;
import android.os.SystemClock;
import android.text.method.ScrollingMovementMethod;
import android.view.Menu;
import android.view.View;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.SeekBar;
import android.widget.SeekBar.OnSeekBarChangeListener;
import android.widget.TextView;

public class MainActivity extends Activity implements SensorEventListener {
	// If the values are set to 0 then we will sample as fast as we can
	private final int MAX_SAMPLE = 100;
	private final int DEFAULT_SAMPLING_RATE = 10;

	private Button mStartButton, mResetButton;
	private TextView mWiFiTextView, mAccXTextView, mAccYTextView,
			mAccZTextView, mGyroXTextView, mGyroYTextView, mGyroZTextView,
			mCompassTextView, mSensorSampleRateTextView,
			mWiFiSampleRateTextView;
	private CheckBox mWiFiCheckBox, mAccCheckBox, mGyroCheckBox,
			mCompassCheckBox;
	private SeekBar mSensorSlider;
	private SensorManager mSensorManager;
	private Sensor mAccelerometer, mGyro, mCompass;
	// all in Hz
	private int mSensorSamplingRate = DEFAULT_SAMPLING_RATE;
	private boolean isStarted = false;
	private WifiManager mWifiManager;
	private WifiReceiver mReceiverWifi;
	private OutputStreamWriter mAccOutWriter, mGyroOutWriter,
			mCompassOutWriter, mWiFiOutWriter;
	private FileOutputStream mAccOut, mGyroOut, mCompassOut, mWiFiOut;

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_main);

		// Initialize the views
		mWiFiTextView = (TextView) findViewById(R.id.WiFiReading);
		mWiFiTextView.setMovementMethod(new ScrollingMovementMethod());

		mAccXTextView = (TextView) findViewById(R.id.accX);
		mAccYTextView = (TextView) findViewById(R.id.accY);
		mAccZTextView = (TextView) findViewById(R.id.accZ);
		mGyroXTextView = (TextView) findViewById(R.id.gyroX);
		mGyroYTextView = (TextView) findViewById(R.id.gyroY);
		mGyroZTextView = (TextView) findViewById(R.id.gyroZ);
		mCompassTextView = (TextView) findViewById(R.id.compassReading);
		mSensorSampleRateTextView = (TextView) findViewById(R.id.sensorSamplingRateTextView);

		mWiFiCheckBox = (CheckBox) findViewById(R.id.WiFiCheckBox);
		mAccCheckBox = (CheckBox) findViewById(R.id.accelerometerCheckBox);
		mGyroCheckBox = (CheckBox) findViewById(R.id.GyroCheckBox);
		mCompassCheckBox = (CheckBox) findViewById(R.id.GeoMagneticCheckBox);

		mSensorSlider = (SeekBar) findViewById(R.id.sensorSamplingRate);

		mSensorSlider.setOnSeekBarChangeListener(new OnSeekBarChangeListener() {
			public void onProgressChanged(SeekBar seekBar, int progress,
					boolean fromUser) {
				mSensorSampleRateTextView.setText("Sampling rate = " + progress
						+ "(Hz)");
			}

			public void onStartTrackingTouch(SeekBar seekBar) {
				// TODO Auto-generated method stub
			}

			public void onStopTrackingTouch(SeekBar seekBar) {
				// TODO Auto-generated method stub
			}
		});

		mSensorSlider.setMax(MAX_SAMPLE);
		mSensorSlider.setProgress(DEFAULT_SAMPLING_RATE);

		// Check the availability of the sensors. Disable the corresponding
		// views if the sensor is not available
		mSensorManager = (SensorManager) getSystemService(Context.SENSOR_SERVICE);
		// get the accelerometer sensor
		if ((mAccelerometer = mSensorManager
				.getDefaultSensor(Sensor.TYPE_ACCELEROMETER)) == null) {
			mAccCheckBox.setEnabled(false);
		}

		if ((mGyro = mSensorManager.getDefaultSensor(Sensor.TYPE_GYROSCOPE)) == null) {
			mGyroCheckBox.setEnabled(false);
		}
		if ((mCompass = mSensorManager
				.getDefaultSensor(Sensor.TYPE_MAGNETIC_FIELD)) == null) {
			mCompassCheckBox.setEnabled(false);
		}
		
		if((mWifiManager = (WifiManager) getSystemService(Context.WIFI_SERVICE)) == null) {
			mWiFiCheckBox.setEnabled(false);
		}


		mStartButton = (Button) findViewById(R.id.start_button);
		mStartButton.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				if (!isStarted) {
					mStartButton.setText("Stop");
					startSensors();
					startWiFi();
					isStarted = true;
				} else {
					mStartButton.setText("Start");
					stopAll();
					isStarted = false;
				}
			}
		});

		mResetButton = (Button) findViewById(R.id.reset_button);
		mResetButton.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				resetAll();
			}
		});

		try {
			// get the file handlers for the external storage
			String state = Environment.getExternalStorageState();

			if (Environment.MEDIA_MOUNTED.equals(state)) {
				String dir = getExternalFilesDir(null) + File.separator
						+ System.currentTimeMillis();
				File projDir = new File(dir);
				if (!projDir.exists())
					projDir.mkdirs();

				File myFile = new File(dir + File.separator + "/acc");
				myFile.createNewFile();
				mAccOut = new FileOutputStream(myFile);
				mAccOutWriter = new OutputStreamWriter(mAccOut);

				myFile = new File(dir + File.separator + "/gyro");
				myFile.createNewFile();
				mGyroOut = new FileOutputStream(myFile);
				mGyroOutWriter = new OutputStreamWriter(mGyroOut);

				myFile = new File(dir + File.separator + "/compass");
				myFile.createNewFile();
				mCompassOut = new FileOutputStream(myFile);
				mCompassOutWriter = new OutputStreamWriter(mCompassOut);

				myFile = new File(dir + File.separator + "/wifi");
				myFile.createNewFile();
				mWiFiOut = new FileOutputStream(myFile);
				mWiFiOutWriter = new OutputStreamWriter(mWiFiOut);
			}
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	protected void onPause() {
		if (mReceiverWifi != null) {
			try {
				unregisterReceiver(mReceiverWifi);
			} catch (IllegalArgumentException e) {
				// receiver not registered
			}
		}
		super.onPause();
	}

	protected void onResume() {
		if (mReceiverWifi != null)
			registerReceiver(mReceiverWifi, new IntentFilter(
					WifiManager.SCAN_RESULTS_AVAILABLE_ACTION));
		super.onResume();
	}
	
	protected void onStop() {
		if (mAccOut != null) {
			try {
				mAccOutWriter.close();
				mAccOut.close();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

		if (mGyroOut != null) {
			try {
				mGyroOutWriter.close();
				mGyroOut.close();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

		if (mCompassOut != null) {
			try {
				mCompassOutWriter.close();
				mCompassOut.close();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

		if (mWiFiOut != null) {
			try {
				mWiFiOutWriter.close();
				mWiFiOut.close();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

		super.onStop();
	}

	void startSensors() {
		if (mAccCheckBox.isChecked()) {
			// YOUR CODE HERE: register the listener for accelerometer
		}

		if (mGyroCheckBox.isChecked()) {
			// YOUR CODE HERE: register the listener for gyro
		}

		if (mCompassCheckBox.isChecked()) {
			// YOUR CODE HERE: register the listener for compass
		}

		// Disable the slider after starting the sensors
		mSensorSlider.setEnabled(false);
	}
	
	void startWiFi() {
		if (mWiFiCheckBox.isChecked()) {
			// YOUR CODE HERE: 1. create an instance of the WiFiReceiver, 2. register the
			// broadcast receiver with SCAN_RESULTS_AVAILABLE_ACTION
		}
	}

	//Note: mWifiManager needs to be instantiated first
	class WifiReceiver extends BroadcastReceiver {
		public void onReceive(Context c, Intent intent) {
			StringBuilder sb = new StringBuilder();
			List<ScanResult> wifiList = mWifiManager.getScanResults();

			// YOUR CODE HERE: 1. Display the reading on the UI, 2. Write to the
			// storage

			// Sleep for 500us and start scanning again
			SystemClock.sleep(500);
			mWifiManager.startScan();
		}
	}

	//Upon stopping the sensors, 
	void stopAll() {
		mSensorSlider.setEnabled(true);

		//YOUR CODE HERE: unregister the sensor listener if exists

		if (mReceiverWifi != null)
			//YOUR CODE HERE: unregister the WiFiReceiver if exists


		try {
			if (mWiFiOutWriter != null)
				mWiFiOutWriter.flush();

			if (mGyroOutWriter != null)
				mGyroOutWriter.flush();
			
			if (mCompassOutWriter != null)
				mCompassOutWriter.flush();
			
			if (mAccOutWriter != null)
				mAccOutWriter.flush();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

	}

	void resetAll() {
		if (isStarted) {
			mStartButton.setText("Start");
			
			stopAll();
			isStarted = false;
		}
		mSensorSlider.setProgress(DEFAULT_SAMPLING_RATE);
		mWiFiCheckBox.setChecked(false);
		mAccCheckBox.setChecked(false);
		mGyroCheckBox.setChecked(false);
		mCompassCheckBox.setChecked(false);

		mAccXTextView.setText("X");
		mAccYTextView.setText("Y");
		mAccZTextView.setText("Z");
		mGyroXTextView.setText("X");
		mGyroYTextView.setText("Y");
		mGyroZTextView.setText("Z");
		mCompassTextView.setText("Azimuth");
		mWiFiTextView.setText("");
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		// Inflate the menu; this adds items to the action bar if it is present.
		getMenuInflater().inflate(R.menu.main, menu);
		return true;
	}

	@Override
	public void onAccuracyChanged(Sensor sensor, int accuracy) {
		// TODO Auto-generated method stub

	}

	// Handling sensor reading changes
	@Override
	public void onSensorChanged(SensorEvent event) {
		SensorReading acc, gyro, comp;
		if (event.sensor == mAccelerometer) {
			// YOUR CODE HERE: 1. Display the reading on the UI, 2. Write to the
			// storage
		}

		if (event.sensor == mGyro) {
			// YOUR CODE HERE: 1. Display the reading on the UI, 2. Write to the
			// storage
		}

		if (event.sensor == mCompass) {
			// YOUR CODE HERE: 1. Display the reading on the UI, 2. Write to the
			// storage
		}

	}

	private class SensorReading {
		String type;
		long timestamp;
		public float values[];

		public SensorReading(String t, long time, float v[]) {
			type = new String(t.toLowerCase());
			timestamp = time;
			values = v.clone();
		}

		@Override
		public String toString() {
			if (type.equals("gyro") || type.equals("acc"))
				return new String(timestamp + String.format("%.4f", values[0])
						+ " " + String.format("%.4f", values[1]) + " "
						+ String.format("%.4f", values[2]));

			if (type.equals("compass"))
				return new String(timestamp + String.format("%.4f", values[0]));

			return new String("Invalid sensor type");
		}
	}
}
