% Author: Rong Zheng, Oct. 20, 2015
%
function [ts angle] = HeadingFusion(acc, gyro, magnetic, weight)
% Heading estimation by fusing acc, gyro and magnetic sensor readings
% NOTE: assume device is facing up with device y aligned with heading
% direction
% -------------------------------------------------------------------------
% IPUT: acc, gyro, magnetic(:,4) -- the first column is time in millseconds
%       weight -- weight for the heading inferred by gyro and (1-weight)
%       for heading inferred by acc-mag. Example, set 1 is only gyro is
%       used
% OUTPUT: ts -- timestamps (in milliseconds) of heading angle estimate
%         angle -- ccw angle of device y with respect to earth magnetic north
%-------------------------------------------------------------------------

sTime = min([acc(1,1), gyro(1,1), magnetic(1,1)]);
eTime = max([acc(length(acc),1), gyro(length(gyro),1), magnetic(length(magnetic),1)]);

%Step 1: preprocess all IMU data to regular sampling interval of 20ms (or
%equal 50Hz)

sampleInterval = 20.0/1000; %ms
sampleFreq = 1/sampleInterval; %Hz

xq = (sTime:sampleInterval*1000:eTime);
newAcc = interp1(acc(:,1),acc(:,2:4), xq, 'nearest','extrap');
newMag = interp1(magnetic(:,1),magnetic(:,2:4), xq, 'nearest','extrap');
newGyro = interp1(gyro(:,1),gyro(:,2:4), xq, 'nearest','extrap');

% apply low-pass filter to acc & mag readings
LPF_cutoff_freq = 10;
[b,a] = butter(5, LPF_cutoff_freq/sampleFreq, 'low');

newAcc = filtfilt(b, a, newAcc);
newMag = filtfilt(b, a, newMag);

n = length(xq);

for i = 1:n
    accMagHeading(i,:) = EstimateHeadingAccMag(newAcc(i,:), newMag(i,:));

    if i > 1       
        gyroHeading(i,:) = EstimatePoseChangeGyro(newGyro(i,:), sampleInterval)*fusedHeading(i-1,:)';
        fusedHeading(i,:) = fuseTwoHeadings(gyroHeading(i,:), accMagHeading(i,:), weight);

    else
        fusedHeading(i,:) = accMagHeading(i,:);
    end
    angle(i) = atan2(fusedHeading(i,2), fusedHeading(i,1));
end
ts = xq;
angle = angle - 0.55; %Xuq: Compensate the measurement error
