#include <libnet.h>
#include <pcap.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define GNIP_FILTER "icmp[0] = 0"
#define ICMP_PAYLOAD_SIZE 56

void
tv_sub(struct timeval *out, struct timeval *in)
{
	if ( (out->tv_usec -= in->tv_usec) < 0) {   /* out -= in */
		--out->tv_sec;
		out->tv_usec += 1000000;
	}
	out->tv_sec -= in->tv_sec;
}

void usage(char *s) {printf("%s\n",s);}

int
main(int argc, char **argv)
{
	libnet_t *l = NULL;
	pcap_t *p = NULL;
	u_int8_t *packet;
	u_int32_t dst_ip, src_ip;
	u_int16_t id, seq, count;
	int c, interval = 0, pcap_fd, timed_out;
	u_int8_t loop; 
	u_int8_t payload[ICMP_PAYLOAD_SIZE];
	u_int32_t payload_s = ICMP_PAYLOAD_SIZE;
	libnet_ptag_t icmp = 0, ip = 0;
	char *device = NULL;
	fd_set read_set;
	struct pcap_pkthdr pc_hdr;
	struct timeval timeout;
	struct bpf_program filter_code;
	bpf_u_int32 local_net, netmask;
	struct libnet_ipv4_hdr *ip_hdr;
	struct libnet_icmpv4_hdr *icmp_hdr;
	char errbuf[LIBNET_ERRBUF_SIZE];

	while((c = getopt(argc, argv, "I:i:c:")) != EOF)
	{
		switch (c)
		{
			case 'I':
				device = optarg;
				break;
			case 'i':
				interval = atoi(optarg);
				break;
			case 'c':
				count = atoi(optarg);
				break;
		}
	}
	c = argc - optind;
	if (c != 1)
	{
		usage(argv[0]);
		exit(EXIT_FAILURE);
	}

	/* initialize the libnet library */
	l = libnet_init(LIBNET_RAW4, device, errbuf);
	if (l == NULL)
	{
		fprintf(stderr, "libnet_init() failed: %s", errbuf);
		exit(EXIT_FAILURE);
	}
	if (device == NULL)
	{
		device = pcap_lookupdev(errbuf);
		if (device == NULL)
		{
			fprintf(stderr, "pcap_lookupdev() failed: %s\n", errbuf);
			goto bad;
		}
	}
	/* handcrank pcap */
	p = pcap_open_live(device, 256, 0, 0, errbuf);
	if (p == NULL)
	{
		fprintf(stderr, "pcap_open_live() failed: %s", errbuf);
		goto bad;
	}
	/* get the subnet mask of the interface */
	if (pcap_lookupnet(device, &local_net, &netmask, errbuf) == -1)
	{
		fprintf(stderr, "pcap_lookupnet(): %s", errbuf);
		goto bad;
	}
	/* compile the BPF filter code */
	if (pcap_compile(p, &filter_code, GNIP_FILTER, 1, netmask) == -1)
	{
		fprintf(stderr, "pcap_compile(): %s", pcap_geterr(p));
		goto bad;
	}
	/* apply the filter to the interface */
	if (pcap_setfilter(p, &filter_code) == -1)
	{
		fprintf(stderr, "pcap_setfilter(): %s", pcap_geterr(p));
		goto bad;
	}
	dst_ip = libnet_name2addr4(l, argv[optind], LIBNET_RESOLVE);
	if (dst_ip == -1)
	{
		fprintf(stderr, "Bad destination IP address (%s).\n",
				libnet_geterror(l));
		goto bad;
	}

	src_ip = libnet_get_ipaddr4(l);
	if (src_ip == -1)
	{
		fprintf(stderr, "Can't determine source IP address (%s).\n",
				libnet_geterror(l));
		goto bad;
	}
	interval ? interval : interval = 1;
	timeout.tv_sec = interval;
	timeout.tv_usec = 0;
	pcap_fd = pcap_fileno(p);
	fprintf(stderr, "GNIP %s (%s): %d data bytes\n",
			libnet_addr2name4(dst_ip, 1), libnet_addr2name4(dst_ip, 0),
			LIBNET_IPV4_H + LIBNET_ICMPV4_ECHO_H + payload_s);
	loop = 1;
	for (id = getpid(), seq = 0, icmp = LIBNET_PTAG_INITIALIZER; loop; seq++)
	{

		memset(&payload, 0xa5, ICMP_PAYLOAD_SIZE); /* fill with pattern */
		gettimeofday((struct timeval *) &payload, NULL);
		icmp = libnet_build_icmpv4_echo(
				ICMP_ECHO, /* type */
				0, /* code */
				0, /* checksum */
				id, /* id */
				seq, /* sequence number */
				(u_int8_t *)&payload, /* payload */
				payload_s, /* payload size */
				l, /* libnet context */
				icmp); /* ptag */
		if (icmp == -1)
		{
			fprintf(stderr, "Can't build ICMP header: %s\n",
					libnet_geterror(l));
			goto bad;
		}
		ip = libnet_build_ipv4(
				LIBNET_IPV4_H + LIBNET_ICMPV4_ECHO_H + payload_s, /* length */
				0, /* TOS */
				id, /* IP ID */
				0, /* IP Frag */
				64, /* TTL */
				IPPROTO_ICMP, /* protocol */
				0, /* checksum */
				src_ip, /* source IP */
				dst_ip, /* destination IP */
				NULL, /* payload */
				0, /* payload size */
				l, /* libnet context */
				ip); /* ptag */
		if (ip == -1)
		{
			fprintf(stderr, "Can't build IP header: %s\n", libnet_geterror(l));
			goto bad;
		}
		c = libnet_write(l);
		if (c == -1)
		{
			fprintf(stderr, "Write error: %s\n", libnet_geterror(l));
			goto bad;
		}
		FD_ZERO(&read_set);
		FD_SET(pcap_fd, &read_set);
		for (timed_out = 0; !timed_out && loop; )
		{
			c = select(pcap_fd + 1, &read_set, 0, 0, &timeout);
			switch (c)
			{
				case -1:
					fprintf(stderr, "select() %s\n", strerror(errno));
					goto bad;
				case 0:
					timed_out = 1;
					continue;
				default:
					if (FD_ISSET(pcap_fd, &read_set) == 0)
					{
						timed_out = 1;
						continue;
					}
					/* fall through to read the packet */
			}
			packet = (u_int8_t *)pcap_next(p, &pc_hdr);
			if (packet == NULL)
			{
				continue;
			}
			ip_hdr = (struct libnet_ipv4_hdr *)(packet + 14);
			icmp_hdr = (struct libnet_icmpv4_hdr *)(packet + 14 +
					(ip_hdr->ip_hl << 2));
			if (ip_hdr->ip_src.s_addr != dst_ip)
			{
				continue;
			}
			if (ntohs(icmp_hdr->icmp_id) == id)
			{
				double rtt;
				struct timeval  *tvsend; 
				struct timeval tvrecv;

				gettimeofday(&tvrecv, NULL);
				tvsend = (struct timeval *) icmp_hdr->icmp_data;
				tv_sub(&tvrecv, tvsend);
				rtt = tvrecv.tv_sec * 1000.0 + tvrecv.tv_usec / 1000.0;


				fprintf(stderr, "%d bytes from %s: icmp_id=%d icmp_seq=%d ttl=%6.4f\n",
						ntohs(ip_hdr->ip_len),
						libnet_addr2name4(ip_hdr->ip_src.s_addr, 0),
						ntohs(icmp_hdr->icmp_id), ntohs(icmp_hdr->icmp_seq), rtt);
			}
		}
	}
	bad: libnet_destroy(l);
	pcap_close(p);
	return (EXIT_SUCCESS);
}
