\section{Construction Based on Atom Sets}

%{{{ Intro
According to the definition,
every homset of a relation algebra is an atomic Boolean lattice,
and the structure of atomic Boolean lattices
is completely determined by the set of atoms.
Together with join-distributivity
and isotonicity of converse,
every relation algebra is therefore
completely determined by the atom sets of its homsets,
and by the behaviour of converse and composition on these atoms.

We now use this fact to arrive at a more economic way
of defining relation algebras.

For an example that comes with a detailed explanation of this principle
see \sectref{McKenzie}.
%}}}

\medskip
Since we keep the morphism data type of atom set categories
abstract, we have to provide an explicit export list for this module:

%{{{ module Atomset
\begin{code}
module Atomset(ACat(..),acat_idmor_default,acat_idmor_defaultM
              ,SetMor(),mkSetMor,unSetMor,atmor
              ,atomsetCat,acat_TEST,acat_TEST'
              ,AAll(..),aall_isObj,aall_isAtom,aall_objects
                       ,aall_atomset,aall_idmor,aall_comp
              ,atomsetAll,atomsetDistrAll,atomsetDivAll,atomsetDed,atomsetRA
              ,aall_TEST
              ,showsAtomset0,showsAtomset', showsAtomset
              ,showsAtCompEntry0,showsAtCompDefault,showsAtCompEntry1
              ,showsAtComp0,showsAtComp', showsAtComp
              ,showsIdmor0,showsIdmor', showsIdmor
              ,showsACat0,showsACat', showsACat
              ,showsConv0,showsAtConv', showsAtConv
              ,showsAAll0,showsAAll', showsAAll
              ,showsARA0,showsARA', showsARA
              ,boolMatARASchows,writeBoolMatARA
              ,Cycle,cycleRepresentatives, cycles
              ,AtomCompTable,addCycle,tableAtComp,negTableAtComp
              ,allCycles,showsCycAtComp
              ,acatB,aallB
              ,distrAll_acat,distrAll_aall
              ,divAll_acat,divAll_aall
              ,ded_acat,ded_aall
              ,ra_acat,ra_aall
              ,MatAt,acatMat,aallMat
              ,matBtoAtCat,atCatToMatB
              ) where

import RelAlg
import Matrix

import FiniteMaps
import Sets

import List (nub,sort)
import ExtPrel
\end{code}
%}}}

%{{{ \subsection{Atom Category Definitions}
\subsection{Atom Category Definitions}

If we intend the morphisms of a category
to be sets of elements of some base set
(and we call these elements ``atoms''
 for their intended r\^ole in relation algebras),
if identical atoms are to be allowed to occur in different homsets, 
and if composition should preserve joins ad meets over these sets,
then such a category is determined by the following items:
\begin{itemize}
\item its objects,
\item for any two objects, the atoms of the respective homset, and
\item for any three objects and two atoms
  (from the respective homsets),
  the set of atoms that occur in the composition of the two atoms.
\end{itemize}
As for full categories,
we complete this list with well-definedness predicates for objects and atoms,
and with information about the identity morphisms:


\index{ACat@{\texttt{ACat}}}%
\index{acat*@{\texttt{acat\_*}}}%
\begin{code}
data ACat obj atom = ACat
  {acat_isObj   :: obj -> Bool
  ,acat_isAtom  :: obj -> obj -> atom -> Bool
  ,acat_objects :: [obj]
  ,acat_atomset :: obj -> obj -> [atom]
  ,acat_idmor   :: obj -> [atom]
  ,acat_comp    :: obj -> obj -> obj -> atom -> atom -> [atom]
  }
\end{code}

If such an atom category definition is well-defined,
then it is redundant;
in particular the information about identity atoms can be derived from
the enumerations and composition:

\index{acat_idmor_default@{\texttt{acat\_idmor\_default*}}}%
\begin{code}
acat_idmor_default :: Eq atom => ACat obj atom -> obj -> [atom]
acat_idmor_default ac o =
  let as = acat_atomset ac o o
      os = acat_objects ac
      testL p a b = all (`elem` [b]) (acat_comp ac o o p a b)
      testO p a = and (map (testL p a) (acat_atomset ac o p))
      reduceO as p = filter (testO p) as
  in foldl reduceO as os
\end{code}

Whenever we want to actually use this default when defining
an atom category description, we can considerably speed up
access to the identity by memorising it;
since demanding \verb|Ord| for objects is not a heavy constraint,
and the overhead for finite maps of the sizes we shall usually need
will be neglegible,
we use finite maps instead of arrays for memoisation,
and, as usual, have to provide the domain for memoisation explicitly:

\begin{code}
acat_idmor_defaultFM :: (Ord obj, Eq atom) =>
                       ACat obj atom -> FiniteMap obj x -> obj -> [atom]
acat_idmor_defaultFM ac dom = memoFMfm' dom (acat_idmor_default ac)
\end{code}

As an abbreviation, we use the whole object list as the domain,
with the ``\verb|M|'' standing for memoisation:

\begin{code}
acat_idmor_defaultM :: (Ord obj, Eq atom) => ACat obj atom -> obj -> [atom]
acat_idmor_defaultM ac =
   let dom = listToFM $ zip (acat_objects ac) (repeat ())
   in acat_idmor_defaultFM ac dom
\end{code}
%}}}

%{{{ \subsection{Building Categories from Atom Category Definitions}
\subsection{Building Categories from Atom Category Definitions}

We introduce an abstract data type
for morphisms built from sets of atoms.

Since in categories,
we need to be able to identify source and target of a morphism,
we have to explicitly include that information here
(even if it was included in atoms, we still would need it for the empty set).

\index{SetMor@{\texttt{SetMor}}}%
\index{unSetMor@{\texttt{unSetMor}}}%
\index{mkSetMor@{\texttt{mkSetMor}}}%
\begin{code}
newtype SetMor obj mor = SetMor (Set mor,obj,obj) deriving (Show, Read)

unSetMor (SetMor t) = t
unSetMor' (SetMor (ms,s,t)) = (toListSet ms, s, t)

mkSetMor a b as = SetMor (listToSet as, a, b)
\end{code}

We have more tools available when morphisms are in \verb|Eq| and \verb|Ord|;
since these instances are not included in the set package we use,
we rely on the (undocumented) feature that \verb|toListSet|
always returns an ordered list of unique elements:

\begin{code}
instance (Eq obj, Ord mor) => Eq (SetMor obj mor) where
  SetMor (as1,s1,t1) == SetMor (as2,s2,t2) =
   s1 == s2 && t1 == t2 && toListSet as1 == toListSet as2
instance (Ord obj, Ord mor) => Ord (SetMor obj mor) where
  SetMor (as1,s1,t1) <= SetMor (as2,s2,t2) =
    (toListSet as1,s1,t1) <= (toListSet as2,s2,t2)
\end{code}

Defining the category is now quite straightforward:

\index{atomsetCat@{\texttt{atomsetCat}}}%
\begin{code}
atomsetCat :: (Eq obj, Ord mor) => ACat obj mor -> Cat obj (SetMor obj mor)
atomsetCat ac = Cat
  {cat_isObj   = acat_isObj ac
  ,cat_isMor   = (\ s t (SetMor (as,s',t')) ->
                    s == s' && t == t' &&
                    foldSet (\ m b -> acat_isAtom ac s t m && b) True as)
  ,cat_objects = acat_objects ac
  ,cat_homset  = (\ a b -> let atoms = acat_atomset ac a b
                           in map (mkSetMor a b) (power atoms))
  ,cat_source  = (\ (SetMor (as,s,t)) -> s)
  ,cat_target  = (\ (SetMor (as,s,t)) -> t)
  ,cat_idmor   = (\ a -> mkSetMor a a $ acat_idmor ac a)
  ,cat_comp    = (\ (SetMor (as1,s1,t1)) (SetMor (as2,s2,t2)) ->
     if t1 /= s2 then error "atomsetCat.comp type error" else
     SetMor (foldSet (\ a1 s ->
                        foldSet (\ a2 s ->
                        	   foldr addToSet s (acat_comp ac s1 s2 t2 a1 a2)
                        	) s as2
                     ) zeroSet as1
            ,s1,t2))
  }
\end{code}

The auxiliary function \verb|power| used to generate homsets
again uses function composition instead of list concatenation for efficiency
and may be found in \sectref{ExtPrel}.

%}}}

%{{{ \subsection{Atom Category Definition Testing}
\subsection{Atom Category Definition Testing}

The above definition of \verb|atomsetCat|
shows how we can directly test well-definedness of atom category definitions;
we group the tests in the following way:
\begin{enumerate}
\item One object: Consistency of object list, and of identity as atom set
\item Two objects:
  \begin{enumerate}
  \item Two objects, one atom: Consistency of atom sets, left-identity
  \item Two objects, one atom in the other direction: Right identity
  \end{enumerate}
\item Three objects, two atoms: Check whether composition yields consistent
atom set
\item Four objects, three atoms: Associativity of composition
\end{enumerate}

\index{acat_TEST@{\texttt{acat\_TEST}}}%
%{{{ acat_TEST
\begin{code}
acat_TEST :: (Eq obj, Ord atom)  => Test ACat obj atom
acat_TEST c =
  let isObj   = acat_isObj   c
      isAtom  = acat_isAtom  c
      objects = acat_objects c
      atomset = acat_atomset c
      idmor   = acat_idmor   c
      comp    = acat_comp    c
  in
  ffold (do o1 <- objects
            testX (isObj o1) [o1] [] "object list contains non-object"
             (do let i1 = idmor o1
                 test (all (isAtom o1 o1) i1) [o1] i1
                      "identity contains non-atoms" : do
                   o2 <- objects
                   (do f <- atomset o1 o2
                       testX (isAtom o1 o2 f) [o1,o2] [f]
                             "atomset contains non-atom"
                        (let f' = nub $ concat $ do i1a <- i1
                                                    return $ comp o1 o1 o2 i1a f
                         in [test ([f] == f') [o1,o2] (i1++f:f')
                                  "left-identity violated"]
                        )
                    ) ++
                    (do g <- atomset o2 o1
                        let g' = nub $ concat $ do i1a <- i1
                                                   return $ comp o2 o1 o1 g i1a
                        [test ([g] == g') [o2,o1] (i1 ++ g:g')
                              "right-identity violated"]
                    )
             )
        ) .
  ffold (do o1 <- objects
            o2 <- objects
            f <- atomset o1 o2
            o3 <- objects
            g <- atomset o2 o3
            let fg = comp o1 o2 o3 f g
            testX (all (isAtom o1 o3) fg) [o1,o2,o3] (f:g:fg)
                  "composition yields non-atom"
             (do o4 <- objects
                 let os = [o1,o2,o3,o4]
                 h <- atomset o3 o4
                 let gh = comp o2 o3 o4 g h
                 let k1 = sort $ nub (gh >>= comp o1 o2 o4 f)
                 let k2 = sort $ nub (fg >>= flip (comp o1 o3 o4) h)
                 [test (k1 == k2) os [f,g,h] "composition is not associative"]
             )
         )
\end{code}
%}}}

Modularising the test for better readability incurs a runtime cost
of about two percent:

%{{{ acat_TEST'
\begin{code}
acat_TEST' :: (Eq obj, Ord atom)  => Test ACat obj atom
acat_TEST' c =
  let isObj   = acat_isObj   c
      isAtom  = acat_isAtom  c
      objects = acat_objects c
      atomset = acat_atomset c
      idmor   = acat_idmor   c
      comp    = acat_comp    c
  in
  ffold (do o1 <- objects
            [test (isObj o1) [o1] [] "object list contains non-object"]
        ) .
  ffold (do o1 <- objects
            let i1 = idmor o1
            [test (all (isAtom o1 o1) i1) [o1] i1 "identity contains non-atoms"]
        ) .
  ffold (do o1 <- objects
            let i1 = idmor o1
            o2 <- objects
            f <- atomset o1 o2
            [test (isAtom o1 o2 f) [o1,o2] [f] "atomset contains non-atom"]
        ) .
  ffold (do o1 <- objects
            let i1 = idmor o1
            o2 <- objects
            let os = [o1,o2]
            let i2 = idmor o2
            f <- atomset o1 o2
            let f'  = nub $ concat $ do i1a <- i1
                                        return $ comp o1 o1 o2 i1a f
            let f'' = nub $ concat $ do i2a <- i2
                                        return $ comp o1 o2 o2 f i2a
            [test ([f] == f' ) os (i1 ++ f:f' ) "left-identity violated" .
             test ([f] == f'') os (i2 ++ f:f'') "right-identity violated"]
        ) .
  ffold (do o1 <- objects
            o2 <- objects
            f <- atomset o1 o2
            o3 <- objects
            g <- atomset o2 o3
            let fg = comp o1 o2 o3 f g
            [test (all (isAtom o1 o3) fg) [o1,o2,o3] (f:g:fg)
                  "composition yields non-atom"]
        ) .
  ffold (do o1 <- objects
            o2 <- objects
            f <- atomset o1 o2
            o3 <- objects
            g <- atomset o2 o3
            let fg = comp o1 o2 o3 f g
            o4 <- objects
            let os = [o1,o2,o3,o4]
            h <- atomset o3 o4
            let gh = comp o2 o3 o4 g h
            let k1 = sort $ nub (gh >>= comp o1 o2 o4 f)
            let k2 = sort $ nub (fg >>= flip (comp o1 o3 o4) h)
            [test (k1 == k2) os [f,g,h] "composition is not associative"]
         )
\end{code}
%}}}
%}}}

%{{{ \subsection{From Allegories to Relation Algebras}
\subsection{From Allegories to Relation Algebras}

With the atom set category definitions from above,
we already have homsets that are atomic complete Boolean lattices,
and (sub-)distributivity of composition over join and meet.
However, we do not yet have even an allegory,
because information about converse is still missing ---
note that the converse of an atom has to be an atom again
because of monotony of converse:

\index{AAll@{\texttt{AAll}}}%
\index{aall*@{\texttt{aall\_*}}}%
\begin{code}
data AAll obj atom = AAll
  {aall_acat :: ACat obj atom
  ,aall_converse :: obj -> obj -> atom -> atom
  }
\end{code}

We expand the interface
to comprise that of the included atom category definition:

\begin{code}
aall_isObj   = acat_isObj   . aall_acat  -- :: obj -> Bool
aall_isAtom  = acat_isAtom  . aall_acat  -- :: obj -> atom -> Bool
aall_objects = acat_objects . aall_acat  -- :: [obj]
aall_atomset = acat_atomset . aall_acat  -- :: obj -> obj -> [atom]
aall_idmor   = acat_idmor   . aall_acat  -- :: obj -> [atom]
aall_comp    = acat_comp    . aall_acat  -- :: o -> o -> o -> at -> at -> [at]
\end{code}

An allegory is easily constructed:

\index{atomsetAll@{\texttt{atomsetAll}}}%
\begin{code}
atomsetAll :: (Eq obj, Ord mor) => AAll obj mor -> All obj (SetMor obj mor)
atomsetAll aa = let ac = aall_acat aa
 in All
  {all_cat = atomsetCat ac
  ,all_converse = (\ (SetMor (as,s,t)) ->
                     SetMor (foldSet (addToSet . aall_converse aa s t) zeroSet as
                            ,t,s))
  ,all_meet = (\ (SetMor (as1,s1,t1)) (SetMor (as2,s2,t2)) ->
                 if s1 /= s2 then error "atomsetAll.meet source type error" else
                 if t1 /= t2 then error "atomsetAll.meet target type error" else
                 SetMor (intersectSet as1 as2, s1, t1))
  ,all_incl = (\ (SetMor (as1,s1,t1)) (SetMor (as2,s2,t2)) ->
                 if s1 /= s2 then error "atomsetAll.incl source type error" else
                 if t1 /= t2 then error "atomsetAll.incl target type error" else
                 isZeroSet (diffSet as1 as2))
  }
\end{code}

Also for distributive allegories everything is straightforward:

\index{atomsetDistrAll@{\texttt{atomsetDistrAll}}}%
\begin{code}
atomsetDistrAll :: (Eq obj, Ord mor) => AAll obj mor ->
                                        DistrAll obj (SetMor obj mor)
atomsetDistrAll aa = DistrAll
  {distrAll_all = atomsetAll aa
  ,distrAll_bottom = (\ a b -> SetMor (zeroSet, a, b))
  ,distrAll_join = (\ (SetMor (as1,s1,t1)) (SetMor (as2,s2,t2)) ->
            if s1 /= s2 then error "atomsetDistrAll.join source type error" else
            if t1 /= t2 then error "atomsetDistrAll.join target type error" else
            SetMor (joinSet as1 as2, s1, t1))
  ,distrAll_atomset = (\ a b -> map (atmor a b) $ aall_atomset aa a b)
  ,distrAll_atoms = (\ (SetMor (as,a,b)) -> map (atmor a b) $ toListSet as)
  }

atmor a b at = SetMor (unitSet at, a, b)
\end{code}

For division allegories we use a little trick:
We know that we already have a relation algebra,
so we use the default residual definitions of that relation algebra
for division allegories,
although, at least formally, that relation algebra is defined in terms of this division allegory.
Since there is however no harmful cyclic dependency between the
record components involved, everything is well-defined and we do not drop into a ``black hole'':

\index{atomsetDivAll@{\texttt{atomsetDivAll}}}%
\begin{code}
atomsetDivAll :: (Eq obj, Ord mor) => AAll obj mor -> DivAll obj (SetMor obj mor)
atomsetDivAll aa = da where
 da = DivAll
  {divAll_distrAll = atomsetDistrAll aa
  ,divAll_rres = ra_rres_default ra
  ,divAll_lres = ra_lres_default ra
  ,divAll_syq  = divAll_syq_default da
  }
 ra = atomsetRA aa
\end{code}

For Dedekind categories and relation algebras there are no further problems:

\index{atomsetDed@{\texttt{atomsetDed}}}%
\index{atomsetRA@{\texttt{atomsetRA}}}%
\begin{code}
atomsetDed :: (Eq obj, Ord mor) => AAll obj mor -> Ded obj (SetMor obj mor)
atomsetDed aa = Ded
  {ded_divAll = atomsetDivAll aa
  ,ded_top    = (\ a b -> mkSetMor a b (aall_atomset aa a b))
  }

atomsetRA :: (Eq obj, Ord mor) => AAll obj mor -> RA obj (SetMor obj mor)
atomsetRA aa = RA
  {ra_ded   = atomsetDed aa
  ,ra_compl = (\ (SetMor (as,s,t)) ->
                  SetMor (listToSet (filter (\ a -> not (a `elemSet` as))
                                            (aall_atomset aa s t))
                         , s, t))
  }
\end{code}
%}}}

%{{{ \subsection{Atom Allegory Definition for $\mathbb{B}$}
\subsection{Atom Allegory Definition for $\mathbb{B}$}

Just for testing,
we provide the second simplest atom allegory definition that is possible:

\index{acatB@{\texttt{acatB}}}%
\index{aallB@{\texttt{aallB}}}%
\begin{code}
acatB :: ACat () ()
acatB = ACat
  {acat_isObj   = const True
  ,acat_isAtom  = (\ s t a -> True)
  ,acat_objects = [()]
  ,acat_atomset = const $ const [()]
  ,acat_idmor   = const [()]
  ,acat_comp    = (\ a b c f g -> [()])
  }

aallB :: AAll () ()
aallB = AAll
  {aall_acat = acatB
  ,aall_converse = const $ const id
  }
\end{code}

With this definition, \verb|atomsetRA aallB| is isomorphic to \verb|raB|;
we leave the definition of the functors as an exercise to the reader
(see also \subsectref{AAllOutput} and \subsectref{matBtoAtCat})
%}}}

%{{{ \subsection{Atom Allegory Definition Testing}
\subsection{Atom Allegory Definition Testing}

It is easy to check that,
in distributive allegories, the Dedekind formula for
$P: \objA\rel\objC$, $Q:\objA\rel\objB$, and $R: \objB \rel \objC$:
\BDA
    P \reland Q\rcmp R
   \rsubs
    (Q \reland P\rcmp\rtrans{R})\rcmp(R\reland\rtrans{Q}\rcmp P)
\EDA
follows from any of the following:
\begin{itemize}
\item $P = P_1 \relor P_2$ and the Dedekind formulae for $P_1,Q,R$
  and for $P_2,Q,R$ hold, or
\item $Q = Q_1 \relor Q_2$ and the Dedekind formulae for $P,Q_1,R$
  and for $P,Q_2,R$ hold, or
\item $R = R_1 \relor R_2$ and the Dedekind formulae for $P,Q,R_1$
  and for $P,Q,R_2$ hold.
\end{itemize}

Therefore it is sufficient to check the Dedekind formula for all atoms,
and we organise the full atom allegory definition test as follows:
\begin{enumerate}
\item One object: Preservation of identities by converse
\item Two objects, one atom: Consistency of result atom of converse,
  involution test
\item Three objects, two atoms: Preservation of composition by converse
\item Three objects, three atoms: Dedekind rule
\end{enumerate}

\index{aall_TEST@{\texttt{aall\_TEST}}}%
\begin{code}
aall_TEST :: (Eq obj, Ord atom)  => Test AAll obj atom
aall_TEST c =
  let isAtom  = aall_isAtom   c
      objects = aall_objects  c
      atomset = aall_atomset  c
      idmor   = aall_idmor    c
      comp    = aall_comp     c
      conv    = aall_converse c
      ameet a l = if a `elem` l then [a] else []
  in
  ffold $ do
    o1 <- objects
    let i1 = idmor o1
    test (all (\ i -> conv o1 o1 i == i) i1) [o1] i1
         "converse does not preserve identity" : do
     o2 <- objects
     q <- atomset o1 o2
     let qC = conv o1 o2 q
     let qCC = conv o2 o1 qC
     (testX (isAtom o2 o1 qC) [o1,o2] [q,qC] "converse yields non-atom" .
      testX (qCC == q) [o1,o2] [q,qC,qCC] "converse not involutory"
      )
      (do
       o3 <- objects
       let os = [o1,o2,o3]
       r <- atomset o2 o3
       let rC = conv o2 o3 r
       let qrC = sort $ nub $ map (conv o1 o3) (comp o1 o2 o3 q r)
       let rCqC = sort $ nub $ comp o3 o2 o1 rC qC
       test (qrC == rCqC) os (q : r : qrC ++ rCqC) "non-functorial converse" : do
         p <- atomset o1 o3
         let p' = p `ameet` (comp o1 o2 o3 q  r )
         let q' = q `ameet` (comp o1 o3 o2 p  rC)
         let r' = r `ameet` (comp o2 o1 o3 qC p )
         let qr' = do qa <- q'; ra <- r'; comp o1 o2 o3 qa ra
         [test (all (`elem` qr') p') os ([p,q,r] ++ qr') "Dedekind violation"]
      )
\end{code}
%}}}

%{{{ \subsection{Atom Allegory Definition Output}
\subsection{Atom Allegory Definition Output}\subsectlabel{AAllOutput}

Once we constructed a relation algebra, we may want
to output its definition in a directly reusable form.
As an example, consider algebras like \verb|raMat raB [[],[()],[(),()]]|,
which (essentially)
have the given lists of unit values as objects
and small matrices of Booleans as morphisms.
We might want to generate dedicated object and atom data types like

\begin{pseudocode}
-- data Obj = P0 | P1 | P2 deriving (Eq,Ord,Show)
-- data Atom = At1 | At2 | At3 | At4 deriving (Eq,Ord,Show)
\end{pseudocode}

together with the spelled-out definitions of the translated
atom category definition.

Now the original atom category definition is given by the following expression:

\begin{pseudocode}
-- ra_acat (raMat raB [[],[()],[(),()]]) :: ACat (Vec ()) (MatMor () Bool)
\end{pseudocode}

Its object and morphism types are already instances of the class \verb|Show|,
so we cannot rely on the functions provided by this class,
but have to explicitly provide the corresponding output functions.

For efficiency, we always use functions of the prelude type \verb|ShowS|.

For enabling to generate such definitions also by other means,
we generally also provide intermediate functions that do not expect
a full atom category definition.

\medskip
%{{{ atomset
The first component we need is the mapping from pairs of objects to atomsets.
We introduce a default definition with the empty atomset as result,
so we need not explicitly output those mappings
that do have the empty atomset as result:

\index{showsAtomset@{\texttt{showsAtomset*}}}%
%{{{ showsAtomset0
\begin{code}
showsAtomset0 :: ShowS -> (obj -> ShowS) -> (atom -> ShowS) ->
  [obj] -> (obj -> obj -> [atom]) -> ShowS
showsAtomset0 indent so sa objects atomset = ffold (do
   x <- objects
   y <- objects
   case atomset x y of
    [] -> []
    atoms -> [indent . ("atomset " ++) . so x . (' ' :) . so y . (" = " ++) .
                                  listShows sa atoms . ('\n' :)]
  ) . indent . ("atomset _ _ = []\n" ++)
\end{code}
%}}}

%{{{ showsAtomset', showsAtomset
\begin{code}
showsAtomset' :: ShowS -> (obj -> ShowS) -> (atom -> ShowS) 
                       -> ACat obj atom -> ShowS
showsAtomset' indent so sa ac =
  showsAtomset0 indent so sa (acat_objects ac) (acat_atomset ac)

showsAtomset :: (Show obj, Show atom) => ShowS -> ACat obj atom -> ShowS
showsAtomset indent ac = showsAtomset' indent shows shows ac
\end{code}
%}}}
%}}}

%{{{ atComp
Since we may generate atom composition table output not only directly from
a given atom category definition,
but also from cycle representations (see below),
we provide separate access to the basic output functions:

%{{{ showsAtCompEntry0, showsAtCompEntry1, showsAtCompDefault
\begin{code}
showsAtCompEntry0 :: (at -> ShowS) -> ShowS -> at -> at -> [at] -> ShowS
showsAtCompEntry0 sa prefix a b c =
  prefix . sa a . (' ' :) . sa b . (" = " ++) . listShows sa c . ('\n' :)

showsAtCompDefault :: ShowS
showsAtCompDefault = ("atComp _ _ _  _ _ = []\n" ++)


showsAtCompEntry1 :: (obj -> ShowS) -> (atom -> ShowS) ->
                     obj -> obj -> obj -> atom -> atom -> [atom] -> ShowS
showsAtCompEntry1 so sa x y z a b c =
   showsAtCompEntry0 sa (atCompPrefix so x y z) a b c

atCompPrefix so x y z =
  ("atComp " ++) . listShowsSep so ' ' [x,y,z] . (' ' :)
\end{code}
%}}}

Normally, we assume the equivalents of \verb|acat_objects|,
\verb|acat_atomset| and \verb|acat_comp| to be available:

\index{showsAtComp@{\texttt{showsAtComp*}}}%
%{{{ showsAtComp0
\begin{code}
showsAtComp0 :: ShowS -> (obj -> ShowS) -> (atom -> ShowS) ->
  [obj] -> (obj -> obj -> [atom]) ->
  (obj -> obj -> obj -> atom -> atom -> [atom]) -> ShowS
showsAtComp0 indent so sa objects atomset comp = ffold (do
   x <- objects
   y <- objects
   z <- objects
   let prefix = indent . atCompPrefix so x y z
   let cmp = comp x y z
   a <- atomset x y
   b <- atomset y z
   case cmp a b of [] -> []
                   c  -> [showsAtCompEntry0 sa prefix a b c]
  ) . indent . showsAtCompDefault
\end{code}
%}}}

Usually these are indeed taken from an atom category definition,
and we also provide a variant that uses existing \verb|Show| instances:

%{{{ showsAtComp', showsAtComp
\begin{code}
showsAtComp' :: ShowS -> (obj -> ShowS) -> (atom -> ShowS) 
                      -> ACat obj atom -> ShowS
showsAtComp' indent so sa ac =
  showsAtComp0 indent so sa (acat_objects ac) (acat_atomset ac) (acat_comp ac)

showsAtComp :: (Show obj, Show atom) => ShowS -> ACat obj atom -> ShowS
showsAtComp indent ac = showsAtComp' indent shows shows ac
\end{code}
%}}}
%}}}

It is essentially the same story for the identity morphism:

%{{{ idmor
\index{showsIdmor@{\texttt{showsIdmor*}}}%
%{{{ showsIdmor0
\begin{code}
showsIdmor0 :: ShowS -> (obj -> ShowS) -> (atom -> ShowS) ->
  [obj] -> (obj -> [atom]) -> ShowS
showsIdmor0 indent so sa objects idmor = ffold (do
   x <- objects
   case idmor x of
    [] -> []
    atoms -> [indent . ("idmor " ++) . so x . (" = " ++) .
                                  listShows sa atoms . ('\n' :)]
  ) . indent . ("idmor _ = []\n" ++)
\end{code}
%}}}

%{{{ showsIdmor', showsIdmor
\begin{code}
showsIdmor' :: ShowS -> (obj -> ShowS) -> (atom -> ShowS) 
                                       -> ACat obj atom -> ShowS
showsIdmor' indent so sa ac =
  showsIdmor0 indent so sa (acat_objects ac) (acat_idmor ac)

showsIdmor :: (Show obj, Show atom) => ShowS -> ACat obj atom -> ShowS
showsIdmor indent ac = showsIdmor' indent shows shows ac
\end{code}
%}}}
%}}}

%{{{ showsACat0
All these together are now used to output a complete atom category definition
with the components defined locally in a \verb|where| clause:

\index{showsACat@{\texttt{showsACat*}}}%
\begin{code}
showsACat0 :: String -> ShowS -> (obj -> ShowS) -> (atom -> ShowS) ->
  [obj] -> (obj -> obj -> [atom]) ->
  (obj -> obj -> obj -> atom -> atom -> [atom]) -> (obj -> [atom]) -> ShowS
showsACat0 name indent so sa objects atomset comp idmor =
 let indent' = indent . ("  " ++) in
  indent . ("aCat_" ++) . (name ++) . (" = ACat\n" ++) .
  indent . ("  {acat_isObj   = (`elem` objects)\n" ++) .
  indent . ("  ,acat_isAtom  = (\\ s t a -> a `elem` atomset s t)\n" ++) .
  indent . ("  ,acat_objects = objects\n" ++) .
  indent . ("  ,acat_atomset = atomset\n" ++) .
  indent . ("  ,acat_idmor   = idmor\n" ++) .
  indent . ("  ,acat_comp    = atComp\n" ++) .
  indent . ("  }\n\n" ++) .
  indent . (" where\n" ++) .
  indent' . ("objects = " ++) . listShows so objects . ("\n\n" ++) .
  showsAtomset0 indent' so sa objects atomset . ('\n' :) .
  showsAtComp0 indent' so sa objects atomset comp .('\n' :) .
  showsIdmor0 indent' so sa objects idmor
\end{code}
%}}}

%{{{ showsACat', showsACat
\begin{code}
showsACat' :: String -> ShowS -> (obj -> ShowS) -> (atom -> ShowS) 
                     -> ACat obj atom -> ShowS
showsACat' name indent so sa ac =
  showsACat0 name indent so sa (acat_objects ac) (acat_atomset ac) 
                               (acat_comp ac)    (acat_idmor ac)

showsACat :: (Show obj, Show atom) => String -> ShowS -> ACat obj atom -> ShowS
showsACat name indent ac = showsACat' name indent shows shows ac
\end{code}
%}}}

%{{{ Converse
%{{{ showsConv0
For the converse table, we collect identical mappings into the default case.
For this purpose we should not compare original atomic morphisms,
but their string representation,
since usually the same atom output name may occur in different atom sets,
but atomic morphisms from different homsets are always different.

\begin{code}
showsConv0 :: ShowS -> (obj -> ShowS) -> (atom -> ShowS) ->
  [obj] -> (obj -> obj -> [atom]) ->
  (obj -> obj -> atom -> atom) -> ShowS
showsConv0 indent so sa objects atomset conv = ffold (do
   x <- objects
   y <- objects
   let cnv = conv x y
   a <- atomset x y
   let c = cnv a
   if sa c "" == sa a ""
    then []
    else [indent . ("conv " ++) . so x . (' ' :) . so y . (' ' :) .
                           sa a . (" = " ++) . sa c . ('\n' :)]
  ) . indent . ("conv _ _  x = x\n" ++)
\end{code}
%}}}

So far, the code might just as well be used for allegories,
since the type of converse there is the same as in atom allegory descriptions.

But now we provide direct output
only for the converse functions of atom allegory descriptions:

\index{showsAtConv@{\texttt{showsAtConv*}}}%
%{{{ showsAtConv', showsAtConv
\begin{code}
showsAtConv' :: ShowS -> (obj -> ShowS) -> (atom -> ShowS) ->
                AAll obj atom -> ShowS
showsAtConv' indent so sa aa =
  showsConv0 indent so sa (aall_objects aa) (aall_atomset aa) (aall_converse aa)

showsAtConv :: (Show obj, Show atom) => ShowS -> AAll obj atom -> ShowS
showsAtConv indent aa = showsAtConv' indent shows shows aa
\end{code}
%}}}
%}}}

%{{{ showsAAll0
When writing an atom allegory definition,
we first output the atom category definition contained within it,
and then put the \verb|AAll| definition on the same level,
again with the converse table as a local definition:

\index{showsAAll@{\texttt{showsAAll*}}}%
\begin{code}
showsAAll0 :: String -> ShowS -> (obj -> ShowS) -> (atom -> ShowS) ->
  [obj] -> (obj -> obj -> [atom]) ->
  (obj -> obj -> obj -> atom -> atom -> [atom]) -> (obj -> [atom]) ->
  (obj -> obj -> atom -> atom) -> ShowS
showsAAll0 name indent so sa objects atomset comp idmor conv =
 let indent' = indent . ("  " ++) in
  showsACat0 name indent so sa objects atomset comp idmor . ('\n' :) .
  indent . ("aAll_" ++) . (name ++) . (" = AAll\n" ++) .
  indent . ("  {aall_acat    = aCat_" ++) . (name ++) . ('\n' :) .
  indent . ("  ,aall_converse = conv\n" ++) .
  indent . ("  }\n\n" ++) .
  indent . (" where\n" ++) .
  showsConv0 indent' so sa objects atomset conv
\end{code}
%}}}

%{{{ showsAAll', showsAAll
\begin{code}
showsAAll' :: String -> ShowS -> (obj -> ShowS) -> (atom -> ShowS) ->
              AAll obj atom -> ShowS
showsAAll' name indent so sa ac =
  showsAAll0 name indent so sa (aall_objects ac) (aall_atomset ac)
             (aall_comp ac) (aall_idmor ac) (aall_converse ac)

showsAAll :: (Show obj, Show atom) => String -> ShowS -> AAll obj atom -> ShowS
showsAAll name indent ac = showsAAll' name indent shows shows ac
\end{code}
%}}}

%{{{ showsARA0
We round this off with functions that in addition output
the definition of the atom set relation algebra
on the same level as the other two definitions:

\index{showsARA@{\texttt{showsARA*}}}%
\begin{code}
showsARA0 :: String -> ShowS -> (obj -> ShowS) -> (atom -> ShowS) ->
  [obj] -> (obj -> obj -> [atom]) ->
  (obj -> obj -> obj -> atom -> atom -> [atom]) -> (obj -> [atom]) ->
  (obj -> obj -> atom -> atom) -> ShowS
showsARA0 name indent so sa objects atomset comp idmor conv =
 let indent' = indent . ("  " ++) in
  showsAAll0 name indent so sa objects atomset comp idmor conv . ('\n' :) .
  indent . ("ra_" ++) . (name ++) .
           (" = atomsetRA aAll_" ++) . (name ++) . ('\n' :)
\end{code}
%}}}

%{{{ showsARA', showsARA
\begin{code}
showsARA' :: String -> ShowS -> (obj -> ShowS) -> (atom -> ShowS) ->
             AAll obj atom -> ShowS
showsARA' name indent so sa ac =
  showsARA0 name indent so sa (aall_objects ac) (aall_atomset ac)
            (aall_comp ac) (aall_idmor ac) (aall_converse ac)

showsARA :: (Show obj, Show atom) => String -> ShowS -> AAll obj atom -> ShowS
showsARA name indent ac = showsARA' name indent shows shows ac
\end{code}
%}}}
%}}}

%{{{ \subsection{Generating Atom Set Definitions for Boolean Matrix Algebras}
\subsection{Generating Atom Set Definitions for Boolean Matrix Algebras}
\subsectlabel{writeBoolMatARA}

%{{{ For experiments ... boolMatAtCompSchows
For experiments, we want to output atom descriptions for
algebras of Boolean matrices --- if a Boolean matrix is an atom,
we can compute its ordinal number in a natural ordering of
atomic matrices of this shape with the following function:

\begin{code}
boolMatAtomPos :: [[Bool]] -> Int
boolMatAtomPos = fst . head . filter snd . zip [1..] . concat
\end{code}

This allows us to define \verb|shows| functions for objects and atoms
of algebras in the range of \verb|distrAllMat distrAllB|:

\begin{code}
boolMatAtomName i = "At" ++ show i
boolMatAtomShows mm = let (m,_,_) = unMatMor mm
                      in ((boolMatAtomName $ boolMatAtomPos m) ++)

boolMatObjShows obj = ('P' :) . shows (length $ unVec obj)
\end{code}

The following function displays the composition tables of
atom category definitions for algebras of Boolean matrices:

\begin{code}
boolMatAtCompSchows indent objs =
   showsAtComp' indent boolMatObjShows boolMatAtomShows $
   distrAll_acat $ distrAllMat distrAllB objs
\end{code}

For example, the following invocation prints the composition table
of Boolean $1\times 1$, $1\times 2$, $2\times 1$, and $2\times 2$ matrices:

\begin{session}
putStr $ boolMatAtCompSchows id [[()],[(),()]] ""
\end{session}

Whole algebras can be printed with the following:

\index{boolMatACatSchows@{\texttt{boolMatACatSchows}}}%
\index{boolMatAAllSchows@{\texttt{boolMatAAllSchows}}}%
\begin{code}
boolMatACatSchows :: String -> ShowS -> [[()]] -> ShowS
boolMatACatSchows name indent objs =
   showsACat' name indent boolMatObjShows boolMatAtomShows $
   distrAll_acat $ distrAllMat distrAllB objs

boolMatAAllSchows :: String -> ShowS -> [[()]] -> ShowS
boolMatAAllSchows name indent objs =
   showsAAll' name indent boolMatObjShows boolMatAtomShows $
   distrAll_aall $ distrAllMat distrAllB objs
\end{code}

When writing whole relation algebras of this form,
we also include the allegory representations between
the newly generated atom set relation algebra
and the corresponding Boolean matrix relation algebra
(see \subsectref{matBtoAtCat}),
including also an equivalence test.
For this purpose we also generate definitions of a few local
auxiliary functions, which are, however, not exported from the resulting module.

\index{boolMatARASchows@{\texttt{boolMatARASchows}}}%
\index{atMat@{\texttt{atMat}}}%
\begin{code}
boolMatARASchows :: String -> ShowS -> [[()]] -> ShowS
boolMatARASchows name indent objlist =
   indent . ("module " ++) . (name ++) .
   ('(' :) . listShowsSep (++) ',' (map (++ name)
               ["aCat_","aAll_","ra_","matBtoAtCat_","atCatToMatB_", "atMat_",
                "raB_","allB_","test_for_equivalence_"]) .
   (") where\n\n" ++) .
   indent . ("import RelAlg \n\n" ++) .
   indent . ("import Matrix \n\n" ++) .
   indent . ("import Atomset \n\n" ++) .
   indent . mkdata ("Obj" ++ name) (map (flip boolMatObjShows "") objects) .
   indent . mkdata ("Atom" ++ name)
     (map (flip boolMatAtomShows "") $
          (do x <- objects; y <- objects; aall_atomset aall x y)) .
   showsARA' name indent boolMatObjShows boolMatAtomShows aall .
   ('\n' :) .
   ((do x <- objs
        indent "vecToObj " ++ show x ++ " = "
                           ++ boolMatObjShows (vec x) "\n") ++) .
   ('\n' :) .
   ((do x <- objs
        indent "objToVec " ++ boolMatObjShows (vec x) (" = "
                           ++ show x ++ "\n")) ++)
   . ('\n' :) .
   ffold (do x <- objs
             let lx = length x
             y <- objs
             let ly = length y
             [indent . ("atMat_" ++) . (name ++) . (' ' :) .
                       boolMatObjShows (vec x) . (' ' :) .
                       boolMatObjShows (vec y) . (" = " ++) .
              listShows (listShows ((++) . boolMatAtomName))
                        (take lx $ unfold (splitAt ly) [1..]) . ('\n' :)]
         ) . ('\n' :) .
   indent . ("matBtoAtCat_" ++) . (name ++) .
            (" = matBtoAtCat vecToObj atMat_" ++) . (name ++) . ("\n\n" ++) .
   indent . ("atCatToMatB_" ++) . (name ++) .
            (" = atCatToMatB objToVec atMat_" ++) . (name ++) . ("\n\n" ++) .
   indent . (("raB_" ++ name ++ " = raMat raB " ++ show objs) ++) .
   ('\n' :) .
   indent . (("allB_" ++ name ++ " = ra_all raB_" ++ name) ++) .
   ("\n\n" ++) .
   indent . (("test_for_equivalence_" ++ name ++ " =\n" ++
              indent " all_equiv_perform allB_" ++ name ++
              "(ra_all ra_" ++ name ++ ") matBtoAtCat_" ++ name ++
              " atCatToMatB_" ++ name) ++) . ('\n' :)
 where mkdata name cs = ("data " ++) . (name ++) . (" = " ++) .
                        ((foldr1 (\l r -> l ++ " | " ++ r)
                                 (foldr insertSet [] cs)) ++) .
                        (" deriving (Eq, Ord, Show)\n\n" ++)
       aall = distrAll_aall $ distrAllMat distrAllB objs
       objects = aall_objects aall
       objs = foldr insertSet [] objlist
\end{code}

Finally, we wrap this into a function that generates a
file containing a literate Haskell module,
adhering to the usual naming convention:

\index{writeBoolMatARA@{\texttt{writeBoolMatARA*}}}%
\begin{code}
writeBoolMatARA' :: String -> [[()]] -> IO ()
writeBoolMatARA' name objs =
  writeFile (name ++ ".lhs")
  (("This file has been automatically generated.\n\n" ++) $
   ("It contains a description of a Boolean matrix relation algebra\n" ++) $
   ("expressed in terms of its atoms.\n\n\n" ++) $
   boolMatARASchows name ("> " ++) objs ""
  )

writeBoolMatARA :: String -> [Int] -> IO ()
writeBoolMatARA name objs =
   writeBoolMatARA' name $ map (flip replicate ()) $ filter (0 <=) objs
\end{code}

Generating an atom definition of a matrix algebra
now boils down to typing the following command in Hugs:

\begin{session}
-- writeBoolMatARA "Q012" [0,1,2]
\end{session}

The resulting Haskell source file ``\verb|Q012.lhs|''
can then immediately be used.
%}}}
%}}}

%{{{ \subsection{Cycles}\subsectlabel{Cycles}
\subsection{Cycles}\subsectlabel{Cycles}

A {\it cycle}\index{cycle} is a triple of atoms {\tt at1, at2, at3} such that
{\tt at3} occurs in {\tt at1 `comp` at2}. This will then
mean that also 
   
{\tt at2} occurs in {\tt (conv at1) `comp` at3}

{\tt  (conv at1) } occurs in {\tt at2 `comp` (conv at3)}

{\tt  (conv at3) } occurs in {\tt (conv at2) `comp` (conv at1)}

{\tt  (conv at2) } occurs in {\tt (conv at3) `comp` at1}

{\tt at1} occurs in {\tt at3 `comp` (conv at2)}

\bigskip
\noindent
This is easily proved starting from the assumption that 
this might not be true and using the properties of atoms.

As this reduces the number of composition table entries,
it may sometimes be used to shorten atom composition definitions.

\medskip
For our \texttt{Cycle} data-type,
we include all three involved objects along with the three atoms:

\index{Cycle@{\texttt{Cycle}}}%
\begin{code}
type Cycle obj atom = ((obj,obj,obj),(atom,atom,atom))
\end{code}

In order to determine the list of cycles, we first generate
all triples of atoms and transposed atoms with the respective composition
property in a list and then cancel them in groups of (at most) 6.

\index{cycles@{\texttt{cycles}}}%
\index{cycleRepresentatives@{\texttt{cycleRepresentatives}}}%
\begin{code}
compTriples aa = let
    objects = aall_objects aa
    atoms = aall_atomset aa
    atComp = aall_comp aa
 in [ ((x, y, z), (a, b, c))  | 
            x <- objects,    y <- objects,    z <- objects,
            a <- atoms x y,  b <- atoms y z,  c <- atComp x y z a b ]


cycles aa = let conv = aall_converse aa
      in nub [ ((x, y, z), sort [(a, b, c), 
                                 (conv x y a, c, b),  
                                 (b, conv x z c, conv x y a),
                                 (conv y z b, conv x y a, conv x z c),
                                 (conv x z c, a, conv y z b),
                                 (c, conv y z b, a)])
             |  ((x, y, z), (a, b, c)) <- compTriples aa
             ]

cycleRepresentatives :: (Ord atom, Eq obj) =>
                        AAll obj atom -> [Cycle obj atom]
cycleRepresentatives aa = map (\ (x, y) -> (x, head y)) $ cycles aa
\end{code}

\noindent
We now recompute the atom composition tables from the cycle representatives.
The first step is to expand a cycle into the six atom triples it represents:

\begin{code}
oneCycle conv ((x, y, z), (a, b, c)) =
   [((x, y, z), (a,          b,          c         )),
    ((y, x, z), (conv x y a, c,          b         )),  
    ((y, z, x), (b,          conv x z c, conv x y a)),   
    ((z, y, x), (conv y z b, conv x y a, conv x z c)),   
    ((z, x, y), (conv x z c, a,          conv y z b)),   
    ((x, z, y), (c,          conv y z b, a         ))
   ]
\end{code}

We use a nested finite map for storing the composition information:

\index{AtomCompTable@{\texttt{AtomCompTable}}}%
\begin{code}
type AtomCompTable obj atom =
     FiniteMap (obj,obj,obj) (FiniteMap (atom,atom) (Set atom))
\end{code}


\index{allCycles@{\texttt{allCycles}}}%
\begin{code}
allCycles :: (Ord obj, Ord atom) => (obj -> obj -> atom -> atom) ->
                                    [Cycle obj atom] -> AtomCompTable obj atom
allCycles conv cycs = foldr addCycle zeroFM $ concatMap (oneCycle conv) cycs

addCycle (objs,ats) fm = let atfm = lookupDftFM fm zeroFM objs
                         in addToFM objs (addCyc ats atfm) fm

addCyc (a,b,c) fm = let
  p = (a,b)
  s = lookupDftFM fm zeroSet p
 in addToFM p (addToSet c s) fm
\end{code}

Using the following functions, such a table can be used
to directly define an atom composition function,
or to define one as the table's complement
wrt.\null{} an atom supply that has to be passed as another argument:

\index{tableAtComp@{\texttt{tableAtComp}}}%
\index{negTableAtComp@{\texttt{negTableAtComp}}}%
\begin{code}
tableAtComp :: (Ord obj, Ord atom) =>
               AtomCompTable obj atom ->
               obj -> obj -> obj -> atom -> atom -> [atom]
tableAtComp atct x y z a b = let
   atfm = lookupDftFM atct zeroFM  (x,y,z)
   cs   = lookupDftFM atfm zeroSet (a,b)
  in toListSet cs

negTableAtComp :: (Ord obj, Ord atom) =>
               (obj -> obj -> [atom]) ->
               AtomCompTable obj atom ->
               obj -> obj -> obj -> atom -> atom -> [atom]
negTableAtComp atomset atct = let
   neg = negAtCompTable atomset atct
 in \ x y z -> let
   atoms = atomset x z
 in \ a b -> case lookupFM neg (x,y,z) of
     Nothing -> atoms
     Just atfm -> case lookupFM atfm (a,b) of
                    Nothing -> atoms
                    Just cs -> toListSet cs

-- not exported, since not independently useable!
negAtCompTable :: (Ord obj, Ord atom) =>
               (obj -> obj -> [atom]) ->
               AtomCompTable obj atom -> AtomCompTable obj atom
negAtCompTable atomset = mapFM (\ (x,y,z) ->
    let atoms = listToSet $ atomset x z
    in mapFM (\ (a,b) cs -> atoms `diffSet` cs))
\end{code}

Finally we present a function that allows to print
an explicit variant of the atom composition table that results
from a cycle list:

\index{showsCycAtComp@{\texttt{showsCycAtComp}}}%
\begin{code}
showsCycAtComp :: (Ord obj, Ord atom) => (obj -> ShowS) -> (atom -> ShowS) ->
  (obj -> obj -> atom -> atom) -> [Cycle obj atom] -> ShowS
showsCycAtComp so sa conv cycs s =
  foldFM (\ (x,y,z) atfm s0 ->
   foldFM (\ (a,b) cs -> showsAtCompEntry1 so sa x y z a b (toListSet cs))
          s0 atfm)
--  foldr (\ (x, y, z, a, b, c) -> showsAtCompEntry1 so sa x y z a b c)
        (showsAtCompDefault s) (allCycles conv cycs)
-- (cycleAtCompTable conv cycs)
\end{code}

%}}}

%{{{ \subsection{Building Atom Category Defs from Distributive Allegories}
\subsection{Building Atom Category Definitions from Distributive Allegories}

We can extract atom category and atom allegory definitions
from a distributive allegory.
Note that this decribes a relation algebra,
but in general only one that is embedded
(as a distributive allegory) in the original distributive allegory.

\index{distrAll_acat@{\texttt{distrAll\_acat}}}%
\index{distrAll_aall@{\texttt{distrAll\_aall}}}%
\begin{code}
distrAll_acat :: (Ord obj,Eq mor) => DistrAll obj mor -> ACat obj mor
distrAll_acat da =
 let objects = distrAll_objects da
     objDom = listToFM (zip objects (repeat ()))
     memoObj = memoFMfm' objDom
     objPairs = do a <- objects; b <- objects; [(a,b)]
     objPairDom = listToFM (zip objPairs (repeat ()))
     memoObjPair = curry . memoFMfm' objPairDom . uncurry
     atoms = distrAll_atoms da
 in ACat
  {acat_isObj   = distrAll_isObj da
  ,acat_isAtom  = distrAll_isAtom da
  ,acat_objects = distrAll_objects da
  ,acat_atomset = memoObjPair (distrAll_atomset da)
  ,acat_idmor   = memoObj (atoms . distrAll_idmor da)
  ,acat_comp    = (\ _ _ _  f g -> atoms (distrAll_comp da f g))
  }

distrAll_aall :: (Ord obj,Eq mor) => DistrAll obj mor -> AAll obj mor
distrAll_aall da = AAll
  {aall_acat = distrAll_acat da
  ,aall_converse = (\ _ _ at -> distrAll_converse da at)
  }
\end{code}

For ease of access, we provide this interface also for the higher structures:

\index{divAll_acat@{\texttt{divAll\_acat}}}%
\index{divAll_aall@{\texttt{divAll\_aall}}}%
\index{ded_acat@{\texttt{ded\_acat}}}%
\index{ded_aall@{\texttt{ded\_aall}}}%
\index{ra_acat@{\texttt{ra\_acat}}}%
\index{ra_aall@{\texttt{ra\_aall}}}%
\begin{code}
divAll_acat :: (Ord obj,Eq mor) => DivAll obj mor -> ACat obj mor
divAll_acat = distrAll_acat . divAll_distrAll

divAll_aall :: (Ord obj,Eq mor) => DivAll obj mor -> AAll obj mor
divAll_aall = distrAll_aall . divAll_distrAll

ded_acat :: (Ord obj,Eq mor) => Ded obj mor -> ACat obj mor
ded_acat = distrAll_acat . ded_distrAll

ded_aall :: (Ord obj,Eq mor) => Ded obj mor -> AAll obj mor
ded_aall = distrAll_aall . ded_distrAll

ra_acat :: (Ord obj,Eq mor) => RA obj mor -> ACat obj mor
ra_acat = distrAll_acat . ra_distrAll

ra_aall :: (Ord obj,Eq mor) => RA obj mor -> AAll obj mor
ra_aall = distrAll_aall . ra_distrAll
\end{code}
%}}}

%{{{ \subsection{Equivalence for Matrix Atom Set Descriptions}{matBtoAtCat}
\subsection{Equivalence for Matrix Atom Set Descriptions}
\subsectlabel{matBtoAtCat}

For Boolean matrices, we provide a fast way
to define the functors between the Boolean matrix algebra
and an equivalent atom set algebra.
This is used in the atom allegory definitions for Boolean matrix relation algebras
that are generated in \subsectref{AAllOutput}.
The most important argument needed here is a function
mapping pairs of objects to their \emph{atom matrix},
which is a matrix of the shape of the corresponding Boolean matrix,
but containing in every position
that atom that stands for the atomic matrix with \verb|True| at that position:

\index{matBtoAtCat@{\texttt{matBtoAtCat}}}%
\index{atCatToMatB@{\texttt{atCatToMatB}}}%
\begin{code}
matBtoAtCat :: Ord atom => ([t] -> obj) -> (obj -> obj -> [[atom]]) ->
                           Fun obj (SetMor obj atom) (Vec t) (MatMor t Bool)
matBtoAtCat bToPobj atMat =
  let fmor m = let (bm,s,t) = unMatMor m
                   s' = bToPobj s
                   t' = bToPobj t
                   enter b a = if b then addToSet a else id
                   am = atMat s' t'
                   as = ffold (concat $ matZipWith enter bm am) zeroSet
               in SetMor (as,s',t')
  in Fun (bToPobj . unVec) fmor

atCatToMatB :: Ord atom => (obj -> [t]) -> (obj -> obj -> [[atom]]) ->
                           Fun (Vec t) (MatMor t Bool) obj (SetMor obj atom)
atCatToMatB pToBobj atMat =
  let fmor (SetMor (as,s,t)) =
         matMor (matMap (`elemSet` as) (atMat s t))
                (pToBobj s) (pToBobj t)
  in Fun (vec . pToBobj) fmor
\end{code}
%}}}

%{{{ \subsection{Matrix Atom Category Definitions}
\subsection{Matrix Atom Category Definitions}

With the tools available so far,
we already can build matrix algebras over atom set algebras.
But such a matrix algebra again has atoms ---
one might extract them using the tools from the previous section.

In this section we provide a direct definition
of atom algebra descriptions for matrix algebras over atom set algebras.

An atomic matrix is a matrix where one coefficient is an atom,
and all other coefficients are zero morphisms, i.e., empty sets.
Therefore, an atomic matrix is described by the single atom it contains
together with the position where it contains that atom:

\index{MatAt@{\texttt{MatAt}}}%
\begin{code}
type MatAt obj atom = (Int,Int,atom)
\end{code}

Working with matrix atoms turns out to be a lot easier
than working with matrices:

\index{acatMat@{\texttt{acatMat}}}%
\index{aallMat@{\texttt{aallMat}}}%
\begin{code}
acatMat :: Eq obj => ACat obj atom -> [[obj]] -> ACat [obj] (MatAt obj atom)
acatMat c oss = let objects = nub oss
 in if not (all (all (acat_isObj c)) oss) then error "acatMat: non-objects"
 else ACat
  {acat_isObj   = (`elem` objects)
  ,acat_isAtom  = (\ as bs (i,j,at) ->
      i >= 0 && i < length as &&
      j >= 0 && j < length bs &&
      let a = as !! i
          b = bs !! j
      in acat_isAtom c a b at)
  ,acat_objects = objects
  ,acat_atomset = (\ as bs ->
                     do (a,i) <- zip as [0..]
                        (b,j) <- zip bs [0..]
                        at <- acat_atomset c a b
                        return (i,j,at))
  ,acat_idmor   = (\ as ->
                     do (a,i) <- zip as [0..]
                        at <- acat_idmor c a
                        return (i,i,at))
  ,acat_comp    = (\ as bs cs (i1,j1,at1) (i2,j2,at2) ->
                     if j1 /= i2 then []
                     else let ats = acat_comp c (as !! i1) (bs !! i2) (cs !! j2)
                                                at1 at2
                          in map (\ at -> (i1,j2,at)) ats)
  }

aallMat :: Eq obj => AAll obj atom -> [[obj]] -> AAll [obj] (MatAt obj atom)
aallMat c oss = AAll
  {aall_acat = acatMat (aall_acat c) oss
  ,aall_converse = (\ as bs (i,j,at) ->
                      (j,i,aall_converse c (as !! i) (bs !! j) at))
  }
\end{code}
%}}}

%{{{ \subsection{Example Atom Sets}
\subsection{Example Atom Sets}

For use in constructing relation algebras
where the atoms do not carry clear-cut identities
we offer a few finite sets, so that redefinitions and clashing exports
are easier to avoid.

The following are therefore separate Haskell modules in separate source files:

\bigskip

\hrule

\index{A2@{\texttt{A2}}}%
\input{A2.lhs}

\hrule

\index{A4@{\texttt{A4}}}%
\input{A4.lhs}

\hrule

\index{A9@{\texttt{A9}}}%
\input{A9.lhs}

\hrule

%}}}


%% Local variables:
%% folded-file: t
%% fold-internal-margins: 0
%% eval: (fold-set-marks "%{{{ " "%}}}")
%% eval: (fold-whole-buffer)
%% end:
