\section{Drawing}\sectlabel{Draw}

Drawings for concrete relations can for example choose
a Boolean matrix representation, or a (directed) graph representation ---
both possibilities are implemented in the RelView system
\cite{Behnke-Berghammer-Schneider-1997}.

For abstract categories or allegories, there is no similarly general approach.
For relation algebras, we may use the atoms to arrive at a general
representation of their morphisms.
Similarly, as soon as morphisms of a coefficient allegory have a graphical
representation, we may use that to build matrix representations.

In this module we provide a very simple drawing interface for
some of the constructions of \chaptref{Construction}.
We target PostScript generation,
delegating some of the tasks to the document formatting system Lout
\cite{Kingston-1995_lout.user}.

\begin{code}
module Draw where

import System

import RelAlg
import Atomset
import Matrix

data Length = NoLength | CM Double | Pt Double   | MM Double 
                                   | Inch Double | FontSize Double
lengthToLoutStr NoLength = ""
lengthToLoutStr (CM l) = shows l "c"
lengthToLoutStr (Pt l) = shows l "p"
lengthToLoutStr (MM l) = shows (l/10.0) "c"
lengthToLoutStr (Inch l) = shows l "i"
lengthToLoutStr (FontSize l) = shows l "f"

data GapMode = None | Edge
instance Show GapMode where
  showsPrec _ None = id
  showsPrec _ Edge = ('e':)

type PostScript = String
\end{code}

Because of its flexibility and powerful alignment operations,
we use a fragment of the graphical object model
of Lout \cite{Kingston-1995_lout.user}:

\begin{code}
data Lout = PS PostScript Lout
          | HCat Bool GapMode Length [Lout]
          | VCat Bool GapMode Length [Lout]
          | Scale (Maybe (Either Double (Double, Double))) Lout
          | Wrap Length Length Lout
          | Box Length Length Lout
          | Limit Length Length Lout
          | Empty

ps w h ps = PS ps (Limit w h Empty)
\end{code}

Since we also use the implementation of Lout to generate images,
here we define an appropriate instance of \verb|Show|
that produces legal Lout code:

\begin{code}
instance Show Lout where
  showsPrec _ (Limit w h l) =
    (case w of NoLength -> id
               _ -> ((lengthToLoutStr h ++ " @Wide ") ++)) .
    (case h of NoLength -> id
               _ -> ((lengthToLoutStr h ++ " @High ") ++)) . shows l
  showsPrec _ Empty = ("{}\n"++)
  showsPrec _ (PS ps l) = ("{\n" ++) . (ps++) . ("\n}\n@Graphic { "++) .
      shows l . ("}\n" ++)
  showsPrec _ (Box lw m l) = ("@Box " ++) .
    (case lw of NoLength -> id
                _ -> (("linewidth {" ++ lengthToLoutStr lw ++ "} ") ++)) .
    (case m of NoLength -> id
               _ -> (("margin {" ++ lengthToLoutStr m ++ "} ") ++)) .
    ("{\n" ++) . shows l . ("}\n" ++)
  showsPrec _ (Wrap h v l) =
    ((lengthToLoutStr v ++ " @High {} // {\n") ++) .
    ((lengthToLoutStr h ++ " @Wide {} || {\n") ++) .
    shows l .
    (("} || " ++ lengthToLoutStr h ++ " @Wide {}\n") ++) .
    (("} // " ++ lengthToLoutStr v ++ " @High {}\n") ++)
  showsPrec _ (Scale s l) =
     ('{':) . scale . (" @Scale " ++) . shows l . ("}\n"++)
   where scale = case s of
           Nothing -> id
           Just (Left f) -> shows f
           Just (Right (h,v)) -> ('{':) . shows h . (',':) . shows v . ('}':)
  showsPrec _ (HCat b m w ls) = case ls of
     [] -> ("{}" ++)
     [l] -> shows l
     _ -> ("{\n" ++) . fold1 hcat shows id ls . ("}\n"++)
   where hcat x y = shows x . ((if b then "|" else "||") ++) .
                        ((lengthToLoutStr w) ++) . shows m . ('\n' : ) . y

  showsPrec _ (VCat b m w ls) = case ls of
     [] -> ("{}" ++)
     [l] -> shows l
     _ -> ("{\n" ++) . fold1 vcat shows id ls . ("}\n"++)
   where vcat x y = shows x . ((if b then "/" else "//") ++) .
                        ((lengthToLoutStr w) ++) . shows m . ('\n' : ) . y


fold1 f g e [] = e
fold1 f g e [x] = g x
fold1 f g e (x : xs) = f x (fold1 f g e xs)
\end{code}

The most important use of the alignment operations
is for producing matrix drawings:

\begin{code}
loutMatrix :: GapMode -> Length -> GapMode -> Length -> [[Lout]] -> Lout 
loutMatrix gh dh gv dv m = VCat True gv dv $ map (HCat True gh dh) m

psMatrix :: GapMode -> Length -> GapMode -> Length -> [[PostScript]] -> Lout
psMatrix gh dh gv dv m = loutMatrix gh dh gv dv (map (map f) m)
 where f p = ps (MM 4) (MM 4) (p ++ '\n' : graphicFramePath ++ "stroke\n")

graphicFramePath = "0     0     moveto    0     ysize lineto\n" ++
                   "xsize ysize lineto    xsize 0     lineto closepath\n"

graphicFrameScale h v = "xsize " ++ shows h " div\n"
                     ++ "ysize " ++ shows v " div scale\n"
graphicCenterFrameScale h v =
 "xsize 2 div ysize 2 div translate\n" ++
 "xsize " ++ shows h " 2 mul div\n" ++
 "ysize " ++ shows v " 2 mul div scale\n"
\end{code}

Since for RelView-like output of Boolean matrices
we also need that frame for filling,
we additionally provide an abbreviated variant:

\begin{code}
boolMatElem :: Bool -> Lout
boolMatElem b = ps (MM 4) (MM 4) (
  graphicFramePath
  ++ if b then "gsave 0.7 setgray fill grestore stroke\n" else "stroke\n"
 )

defaultLoutMatrix = loutMatrix None NoLength None NoLength
boolMatLout m = defaultLoutMatrix (map (map boolMatElem) m)

loutMatMor :: Length -> Length ->
              GapMode -> Length -> GapMode -> Length ->
              (obj -> Lout) -> (mor -> Lout) -> MatMor obj mor -> Lout
loutMatMor seph sepv gh dh gv dv objLout morLout mm =
  let (m,s,t) = unMatMor mm
      mat = loutMatrix gh dh gv dv $ map (map morLout) m
      src = VCat True gv dv $ map objLout s
      trg = HCat True gh dh $ map objLout t
  in VCat True None sepv
          [HCat True None seph [Empty,trg], HCat True None seph [src,mat]]
\end{code}

For atom set relations,
we rely on the verbatim PostScript inclusion features of Basser Lout
with the idea that every atom has some PostScript encoding,
and the PostScript encodings of all atoms present in a relation
are overlaid to produce the presentation of that relation.
For examples where this works out nicely see the compass algebras in
\sectref{Compass}.

At first we define a function that expects separate Lout-producing functions
for atoms and lists of atoms:

\begin{code}      
loutAtComp :: Length -> Length ->
              GapMode -> Length -> GapMode -> Length ->
              (atom -> Lout) -> ([atom] -> Lout) ->
              ACat obj atom -> obj -> obj -> obj -> Lout
loutAtComp seph sepv gh dh gv dv atLout morLout ac o1 o2 o3 =
  let atoms1 = acat_atomset ac o1 o2
      atoms2 = acat_atomset ac o2 o3
      m = map
          (\ a1 -> map (\ a2 -> morLout (acat_comp ac o1 o2 o3 a1 a2)) atoms2) 
          atoms1
      mat = loutMatrix gh dh gv dv m
      src = VCat True gv dv $ map atLout atoms1
      trg = HCat True gh dh $ map atLout atoms2
  in VCat True None sepv [HCat True None seph [Empty,trg], 
                          HCat True None seph [src,mat]]
\end{code}

These two functions are now produced together,
based on a common PostScript prologue,
a function turning individual atoms into PostScript code fragments,
and a fixed Lout object to wrap the PostScript around:

\begin{code}
mkAtMorLout :: PostScript -> (atom -> PostScript) -> Lout ->
               (atom -> Lout, [atom] -> Lout)
mkAtMorLout base atPS atLout =
  (\ at -> PS (base ++ atPS at) atLout
  ,\ mor -> PS (base ++ concatMap atPS mor) atLout
  )

loutPSAtComp :: PostScript -> (atom -> PostScript) -> Lout ->
    Length -> Length -> GapMode -> Length -> GapMode -> Length ->
    ACat obj atom -> obj -> obj -> obj -> Lout
loutPSAtComp base atPS atLout  seph sepv gh dh gv dv ac o1 o2 o3 =
  loutAtComp seph sepv gh dh gv dv atomLout morLout ac o1 o2 o3
   where (atomLout, morLout) = mkAtMorLout base atPS atLout
\end{code}

For obtaining separating lines between the composition table proper
and its labellings
we might revert to Lout's tables instead of employing plain object
compositions; here is a simple ``hack'' that adds the two lines
{\sl a posteriori}, and with manually adjusted placement via the
argument {\tt corr}:

\begin{code}
loutPSAtComp' base atPS atLout corr seph sepv gh dh gv dv ac o1 o2 o3 =
  PS
  (unlines
  ["newpath"
  ,"\"/dx\" xsize " ++ show xl ++ " div " ++ show corr ++ " add def"
  ,"\"/dy\" ysize dup " ++ show yl ++ " div " ++ show corr ++ " add sub def"
  ,"dx 0 moveto dx ysize lineto"
  ,"0 dy moveto xsize dy lineto"
  ,"stroke"
  ])
  $
  loutPSAtComp base atPS atLout  seph sepv gh dh gv dv ac o1 o2 o3
 where
  xl = length (acat_atomset ac o1 o2) + 1
  yl = length (acat_atomset ac o2 o3) + 1
\end{code}

For testing these capabilities,
we play around a little bit:

\begin{code}
mat1 = matX 8 8

matX i j = do r <- [1..i]
              [do c <- [1..j]
                  [r `mod` c == 0 || (r + c >= r * c)]]

mat2 = do i <- [1..4]
          [do j <- [1..6]
              [Box NoLength NoLength (boolMatLout (matX i j))]]

loutDocFile file lout = writeFile file $ unlines
  ["@SysInclude {doc}"
  ,"@Doc @Text @Begin"
  ,show lout
  ,"@End @Text"
  ]

loutPicFile file lout = writeFile file $ unlines
  ["@SysInclude {picture}"
  ,"@Illustration {"
  ,show (Wrap (MM 2) (MM 2) lout)
  ,"}"
  ]

mkLoutPic base lout = let ltfile = base ++ ".lt" in
  do loutPicFile ltfile lout
     system ("lout -EPS -c " ++ base ++ " -o " ++ base ++ ".eps " ++ ltfile)

mk_mat1 = mkLoutPic "mat1" $ boolMatLout mat1
mk_mat2 = mkLoutPic "mat2" $ defaultLoutMatrix mat2
\end{code}

This produces the following picture:

\medskip
\centerline{\includegraphics{mat2.eps}}

\medskip
For demonstrating the usefulness of Lout's alignment operations
we also build a variant with transposed coefficient matrices:

\begin{code}
mat2a = do i <- [1..4]
           [do j <- [1..6]
               [PS (graphicFramePath ++ "stroke\n")
                   $ Wrap (MM 2) (MM 2) (boolMatLout (matX j i))]]
\end{code}

\medskip
\centerline{\includegraphics{mat2a.eps}}
\medskip
\centerline{\small Showing a matrix of matrices}



%% Local variables:
%% folded-file: t
%% fold-internal-margins: 0
%% eval: (fold-set-marks "%{{{ " "%}}}")
%% eval: (fold-whole-buffer)
%% end:
