\subsection{Standard Iterations}

\noindent
%{{{ Intro
There is a well-developed theory
of standard iterations for boolean matrices
to be found along with matching, assignment, games, correctness, etc. 
We will present tools for executing these iterations 
in the general framework studied here.


To this end, 
we concentrate on pairs of antitone mappings\index{antitone mappings}
occurring in relation algebras. These mappings are usually determined 
by an obviously antitone relational construct,
e.g., $w\mapsto\pi(w):= \RELneg{\RELcmp B w}$
based on the relation $B:V\leftrightarrow W$.
Many other antitone mappings are conceivable.

Such pairs lead to interesting Galois correspondences of their
fixed-points\index{fixed-point}, and give rise to iteration procedures.
From the numerous application areas studied with relations in the
classical sense we mention the following:

{\renewcommand{\arraystretch}{1.3}
\begin{tabular}{l@{\hskip1em}l@{\hskip1em}r@{\hskip1em}r}
{} & given relations & $\sigma(v)=\;$ & $\pi(w)=\;$
\\[.5ex]
bi-matrix games & $B: W\rel V, B'\: V\rel W$ &
$\relnot{B \rcmp v}$ & $\relnot{B' \rcmp w}$
\\
correctness & $B: V\rel V$ & $B \rcmp \relnot{v}$ & $\relnot{w}$
\\
minorants, majorants & $E: V\rel V$
 & $\relnot{\relnot{E}\rcmp  w}$ & $\relnot{\relnot{\rtrans{E}}\rcmp  v}$
\\
coverings & $Q: V\rel W$ & $\rtrans{Q}\rcmp  \relnot{v}$ & $Q\rcmp \relnot{w}$
\\
independence & $Q: V\rel W$ & $\relnot{\rtrans{Q}\rcmp  v}$ & $\relnot{Q\rcmp  w}$
\\
assignment & $\lambda\rsubs Q: V\rel W$ & $\relnot{\rtrans{Q} \rcmp v}$ & $\relnot{\lambda \rcmp w}$
\end{tabular}
}%renewcommand{\arraystretch}

\bigskip
\noindent
The antitone mappings are related to study questions such as the following:

\medskip
\BM
\begin{array}{r@{\ \rsubs\ }r@{\hskip2em}l@{\hskip2em}l}
B \rcmp \relnot{q} &  \relnot{q}
& \mbox{contraction} & \mbox{partial correctness}
\\
\relnot{q} &  B \rcmp \relnot{q}
& \mbox{complement expansion} & \mbox{total correctness}
\\
B \rcmp x & \relnot{x}
& \mbox{stability} & \mbox{kernels, games}
\\
\relnot{x} & B \rcmp x
& \mbox{absorption}
\end{array}
\EM

\medskip
\noindent
Some of these classical iterations are presented in
\cite{Schmidt-Stroehlein-1985a,Schmidt-Stroehlein-1993}. Here, they are
schematically transferred into the present setting. 
\bigskip
\bigskip

\centerline{\includegraphics[scale=0.7]{Antimorph1.eps}}

\bigskip
\centerline{\small  Bounds and fixed-points of antitone mappings}

We express $\sigma, \pi$
with the elementary operations of the respective algebra
and apply them to an appropriate starting configuration $a_0, b_0$.
Appropriate means that $a_0\rsubs\pi(b_0)\rsubs a$
and $b\rsubs\sigma(a_0)\rsubs b_0$,
where $a$ is the least fixed-point of
$\rho = \lambda v . \pi(\sigma(v))$
and $b$ is the greatest fixed-point of
$\psi = \lambda w . \sigma(\pi(w))$.
This complicated condition
is usually satisfied rather trivially with $a_0 = \RO{}$ and $b_0 = \RL{}$.
Nested iterations will then start with 
$a_0$ on the left and $b_0$ on the right:
$$
    a_{i+1}
  :=
    \pi(b_i)
\enskip, \qquad
    b_{i+1}
  :=
    \sigma(a_i)
\enskip.
$$
\bigskip
\noindent
These iterations will end up in the following two sequences, one of
which is ascending while the other descends.
$$
a_0\rsubs a_1\rsubs\ldots\rsubs a_\infty
\rsubs\pi(b_\infty)\rsubs a,\quad
b \rsubs \sigma  (a_{\infty}  ) \rsubs b_{\infty} 
\rsubs \ldots \rsubs b_1  \rsubs b_0. 
$$
The effect of the iteration is that the least fixed-point $a$ 
of $v\mapsto\pi(\sigma(v))$ on the 
side started with $a_0$ is related to the
greatest fixed-point $b$ of $w\mapsto\sigma(\pi(w))$
on the side started from $b_0$.
The final situation obtained will be characterised 
by $a=\pi(b)$ and $\sigma(a)=b$.
It will always produce another admissible
starting configuration $a_0':=a_\infty, b_0':=b_\infty$.

It is not clear from the beginning whether the iteration
will reach the fixed-points in a finite number of steps, 
as the mapping $w\mapsto\pi(w)$, e.g.,
might not be continuous.
In interesting applications, however, this is the case;
in particular in the finite case.
(Of course this iteration may also be executed the other way round, i.e.,
starting with $\RL{}$ on the left 
and with $\RO{}$ on the right.)



The configuration is more specific in the homogeneous case.
The sequences may meet each other or may fail to do so. 

In any case, interesting investigations have been possible 
in the case of relations in the classical sense.
It is challenging to look for interpretations of similar results 
in the context of the more general examples of 
relation algebras presented here.

\bigskip

The program module for standard iterations 
starts with a module heading.

\begin{code}
module Iterations where

import RelAlg
import Matrix
import ExtPrel
\end{code}

\medskip
\noindent
Then the basic iterations along the well-known
{\tt until}-construct of Haskell with {\tt lr} for $\sigma$ and
{\tt rl} for $\pi$ are formulated.

\index{antiFix@{\texttt{antiFix}}}
\begin{code}
antiFix :: (Eq a, Eq b) => (b -> a) -> (a -> b) -> (b,a) -> (b,a)
antiFix lr rl = untilFix f
  where f (v,w) = (rl w, lr v)
\end{code}

\noindent
For reasons of monotony, the iteration will always 
terminate at the fixed-points in the finite case;
see \cite[A.3.11]{Schmidt-Stroehlein-1993}.

The start may often be determined from the row
and column number of the given basic relations
inserting $\RO{}$ and $\RL{}$ as appropriate.

\index{startVector@{\texttt{startVector}}}
\begin{code}
startVector :: MatMor obj mor -> (obj -> obj -> mor') -> MatMor obj mor'
startVector b tf = let (m,s,t) = unMatMor b
                       t' = head s
                       m' = map ((:[]) . flip tf t') t
                   in MatMor (m',t,[t'])
\end{code}

\bigskip
\noindent
We will now apply this general scheme to several applications.
%}}}

%{{{ \subsubsection*{Initial Part}
\subsubsection*{Initial Part}

\noindent
A nice example for a fixed-point of antitone functionals
is determining the initial part of a relation
along the lines of \cite[6.3.4]{Schmidt-Stroehlein-1993}.
(Remember, however, that for nonfinite relations the concepts of 
being progressively finite and progressively bounded will be different.)
The two antitone functionals 
$v \mapsto B\rcmp \RELneg{v}$ and $w \mapsto \RELneg{w}$ 
are given as follows:

\begin{code}
antitoneFctlCorr1 ra b = \ v -> ra_comp ra b $ ra_compl ra v
antitoneFctlCorr2 ra b = \ w -> ra_compl ra w          -- independent of b!
\end{code}

\noindent
Applying the general scheme, we obtain the initial part 
in the resulting pair of

\index{initialPart@{\texttt{initialPart}}}
\begin{code}
initialPart ra b = antiFix (antitoneFctlCorr1 ra b)
                           (antitoneFctlCorr2 ra b)
                           (startVector b ccFalse, startVector b ccTrue)
\end{code}
%}}}

%{{{ \subsubsection*{Bi-Matrix Games}
\subsubsection*{Bi-Matrix Games}

\noindent
Next, we look for solutions of bi-matrix games. 
Let two matrices $B:V\leftrightarrow W, B':W\leftrightarrow V$ be given.
The antitone functionals based on these relations
are formed in quite a similar manner.

\begin{code}
antitonFctlGame ra = \ b -> (\ x -> ra_compl ra (ra_comp ra b x))
\end{code}

\noindent
The solution of the game is then again determined 
following the general scheme.

\index{gameSolution@{\texttt{gameSolution}}}
\begin{code}
gameSolution ra b b' = antiFix (antitonFctlGame ra b ) 
                               (antitonFctlGame ra b') 
                               (startVector b' ccTrue, startVector b ccFalse)
gameSolutINV ra b b' = antiFix (antitonFctlGame ra b ) 
                               (antitonFctlGame ra b') 
                               (startVector b' ccFalse, startVector b ccTrue)
\end{code}

\noindent
The final situation is characterised by the formulae
$a=\RELneg{B\rcmp b}$ and $\RELneg{B'\rcmp a}=b$
for the game iteration as well as for the inverted iteration.
The respective smaller resulting relation gives loss positions, while
the larger ones indicate loss positions together with draw positions.
%}}}

%{{{ \subsubsection*{Matching and Assignment}
\subsubsection*{Matching and Assignment}

\noindent
An additional antimorphism situation is known to exist in connection with
matching and assignment.
Let two matrices $Q,\lambda:V\leftrightarrow W$ be given where
$\lambda\RELenthOP Q$ is univalent.

\index{assignSolution@{\texttt{assignSolution}}}
\begin{code}
antitoneFctlAssign ra = \ b -> (\ x -> ra_compl ra (ra_comp ra b x))

assignSolution ra q lambda 
      = antiFix (antitoneFctlAssign ra (ra_converse ra q)) 
                (antitoneFctlAssign ra lambda)
                (startVector q ccFalse
                ,startVector (ra_converse ra lambda) ccTrue)

assignSolutINV ra q lambda 
      = antiFix (antitoneFctlAssign ra (ra_converse ra q))  
                (antitoneFctlAssign ra lambda)
                (startVector q ccTrue
                ,startVector (ra_converse ra lambda) ccFalse)
\end{code}
In the classical case, the results of these iterations produce 
appropriate starting points, if any, 
where to apply successfully the alternating chain procedure.
%}}}

%{{{ \subsubsection*{Bounds wrt. Orderings}
\subsubsection*{Bounds wrt. Orderings}

\noindent
The following functions offer the possibility to calculate 
the majorants and the minorants of a relation as well as least upper and
greatest lower bounds,
provided the corresponding ordering is given as a first argument.

\index{mi@{\texttt{mi}}}
\index{ma@{\texttt{ma}}}
\index{lub@{\texttt{lub}}}
\index{glb@{\texttt{glb}}}
\begin{code}
mi, ma, lub, glb :: DivAll obj mor -> mor -> mor -> mor

mi da ord m = divAll_lres da ord (divAll_converse da m)
ma da ord m = let conv = divAll_converse da
              in divAll_lres da (conv ord) (conv m)

lub da ord m = divAll_meet da maom (mi da ord maom)  where maom = ma da ord m
glb da ord m = divAll_meet da miom (ma da ord miom)  where miom = mi da ord m
\end{code}

This might also be done using the {\tt antiFix}-scheme presented 
in this section.
From theoretical considerations, however, 
it is clear that these iterations
will be stationary after one step back and forth.
This makes a direct computation the better choice.
%}}}

%{{{ \subsubsection*{Conclusion}
\subsubsection*{Conclusion}

\noindent
It seems extremely interesting, 
to find out how these standard iterations behave
if matrices are taken the coefficients of which are drawn from
other relation algebras.
Do, e.g., matrices over an interval algebra lead to
steering algorithms? Will game algorithms over matrices with
pairs (interval, compass) give hints to escape games?
Will there be targeting games?
%}}}


%% Local variables:
%% folded-file: t
%% fold-internal-margins: 0
%% eval: (fold-set-marks "%{{{ " "%}}}")
%% eval: (fold-whole-buffer)
%% end:
