
\section{Matrix Algebra Construction}

Concrete relations can usefully be represented as Boolean matrices.
We have seen that the Boolean algebra of truth values
in itself can already be considered as a relation algebra,
the relation algebra of Boolean $1\times 1$-matrices.

We now generalise the construction of matrix relation algebras to coefficients
stemming from arbitrary relation algebras,
or, for simpler structures,
to coefficients from distributive allegories.

%{{{ \subsection{Matrix Categories}
\subsection{Matrix Categories}

Given a base allegory or relation algebra,
we now want to define matrix algebras over this base.
Objects of the matrix algebra are going to be lists of objects of the base,
and morphisms are going to be matrices of morphisms of the base,
where source and target depend on the position in the matrix.

Composition will be based on an appropriate variant of the skalar product:
we have to use composition as multiplication,
and join as addition --- therefore,
already for defining just a category of matrices,
we need coefficients from a distributive allegory.

%{{{ {The}Given a distributive allegory, a category $Mat_{\catC}$
\begin{The}\index{matrix algebras}
If
$\catC = (Obj_{\catC}, Mor_{\catC}, \_ : \_ \rel \_, \RId{}, \rcmp, \RAconverse{}, \reland, \relor, \RO{})$
is a distributive allegory, then
a category $Mat_{\catC}$ may be defined as follows:
\begin{itemize}
\item objects of $Mat_{\catC}$ are finite sequences of objects of $\catC$,
\item for two objects $\objA = [\objA_1,\ldots,\objA_a]$ and $\objB = [\objB_1,\ldots,\objB_b]$ of $Mat_{\catC}$,
  the associated homset $\CThom{Mat_{\catC}}{\objA}{\objB}$
  contains all matrices $(f_{i,j})_{i \in \{1,\ldots,a\},j \in \{1,\ldots,b\}}$
  for which for every $i \in \{1,\ldots,a\}$ and every $j \in \{1,\ldots,b\}$
  the coefficient $f_{i,j}$ is a homomorphism from $\objA_i$ to $\objB_j$ in $\catC$,
\item given three objects
$\objA = [\objA_1,\ldots,\objA_a]$,
$\objB = [\objB_1,\ldots,\objB_b]$, and
$\objC = [\objC_1,\ldots,\objC_c]$, and two morphisms
$R : \objA \tfun \objB$, and
$S : \objB \tfun \objC$,
their composition is defined by the following:
\BD
    (R\rcmp S)_{i,k}
  \defeq
    \bigrelor_{j\in\{1,\ldots,b\}}\; (R_{i,j}\rcmp S_{j,k})
\qquad
\qquad
\mbox{for all $i \in \{1,\ldots,a\}$ and $j \in \{1,\ldots,c\}$,}
\ED
\item for an object $\objA = [\objA_1,\ldots,\objA_m]$,
  the identity morphism $\RId{\objA}: \objA \tfun \objA$
  is defined by
\BD
    (\RId{\objA})_{i,j}
  =
   \left\{\begin{array}{l@{\hskip2cm}l}
    \RId{\objA_i} & \mbox{if $i = j$}
   \\
    \RO{\objA_i,\objA_j} & \mbox{if $i\neq j$}
   \end{array}\right.
\ED
\end{itemize}
\end{The}
\Proof{Well-definedness of identity and composition morphisms is obvious.
Associativity of composition and  the identity properties
are shown by standard matrix arguments as follows.

Given four objects
\BD
\objA = [\objA_1,\ldots,\objA_a],
\objB = [\objB_1,\ldots,\objB_b],
\objC = [\objC_1,\ldots,\objC_c], \mbox{\ and}\;
\objD = [\objD_1,\ldots,\objD_d],
\ED and three morphisms
$R : \objA \tfun \objB$,
$S : \objB \tfun \objC$, and
$T : \objC \tfun \objD $,
we can prove the associativity of composition by the following calculation:
\BDAR {} \sepA{}
    ((R\rcmp S) \rcmp T)_{i,l}
  \sepB{=}
    \bigrelor_k\; ((R\rcmp S)_{i,k}\rcmp T_{k,l})
  \sepB{=}
    \bigrelor_k\; ((\bigrelor_j\; (R_{i,j}\rcmp S_{j,k}))\rcmp T_{k,l})
  \sepBR{=}{\mbox{join-distributivity in $\catC$}}
    \bigrelor_k\; (\bigrelor_j\; ((R_{i,j}\rcmp S_{j,k})\rcmp T_{k,l}))
  \sepBR{=}{\mbox{associativity of composition in $\catC$}}
    \bigrelor_k\; (\bigrelor_j\; (R_{i,j}\rcmp (S_{j,k}\rcmp T_{k,l})))
  \sepBR{=}{\mbox{commutativity and associativity of join in $\catC$}}
    \bigrelor_j\; (\bigrelor_k\; (R_{i,j}\rcmp (S_{j,k}\rcmp T_{k,l})))
  \sepBR{=}{\mbox{join-distributivity in $\catC$}}
    \bigrelor_j\; (R_{i,j}\rcmp (\bigrelor_k\; (S_{j,k}\rcmp T_{k,l})))
  \sepB{=}
    \bigrelor_j\; (R_{i,j}\rcmp (S\rcmp T)_{j,l})
  \sepB{=}
    (R\rcmp (S\rcmp T))_{i,l}
\EDAR
Right-identity:
\BDA
    (R \rcmp \RId{\objB})_{i,j}
  \sepA{=}
    \bigrelor_{j'\in\{1,\ldots,b\}}\; (R_{i,j}\rcmp (\RId{\objB})_{j',j})
  \sepB{=}
    R_{i,j}\rcmp (\RId{\objB})_{j,j} \relor
    \bigrelor_{j'\in\{1,\ldots,b\} - \{j\}}\; (R_{i,j}\rcmp (\RId{\objB})_{j',j})
  \sepB{=}
    R_{i,j}\rcmp \RId{\objB_j} \relor
    \bigrelor_{j'\in\{1,\ldots,b\} - \{j\}}\; (R_{i,j}\rcmp \RO{\objB_{j'},\objB_j})
  \sepB{=}
    R_{i,j} \relor
    \bigrelor_{j'\in\{1,\ldots,b\} - \{j\}}\; \RO{\objA_i,\objB_j}
  \sepB{=}
    R_{i,j} \relor \RO{\objA_{i},\objB_j}
  \sepB{=}
    R_{i,j}
\EDA
Left-identity is shown in an analogous way.
\qed
}%Proof
%}}}

Note that the choice of composition for the multiplication of coefficients
occurring in the definition of composition
is not arbitrary;
it would be misguided to orient oneself at the meet
this composition degenerates to
in the case of the simple Boolean lattice of truth values.
The meet would not even be well-defined
since we have $R_{i,j} : \objA_i \rel \objB_j$
and $S_{j,k} : \objB_j \rel \objC_k$.

%{{{ module
For the time being, we use a simple list implementation of matrices,
but we make it abstract so that we can exchange it later for something
more efficient.
Therefore we have to provide an explicit export list
that makes the names of the abstract types \verb|Vec| and \verb|MatMor|
available to importing modules, but hides their implementation:

\begin{code}
module Matrix(catMat,allMat,distrAllMat,divAllMat,dedMat,raMat
             ,Vec(),vec,unVec,MatMor(),matMor,unMatMor,matMorMap,bM
             ,matMap,matZipWith
             ) where

import RelAlg

import List(nub)
import qualified List(transpose)
\end{code}
%}}}

Objects are just lists of objects of the base category:

\index{Vec@{\texttt{Vec}}}%
\index{vec@{\texttt{vec}}}%
\index{unVec@{\texttt{unVec}}}%
\begin{code}
newtype Vec a = Vec [a] deriving (Eq, Ord, Show, Read)
vec = Vec
unVec (Vec s) = s
\end{code}

For the matrix itself we use the usual list-of-lists approach:

\index{Mat@{\texttt{Mat}}}%
\index{matMap@{\texttt{matMap}}}%
\index{matZipWith@{\texttt{matZipWith}}}%
\begin{code}
type Mat a = [[a]]
matMap = map . map
matZipWith = zipWith . zipWith
\end{code}

It is important to note that we demand the following
\textbf{consistency condition}:
A matrix representing a morphism
$R:[\objA_1,\ldots,\objA_a] \rel [\objB_1,\ldots,\objB_b]$
is a list with exactly $a$ elements (called \emph{rows}),
each of which is a list containing exactly $b$ elements.
For ease of implementation we demand this also for cases where
$a$ or $b$ are zero.

In order to be able to reconstruct source and target of a morphism
even in these cases, we need to include the source and target
object lists with the matrix proper in our morphism type:

\index{MatMor@{\texttt{MatMor}}}%
\begin{code}
newtype MatMor obj mor = MatMor (Mat mor, [obj], [obj])
                                      deriving (Eq, Ord, Show, Read)
\end{code}
We export a variant of the constructor
without checking the consistency condition;
for this purpose one may use \verb|cat_isMor| from below:

\index{matMor@{\texttt{matMor}}}%
\index{unMatMor@{\texttt{unMatMor}}}%
\index{matMorMap@{\texttt{matMorMap}}}%
\index{bM@{\texttt{bM}}}%
\begin{code}
matMor m s t = MatMor (m, s, t) 
unMatMor (MatMor tr) = tr

matMorMap f (MatMor (m, s, t)) = MatMor (matMap f m, s, t)

bM m = let s = replicate (length m) ()
           t = replicate (length (head m)) ()
       in MatMor (m, s, t)
\end{code}

At the heart of the composition of matrices is the
``skalar product''\index{skalar product},
which needs to be given $\objA_x$ and $\objC_y$
along with the row $[R_{x,1},\ldots,R_{x,b}]$
and the column $[S_{1,y},\ldots,S_{b,y}]$
because both might be empty and we still need to find the correct bottom
element:

\begin{code}
skalprod :: DistrAll obj mor -> (obj,[mor]) -> (obj,[mor]) -> mor
skalprod da (a,ms1) (b,ms2) =
   foldr (distrAll_join da) (distrAll_bottom da a b) $
   zipWith (distrAll_comp da) ms1 ms2
\end{code}


%{{{ transpose
We shall obtain the columns needed for the skalar products
by transposition of the matrix.
Because of our consistency condition,
we need to be careful when transposing empty matrices,
where we need non-empty results if the original target object vector
is non-empty:

\begin{code}
transpose :: MatMor obj mor -> MatMor obj mor
transpose (MatMor (m, s, t)) =
  let m' = if null s then map (const []) t
           else List.transpose m
  in MatMor (m', t, s)
\end{code}
%}}}

%{{{ matIdmor
An identity morphism is easily constructed:

\begin{code}
matIdmor :: DistrAll obj mor -> [obj] -> Mat mor
matIdmor _   []     = []
matIdmor all (a:as) =
  (distrAll_idmor all a : map (\ a' -> distrAll_bottom all a a') as) :
  zipWith (\ a' ms -> distrAll_bottom all a' a : ms) as (matIdmor all as)
\end{code}
%}}}

%{{{ shape, instantiate
For generating the list of all morphisms (in the finite case only)
we first use \verb|shape| to generate a matrix containing the respective object pairs:

\begin{code}
shape :: [a] -> [b] -> Mat (a,b)
shape as bs = [[(a,b)| b <- bs] | a <- as]
\end{code}

This matrix is then instantiated in all possible ways
by providing the \verb|homset| function as first argument to the following:

\begin{code}
instantiate :: (a -> [b]) -> Mat a -> [Mat b]
instantiate g m =
  let -- inst' :: [a] -> [[b]]
      inst' [] = [[]]
      inst' (a:as) = [b:bs | b <- g a , bs <- inst' as]
      -- inst'' :: [[a]] -> [Mat b]
      inst'' [] = [[]]
      inst'' (as:ass) = [bs:bss | bs <- inst' as, bss <- inst'' ass]
  in inst'' m
\end{code}
%}}}

%{{{ catMat
That is all we need to define a matrix category:

\index{catMat@{\texttt{catMat}}}%
\begin{code}
catMat :: (Ord obj, Eq mor) => DistrAll obj mor -> [[obj]]
                            -> Cat (Vec obj) (MatMor obj mor)
catMat da objss = let
    objs = map Vec $ nub objss
 in Cat
  {cat_isObj   = (`elem` objs)
  ,cat_isMor   = (\ (Vec s) (Vec t) (MatMor (mss, s', t')) ->
                    s == s' && t == t' &&
                    length mss == length s' &&
                    let lt = length t' in
                    all (\ row -> length row == lt) mss &&
                    and (do (a,ms) <- zip s mss
                            (b,m) <- zip t ms
                            return $ distrAll_isMor da a b m))
  ,cat_objects = objs
  ,cat_homset  = (\ (Vec s) (Vec t) ->
                     let sh = shape s t
                         mats = instantiate (uncurry $ distrAll_homset da) sh
                     in map (\m -> MatMor (m,s,t)) mats)
  ,cat_source  = (\ (MatMor (_,s,_)) -> Vec s)
  ,cat_target  = (\ (MatMor (_,_,t)) -> Vec t)
  ,cat_idmor   = (\ (Vec s) -> MatMor (matIdmor da s, s, s))
  ,cat_comp    = (\ (MatMor (mss1,s1,t1)) m2@(MatMor (_ ,s2,t2)) ->
                    if t1 /= s2
                    then error ("matrix composition type error " ++
                                show (length t1) ++ ' ' : show (length s2))
                    else let MatMor (mss2T,_,_) = transpose m2
                             mss2C = zip t2 mss2T
                             mkline ms1 = map (skalprod da ms1) mss2C
                             mat = map mkline (zip s1 mss1)
                         in MatMor (mat, s1, t2))
  }
\end{code}
%}}}
%}}}

%{{{ \subsection{Matrix Allegories}
\subsection{Matrix Allegories}

With coefficients from a distributive allegory,
the additional allegory operations are easily added to a matrix category:

%{{{ {The}Given a distributive allegory, an allegory $Mat_{\catC}$
\begin{The}
If
$\catC = (Obj_{\catC}, Mor_{\catC}, \_ : \_ \rel \_, \RId{}, \rcmp, 
\RAconverse{}, \reland, \relor, \RO{})$
is a distributive allegory, then
$Mat_{\catC}$ may be extended to an allegory by defining,
for any two objects
$\objA = [\objA_1,\ldots,\objA_a]$ and
$\objB = [\objB_1,\ldots,\objB_b]$,
\begin{itemize}
\item the converse of any morphism $R:\objA\rel\objB$
as follows:
\BD
  (\rtrans{R})_{j,i} = \rtrans{(R_{i,j})}
\qquad
\qquad
\mbox{for all $i \in \{1,\ldots,a\}$ and $j \in \{1,\ldots, b\}$,}
\ED
\item the meet of any two morphisms $R, S:\objA\rel\objB$
component-wise:
\BD
  (R \reland S)_{i,j} = R_{i,j} \reland S_{i,j}
\qquad
\qquad
\mbox{for all $i \in \{1,\ldots,a\}$ and $j \in \{1,\ldots, b\}$;}
\ED
\item inclusion between any two morphisms $R, S:\objA\rel\objB$
is then component-wise inclusion:
\BD
    R \rsubs S
  \quad\zeq\quad
    \forall i \in \{1,\ldots,a\}, j \in \{1,\ldots, b\} \dot
            R_{i,j} \rsubs S_{i,j}
\enskip.
\ED
\end{itemize}
\end{The}
\Proof{Definition of inclusion from meet,
lattice properties of meet,
distribution of converse over meet,
and that converse is an involution all follow directly from the
component-wise definitions.

Still to be checked are the following:
\begin{itemize}
\item distribution of converse over composition:
\BDA
    (\rtrans{(R\rcmp S)})_{k,i}
  \sepA{=}
    \rtrans{((R\rcmp S)_{i,k})}
  \sepB{=}
    \rtrans{(\bigrelor_j\; (R_{i,j}\rcmp S_{j,k}))}
  \sepB{=}
    \bigrelor_j\; \rtrans{(R_{i,j}\rcmp S_{j,k})}
  \sepB{=}
    \bigrelor_j\; (\rtrans{(S_{j,k})}\rcmp\rtrans{(R_{i,j})})
  \sepB{=}
    \bigrelor_j\; ((\rtrans{S})_{k,j}\rcmp(\rtrans{R})_{j,i})
  \sepB{=}
    (\rtrans{S} \rcmp \rtrans{R})_{k,i}
\EDA
\item meet-subdistributivity:
\BDA
    (Q\rcmp(R\reland S))_{i,k}
  \sepA{=}
    \bigrelor_j\; (Q_{i,j}\rcmp (R\reland S)_{j,k})
  \sepB{=}
    \bigrelor_j\; (Q_{i,j}\rcmp (R_{j,k}\reland S_{j,k}))
  \sepB{\rsubs}
    \bigrelor_j\; (Q_{i,j}\rcmp R_{j,k}\reland Q_{i,j}\rcmp S_{j,k})
  \sepB{\rsubs}
    (\bigrelor_j\; (Q_{i,j}\rcmp R_{j,k}))\reland \bigrelor_l\; (Q_{i,l}\rcmp S_{l,k})
  \sepB{=}
    (Q \rcmp R)_{i,k} \reland (Q \rcmp S)_{i,k}
  \sepB{=}
    (Q \rcmp R \reland Q \rcmp S)_{i,k}
\EDA
\item modal rule:
\BDA
    (Q\rcmp R\reland S)_{i,k}
  \sepA{=}
    (Q\rcmp R)_{i,k} \reland S_{i,k}
  \sepB{=}
    (\bigrelor_j\; Q_{i,j}\rcmp R_{j,k}) \reland S_{i,k}
  \sepB{=}
    \bigrelor_j\; (Q_{i,j}\rcmp R_{j,k} \reland S_{i,k})
  \sepB{\rsubs}
    \bigrelor_j\; ((Q_{i,j} \reland S_{i,k}\rcmp \rtrans{(R_{j,k})}) \rcmp R_{j,k})
  \sepB{=}
    \bigrelor_j\; ((Q_{i,j} \reland S_{i,k}\rcmp (\rtrans{R})_{k,j}) \rcmp R_{j,k})
  \sepB{\rsubs}
    \bigrelor_j\; ((Q_{i,j} \reland \bigrelor_{k'}\ (S_{i,k'}\rcmp (\rtrans{R})_{k',j})) \rcmp R_{j,k})
  \sepB{=}
    \bigrelor_j\; ((Q_{i,j} \reland (S \rcmp \rtrans{R})_{i,j}) \rcmp R_{j,k})
  \sepB{=}
    \bigrelor_j\; ((Q \reland S \rcmp \rtrans{R})_{i,j})) \rcmp R_{j,k})
  \sepB{=}
    ((Q \reland S \rcmp \rtrans{R})\rcmp R)_{i,k}
\EDAQ
\end{itemize}
}%Proof
%}}}

\index{allMat@{\texttt{allMat}}}%
%{{{ allMat
For implementing the matrix allegory,
we therefore need component-wise definitions for meet and inclusion,
and for conversion
we not only have to transpose the matrix (carefully, see above),
but also converse every coefficient:

\begin{code}
allMat :: (Ord obj, Eq mor) => DistrAll obj mor -> [[obj]]
                            -> All (Vec obj) (MatMor obj mor)
allMat da objss = All
  {all_cat = catMat da objss
  ,all_converse = (\ m1 -> let MatMor (m, s, t) = transpose m1
                               m' = matMap (distrAll_converse da) m
                           in MatMor (m', s, t))
  ,all_meet = (\ (MatMor (mss1,s1,t1)) (MatMor (mss2,s2,t2)) ->
                 if s1 /= s2
                 then error ("matrix meet source type error")
                 else if t1 /= t2
                 then error ("matrix meet target type error")
                 else let mat = zipWith (zipWith (distrAll_meet da)) mss1 mss2
                      in MatMor (mat,s1,t1))
  ,all_incl = (\ (MatMor (mss1,s1,t1)) (MatMor (mss2,s2,t2)) ->
                 if s1 /= s2
                 then error ("matrix inclusion source type error")
                 else if t1 /= t2
                 then error ("matrix inclusion target type error")
                 else all and $ zipWith (zipWith (distrAll_incl da)) mss1 mss2)
  }
\end{code}
%}}}
%}}}

%{{{ \subsection{Distributive Allegories}
\subsection{Distributive Allegories}

The component-wise definitions of the additional components bottom and join
make most of the required laws trivial:

%{{{ {The}Given a distributive allegory, a distr. allegory $Mat_{\catC}$
\begin{The}
If
$\catC = (Obj_{\catC}, Mor_{\catC}, \_ : \_ \rel \_, \RId{}, \rcmp, \RAconverse{}, \reland, \relor, \RO{})$
is a distributive allegory, then
$Mat_{\catC}$ may be extended to a distributive allegory by defining,
for any two objects
$\objA = [\objA_1,\ldots,\objA_a]$ and
$\objB = [\objB_1,\ldots,\objB_b]$,
\begin{itemize}
\item the zero morphism $\RO{\objA,\objB}:\objA\rel\objB$
as follows:
\BD
  (\RO{\objA,\objB})_{i,j} = \RO{\objA_i,\objB_j}
\qquad
\qquad
\mbox{for all $i \in \{1,\ldots,a\}$ and $j \in \{1,\ldots, b\}$,}
\ED
\item the join of any two morphisms $R, S:\objA\rel\objB$
component-wise:
\BD
  (R \relor S)_{i,j} = R_{i,j} \relor S_{i,j}
\qquad
\qquad
\mbox{for all $i \in \{1,\ldots,a\}$ and $j \in \{1,\ldots, b\}$.}
\ED
\end{itemize}
\end{The}
\Proof{The lattice properties of join and the zero law are trivial;
we only show join-distributivity:
\BDA
    (Q\rcmp(R\relor S))_{i,k}
  \sepA{=}
    \bigrelor_j\; (Q_{i,j}\rcmp (R\relor S)_{j,k})
  \sepB{=}
    \bigrelor_j\; (Q_{i,j}\rcmp (R_{j,k}\relor S_{j,k}))
  \sepB{=}
    \bigrelor_j\; (Q_{i,j}\rcmp R_{j,k}\relor Q_{i,j}\rcmp S_{j,k})
  \sepB{=}
    (\bigrelor_j\; (Q_{i,j}\rcmp R_{j,k}))\relor \bigrelor_j\; (Q_{i,j}\rcmp S_{j,k})
  \sepB{=}
    (Q \rcmp R)_{i,k} \relor (Q \rcmp S)_{i,k}
  \sepB{=}
    (Q \rcmp R \relor Q \rcmp S)_{i,k}
\EDAQ
}%Proof
%}}}

\bigskip
%{{{ bottomMat, bottomRow
Accordingly, defining bottom is easy:

\begin{code}
bottomMat da as bs = [bottomRow da a bs | a <- as]

bottomRow da a bs = map (distrAll_bottom da a) bs
\end{code}
%}}}

%{{{ atomMats, atomRows
To obtain a list of atoms --- we treat the global atom list and
the list of atoms contained in a given morphism in parallel ---
is, however, slightly more effort:

\begin{code}
atomMats da [] bs = []
atomMats da [a] bs = map (:[]) (atomRows da a bs)
atomMats da (a:as) bs =
  map (: (bottomMat da as bs)) (atomRows da a bs) ++
  map ((bottomRow da a bs) :) (atomMats da as bs)

atomsMats da [] bs _ = []
atomsMats da [a] bs [r] = map (:[]) (atomsRows da a bs r)
atomsMats da (a:as) bs (r:rs) =
  map (: (bottomMat da as bs)) (atomsRows da a bs r) ++
  map ((bottomRow da a bs) :) (atomsMats da as bs rs)
atomsMats _ _ _ _ = error "atomsMats"

atomRows da a []     = [[]]
atomRows da a [b] = map (:[]) (distrAll_atomset da a b)
atomRows da a (b:bs) =
  map (: (bottomRow da a bs)) (distrAll_atomset da a b) ++
  map ((distrAll_bottom da a b) :) (atomRows da a bs)

atomsRows da a [] _    = [[]]
atomsRows da a [b] [m] = map (:[]) (distrAll_atoms da m)
atomsRows da a (b:bs) (m:ms) =
  map (: (bottomRow da a bs)) (distrAll_atoms da m) ++
  map ((distrAll_bottom da a b) :) (atomsRows da a bs ms)
atomsRows _ _ _ _ = error "atomsRows"
\end{code}
%}}}

These definitions are not adequate if empty objects are involved,
so we have to treat these cases separately:

\index{distrAllMat@{\texttt{distrAllMat}}}%
%{{{ distrAllMat
\begin{code}
distrAllMat :: (Ord obj, Eq mor) => DistrAll obj mor -> [[obj]]
                              -> DistrAll (Vec obj) (MatMor obj mor)
distrAllMat da objss = DistrAll
  {distrAll_all  = allMat da objss
  ,distrAll_bottom = (\ (Vec s) (Vec t) ->
                   let mat = bottomMat da s t
                   in MatMor (mat, s, t))
  ,distrAll_join = (\ (MatMor (mss1,s1,t1)) (MatMor (mss2,s2,t2)) ->
                 if s1 /= s2
                 then error ("matrix join source type error")
                 else if t1 /= t2
                 then error ("matrix join target type error")
                 else let mat = zipWith (zipWith (distrAll_join da)) mss1 mss2
                      in MatMor (mat,s1,t1))
  ,distrAll_atomset = (\ (Vec s) (Vec t) ->
                         if null s || null t then [] else
                          map (\ m -> MatMor (m,s,t)) $ atomMats da s t)
  ,distrAll_atoms = (\ (MatMor (m, s, t)) ->
                       if null s || null t then [] else
                       map (\n -> MatMor (n,s,t)) $ atomsMats da s t m)
  }
\end{code}
%}}}
%}}}

%{{{ \subsection{Division Allegories}
\subsection{Division Allegories}

As we shall see, the definition of the residual coefficients is dual
to the definition of the composition coefficients.
It therefore relies on meet, and, for empty intermediate objects,
also on the presence of top as the unit of meet.

Since we do not want to differentiate between matrix algebras with and
without empty objects,
we therefore need coefficients from a Dedekind category:


%{{{ {The}Given a Dedekind category, a div. allegory $Mat_{\catC}$
\begin{The}
If
$\catC = (Obj_{\catC}, Mor_{\catC}, \_ : \_ \rel \_, \RId{}, \rcmp, \RAconverse{}, \reland, \relor, \RO{},\rres,\lres,\RL{})$
is a Dedekind category, then
$Mat_{\catC}$ may be extended to a division allegory by defining,
for any three objects
$\objA = [\objA_1,\ldots,\objA_a]$,
$\objB = [\objB_1,\ldots,\objB_b]$, and
$\objC = [\objC_1,\ldots,\objC_c]$, and
any three matrix morphisms
$Q:\objA \rel \objC$,
$R:\objA \rel \objB$, and 
$S : \objB \rel \objC$,
the coefficients of their residuals
in terms of the residuals of their coefficients:
\BDA
    (R\rres Q)_{j,k}
  \sepA{\defeq}
    \bigreland_i\; (R_{i,j}\rres Q_{i,k})
\sepC
    (Q\lres S)_{i,j}
  \sepA{\defeq}
    \bigreland_k\; (Q_{i,k}\lres S_{j,k})
\EDA
\end{The}
\Proof{We only carry out the proof for the left residual:
\BDA
    R \rsubs (Q\lres S)
  \sepA{\zeq}
    \forall i,j \dot R_{i,j} \rsubs (Q\lres S)_{i,j}
  \sepB{\zeq}
    \forall i,j \dot R_{i,j} \rsubs \bigreland_k\; (Q_{i,k}\lres S_{j,k})
  \sepB{\zeq}
    \forall i,j \dot \forall k \dot R_{i,j} \rsubs Q_{i,k}\lres S_{j,k}
  \sepB{\zeq}
    \forall i,j, k \dot R_{i,j}\rcmp S_{j,k} \rsubs Q_{i,k}
  \sepB{\zeq}
    \forall i, k \dot \forall j \dot R_{i,j}\rcmp S_{j,k} \rsubs Q_{i,k}
  \sepB{\zeq}
    \forall i, k \dot (\bigrelor_j \ (R_{i,j}\rcmp S_{j,k})) \rsubs Q_{i,k}
  \sepB{\zeq}
    \forall i, k \dot (R\rcmp S)_{i,k} \rsubs Q_{i,k}
  \sepB{\zeq}
    R\rcmp S \rsubs Q
\EDAQ
}%Proof
%}}}

We first define the common structure of both residuals
in an auxiliary function:

%{{{ skalres
\begin{code}
skalres :: Ded obj mor -> (mor -> mor -> mor) ->
                          (obj,[mor]) -> (obj,[mor]) -> mor
skalres d res (a,ms1) (b,ms2) =
   foldr (ded_meet d) (ded_top d a b) $
   zipWith res ms1 ms2
\end{code}
%}}}

The right residual now needs two transpositions to get the columns
lined up properly, while the left residual directly uses the rows.
For the symmetric quotient, we simply use the default definition:

\index{divAllMat@{\texttt{divAllMat}}}%
\begin{code}
divAllMat :: (Ord obj, Eq mor) => Ded obj mor -> [[obj]]
                            -> DivAll (Vec obj) (MatMor obj mor)
divAllMat d objss = diva where
 diva = DivAll 
  {divAll_distrAll = distrAllMat (ded_distrAll d) objss
  ,divAll_rres = (\ (m1@(MatMor (_,s1,t1))) m2@(MatMor (_ ,s2,t2)) ->
                    if s1 /= s2
                    then error ("matrix right residual type error " ++
                                show (length s1) ++ ' ' : show (length s2))
                    else let MatMor (mss1T,_,_) = transpose m1
                             MatMor (mss2T,_,_) = transpose m2
                             mss1C = zip t1 mss1T
                             mss2C = zip t2 mss2T
                             mkline ms1 = map (skalres d (ded_rres d) ms1) mss2C
                             mat = map mkline (zip t1 mss1T)
                         in MatMor (mat, t1, t2))
  ,divAll_lres = (\ (MatMor (mss1,s1,t1)) (MatMor (mss2,s2,t2)) ->
                    if t1 /= t2
                    then error ("matrix left residual type error " ++
                                show (length t1) ++ ' ' : show (length t2))
                    else let mss2C = zip s2 mss2
                             mkline ms1 = map (skalres d (ded_lres d) ms1) mss2C
                             mat = map mkline (zip s1 mss1)
                         in MatMor (mat, s1, s2))
  ,divAll_syq  = divAll_syq_default diva
  }
\end{code}
%}}}

%{{{ \subsection{Dedekind Categories and Relation Algebras}
\subsection{Dedekind Categories and Relation Algebras}

Since we already have a Dedekind category at the coefficient level,
getting the top morphism is now easy again,
and complement at the coefficient level is lifted component-wise to
yield matrix complements:

%{{{ {The}Given a Dedekind category, a div. allegory $Mat_{\catC}$
\begin{The}
If
$\catC = (Obj_{\catC}, Mor_{\catC}, \_ : \_ \rel \_, \RId{}, \rcmp, \RAconverse{}, \reland, \relor, \RO{},\rres,\lres,\RL{})$
is a Dedekind category, then
$Mat_{\catC}$ may be extended to a Dedekind category by defining,
for any two objects
$\objA = [\objA_1,\ldots,\objA_a]$ and
$\objB = [\objB_1,\ldots,\objB_b]$, the top morphism component-wise as follows:
\BD
    (\RL{\objA,\objB})_{i,j} = \RL{\objA_i,\objB_j}
\ED
Also, if
$\catC$
is a relation algebra, then
$Mat_{\catC}$ may be extended to a relation algebra by defining,
for any two objects
$\objA = [\objA_1,\ldots,\objA_a]$ and
$\objB = [\objB_1,\ldots,\objB_b]$, and
for any matrix morphism
$R:\objA \rel \objB$, the complement component-wise as follows:
\BD
    (\relnot{R})_{i,j} = \relnot{R_{i,j}}
\ED
\end{The}
\Proof{All remaining properties follow by simple component-wise reasoning.
A direct proof of the Schr\"oder rule is the following:
\BDAR
    R\rcmp S\rsubs Q
  \sepA{\zeq}
    \forall i,k \dot (R\rcmp S)_{i,k}\rsubs Q_{i,k}
  \sepB{\zeq}
    \forall i,k \dot\bigrelor_j\; (R_{i,j}\rcmp S_{j,k}) \rsubs Q_{i,k}
  \sepB{\zeq}
    \forall i,k,j \dot R_{i,j}\rcmp S_{j,k}\rsubs Q_{i,k}
  \sepBR{\zeq}{\mbox{Schr\"oder for coefficients}}
    \forall i,k,j \dot \rtrans{R}_{j,i}\rcmp \relnot{Q}_{i,k}\rsubs
    \relnot{S}_{j,k}
  \sepB{\zeq}
    \forall k,j \dot \bigrelor_i\;(\rtrans{R}_{j,i}\rcmp\relnot{Q}_{i,k})\rsubs \relnot{S}_{j,k}
  \sepB{\zeq}
    \forall k,j \dot (\rtrans{R}\rcmp\relnot{Q})_{j,k}\rsubs \relnot{S}_{j,k}
  \sepB{\zeq}
    \rtrans{R}\rcmp\relnot{Q} \rsubs \relnot{S}
\EDARQ
}%Proof
%}}}

The remaining Haskell definitions are therefore completely straightforward:

\index{dedMat@{\texttt{dedMat}}}%
\index{raMat@{\texttt{raMat}}}%
\begin{code}
dedMat :: (Ord obj, Eq mor) => Ded obj mor -> [[obj]]
                            -> Ded (Vec obj) (MatMor obj mor)
dedMat ded objss = Ded
  {ded_divAll = divAllMat ded objss
  ,ded_top  = (\ (Vec s) (Vec t) ->
                   let mat = [[ded_top ded a b | b <- t] | a <- s]
                   in MatMor (mat, s, t))
  }

raMat ra objss = RA
  {ra_ded   = dedMat (ra_ded ra) objss
  ,ra_compl = (\ (MatMor (m, s, t)) ->
                  MatMor (matMap (ra_compl ra) m, s, t))
  }
\end{code}
%}}}


%% Local variables:
%% folded-file: t
%% fold-internal-margins: 0
%% eval: (fold-set-marks "%{{{ " "%}}}")
%% eval: (fold-whole-buffer)
%% end:
