\section{Properties and Interesting Configurations}

We start the definition of tests with some very simple ones,
since it is important to look for all the details, too.

\begin{code}
module Properties where

import RelAlg
\end{code}

%{{{ \subsection{Simple Morphism Properties}
\subsection{Simple Morphism Properties}

The following tests for the categorical definition of monomorphisms%
\index{monomorphism}:

\index{cat_mono_TEST@{\texttt{cat\_mono\_TEST}}}%
%{{{ cat_mono_TEST
\begin{code}
cat_isMono :: Eq mor => Cat obj mor -> mor -> Bool
cat_isMono c m = noResults (cat_mono_TEST m) c

cat_mono_TEST :: Eq mor => mor -> Test Cat obj mor
cat_mono_TEST h c =
  let objects = cat_objects c
      homset = cat_homset c
      (^) = cat_comp c
      s = cat_source c h
      t = cat_target c h
  in ffold $ do
   a <- objects
   let homs = homset a s
   f <- homs
   g <- homs
   let fh = f ^ h
   let gh = g ^ h
   [test ((fh == gh) == (f == g)) [a,s,t] [f,g,h,fh,gh] "mono counterexample"]
\end{code}
%}}}

The dual then tests for epimorphisms%
\index{epimorphism}:

\index{cat_epi_TEST@{\texttt{cat\_epi\_TEST}}}%
%{{{ cat_epi_TEST
\begin{code}
cat_isEpi :: Eq mor => Cat obj mor -> mor -> Bool
cat_isEpi c m = noResults (cat_epi_TEST m) c

cat_epi_TEST :: Eq mor => mor -> Test Cat obj mor
cat_epi_TEST h c =
  let objects = cat_objects c
      homset = cat_homset c
      (^) = cat_comp c
      s = cat_source c h
      t = cat_target c h
  in ffold $ do
   c <- objects
   let homs = homset t c
   f <- homs
   g <- homs
   let hf = h ^ f
   let hg = h ^ g
   [test ((hf == hg) == (f == g)) [s,t,c] [h,f,g,hf,hg] "epi counterexample"]
\end{code}
%}}}

In allegories, we already have the usual relational definitions of
univalence\index{univalent},
totality\index{total},
injectivity\index{injective},
and surjectivity\index{surjective}:

\index{all_univalent_TEST@{\texttt{all\_univalent\_TEST}}}%
\index{all_injective_TEST@{\texttt{all\_injective\_TEST}}}%
\index{all_total_TEST@{\texttt{all\_total\_TEST}}}%
\index{all_surjective_TEST@{\texttt{all\_surjective\_TEST}}}%
\begin{code}
all_univalent_TEST m a =
  let mC = all_converse a m
      t = all_target a m
      iT = all_idmor a t
      mCm = all_comp a mC m
  in test (all_incl a mCm iT) [all_source a m, t] [m,mC,mCm,iT] "not univalent"

all_injective_TEST m a =
  let mC = all_converse a m
      s = all_source a m
      iS = all_idmor a s
      mmC = all_comp a m mC
  in test (all_incl a mmC iS) [s, all_target a m] [m,mC,mmC,iS] "not injective"

all_total_TEST m a =
  let mC = all_converse a m
      s = all_source a m
      iS = all_idmor a s
      mmC = all_comp a m mC
  in test (all_incl a iS mmC) [s, all_target a m] [m,mC,mmC,iS] "not total"

all_surjective_TEST m a =
  let mC = all_converse a m
      t = all_target a m
      iT = all_idmor a t
      mCm = all_comp a mC m
  in test (all_incl a iT mCm) [all_source a m, t] [m,mC,mCm,iT] "not surjective"
\end{code}

Of these, we also provide Boolean variants:

\index{all_isUnivalent@{\texttt{all\_isUnivalent}}}%
\index{all_isInjective@{\texttt{all\_isInjective}}}%
\index{all_isTotal@{\texttt{all\_isTotal}}}%
\index{all_isMapping@{\texttt{all\_isMapping}}}%
\index{all_isSurjective@{\texttt{all\_isSurjective}}}%
\begin{code}
all_isUnivalent  a m = noResults (all_univalent_TEST  m) a
all_isInjective  a m = noResults (all_injective_TEST  m) a
all_isTotal      a m = noResults (all_total_TEST      m) a
all_isSurjective a m = noResults (all_surjective_TEST m) a
all_isMapping    a m = all_isUnivalent a m && all_isTotal a m
\end{code}

Sometimes it may be interesting
what the non-trivial mappings in an allegory are;
here we offer an accordingly restricted version of \verb|all_homset|
and a function that collects all \emph{non-trivial} mappings of an allegory
into a \verb|TestResult|:

\index{all_mappings@{\texttt{all\_isMapping}}}%
\index{all_mapTest@{\texttt{all\_mapTest}}}%
\begin{code}
all_mappings a s t = let
 in filter (all_isMapping a) $ all_homset a s t

all_mapTest :: (Eq obj, Eq mor) => Test All obj mor
all_mapTest a = let
    objects = all_objects a
 in ffold $ do
     s <- objects
     t <- objects
     let noId = if s /= t then id else let i = all_idmor a s in filter (/= i)
     let ms = noId $ all_mappings a s t
     case ms of [] -> []
                _ -> [test False [s,t] ms "mappings"]
\end{code}

%{{{ functions, funTest
The same can be done for functions;
since we consider not only identities, but also empty relations
as trivial functions,
the test has a separate variant for distributive allegories:

\index{all_functions@{\texttt{all\_functions}}}%
\index{all_funTest@{\texttt{all\_funTest}}}%
\index{distrAll_funTest@{\texttt{distrAll\_funTest}}}%
\begin{code}
all_functions a s t = let
 in filter (all_isUnivalent a) $ all_homset a s t

all_funTest :: (Eq obj, Eq mor) => Test All obj mor
all_funTest a = let
    objects = all_objects a
 in ffold $ do
     s <- objects
     t <- objects
     let noId = if s /= t then id else let i = all_idmor a s in filter (/= i)
     let ms = noId $ all_functions a s t
     case ms of [] -> []
                _ -> [test False [s,t] ms "functions"]

distrAll_funTest :: (Eq obj, Eq mor) => Test DistrAll obj mor
distrAll_funTest a = let
    objects = distrAll_objects a
 in ffold $ do
     s <- objects
     t <- objects
     let noId = if s /= t then id
                          else let i = distrAll_idmor a s in filter (/= i)
     let ms = filter (/= (distrAll_bottom a s t)) $ noId
              $ all_functions (distrAll_all a) s t
     case ms of [] -> []
                _ -> [test False [s,t] ms "functions"]
\end{code}
%}}}

If $R : \objA \rel \objB$ is injective and total,
then $R \rcmp \rtrans{R} = \RId{A}$,
so $R$ is obviously mono.
It is also trivial that every mono has to be total.
But it is not so easy to see that, in general,
not every mono has to be injective,
so we write a quick test:

\index{all_mono_inj_TEST@{\texttt{all\_mono\_inj\_TEST}}}%
\begin{code}
all_mono_inj_TEST a =
  let objects = all_objects a
      homset = all_homset a
      isMono m = cat_isMono (all_cat a) m
      isTot m = all_isTotal a m
      isInj m = all_isInjective a m
  in ffold $ do
   s <- objects
   t <- objects
   m <- homset s t
   let mono = isMono m
   let inj = isInj m
   [test (mono <= inj) [s,t] [m] "mono, but not injective"]
\end{code}

A monomorphism\index{monomorphism} which is not injective\index{injective} is the following Boolean
$2\times 3$-matrix:

\smallskip
\centerline{\includegraphics{mono_not_inj.eps}}

\medskip
This is one of the smallest Boolean matrices with this property;
in most of the relation algebras of the third chapter,
all monomorphisms are injective.
%}}}

%{{{ \subsection{Homogeneous Relations}
\subsection{Homogeneous Relations}

We also provide a few tests for frequently-used properties
of homogeneous relations (all without checking for homogeneity):

\index{all_reflexive_TEST@{\texttt{all\_reflexive\_TEST}}}%
\index{all_coreflexive_TEST@{\texttt{all\_coreflexive\_TEST}}}%
\index{all_symmetric_TEST@{\texttt{all\_symmetric\_TEST}}}%
\index{all_transitive_TEST@{\texttt{all\_transitive\_TEST}}}%
\index{all_antisymmetric_TEST@{\texttt{all\_antisymmetric\_TEST}}}%
\begin{code}
all_reflexive_TEST m a =
  let s = all_source a m
      iS = all_idmor a s
  in test (all_incl a iS m) [s] [m,iS] "not reflexive"

all_coreflexive_TEST m a =
  let s = all_source a m
      iS = all_idmor a s
  in test (all_incl a m iS) [s] [m,iS] "not coreflexive"

all_symmetric_TEST m a =
  let mC = all_converse a m
  in test (all_incl a m mC) [all_source a m] [m,mC] "not symmetric"

all_transitive_TEST m a =
  let mm = all_comp a m m
  in test (all_incl a mm m) [all_source a m] [m,mm] "not transitive"

all_antisymmetric_TEST m a =
  let mC = all_converse a m
      x = all_meet a m mC
      s = all_source a m
      iS = all_idmor a s
  in test (all_incl a x iS) [s] [m,mC,x,iS] "not antisymmetric"
\end{code}

\index{all_order_TEST@{\texttt{all\_order\_TEST}}}%
\index{all_equivalence_TEST@{\texttt{all\_equivalence\_TEST}}}%
\index{all_preorder_TEST@{\texttt{all\_preorder\_TEST}}}%
\begin{code}
all_preorder_TEST    m a = all_reflexive_TEST m a . all_transitive_TEST m a
all_order_TEST       m a = all_preorder_TEST  m a . all_antisymmetric_TEST m a
all_equivalence_TEST m a = all_preorder_TEST  m a . all_symmetric_TEST m a
\end{code}

\index{all_isReflexive@{\texttt{all\_isReflexive}}}%
\index{all_isCoreflexive@{\texttt{all\_isCoreflexive}}}%
\index{all_isSymmetric@{\texttt{all\_isSymmetric}}}%
\index{all_isTransitive@{\texttt{all\_isTransitive}}}%
\index{all_isAntisymmetric@{\texttt{all\_isAntisymmetric}}}%
\index{all_isOrder@{\texttt{all\_isOrder}}}%
\index{all_isPreorder@{\texttt{all\_isPreorder}}}%
\index{all_isEquivalence@{\texttt{all\_isEquivalence}}}%
\begin{code}
all_isReflexive     a m = noResults (all_reflexive_TEST     m) a
all_isCoreflexive   a m = noResults (all_coreflexive_TEST   m) a
all_isSymmetric     a m = noResults (all_symmetric_TEST     m) a
all_isTransitive    a m = noResults (all_transitive_TEST    m) a
all_isAntisymmetric a m = noResults (all_antisymmetric_TEST m) a
all_isOrder         a m = noResults (all_order_TEST         m) a
all_isPreorder      a m = noResults (all_preorder_TEST      m) a
all_isEquivalence   a m = noResults (all_equivalence_TEST   m) a
\end{code}
%}}}

%{{{ \subsection{Uniformity}
\subsection{Uniformity}\subsectlabel{uniform}

\begin{Def}
A Dedekind category is called \emph{uniform}\index{uniform}
if for all objects $\objA$, $\objB$, and $\objC$ we have
\BD
    \RL{\objA,\objB}\rcmp\RL{\objB,\objC} = \RL{\objA,\objC}
\EDQ
\end{Def}

In heterogeneous relation algebras,
uniformity is implied by the Tarski rule\index{Tarski rule}.
It is, however, cheaper to test:

\index{ded_uniform_TEST@{\texttt{ded\_uniform\_TEST}}}%
\begin{code}
ded_uniform_TEST :: Eq mor => Test Ded obj mor
ded_uniform_TEST d =
  let objects = ded_objects d
      top = ded_top d
  in ffold $ do
       o1 <- objects
       o2 <- objects
       let t12 = top o1 o2
       o3 <- objects
       let t23 = top o2 o3
       let t13 = top o1 o3
       let t = ded_comp d t12 t23
       [test (t == t13) [o1,o2,o3] [t12,t23,t13,t] "non-uniform"]
\end{code}
%}}}

%{{{ \subsection{Units}
\subsection{Units}

According to \cite[2.15]{Freyd-Scedrov-1990}:

\begin{Def}
An object $\objU$ in an allegory is a \emph{partial unit}\index{partial unit}
if $\RId{\objU}$ is its maximum endomorphism.
$\objU$ is a \emph{unit}\index{unit} if, further,
every object is the source of a total morphism targeted at $\objU$.
An allegory is said to be \emph{unitary}\index{unitary} if it has a unit.
\qed
\end{Def}

Testing for partial units can be done in allegories,
but is (usually) much more efficient in Dedekind categories
where there is immediate access to the maximum morphisms:

\index{all_partialUnit@{\texttt{all\_partialUnit}}}%
\index{ded_partialUnit@{\texttt{ded\_partialUnit}}}%
%{{{ all_partialUnit, ded_partialUnit
\begin{code}
all_partialUnit_TEST :: obj -> Test All obj mor
all_partialUnit_TEST u a =
  let iU = all_idmor a u
      (<<==) = all_incl a
  in ffold (do m <- all_homset a u u
               [test (m <<== iU) [u] [iU,m] "identity is not maximal"])

ded_partialUnit_TEST :: Eq mor => obj -> Test Ded obj mor
ded_partialUnit_TEST u a =
  let iU = ded_idmor a u
      tU = ded_top a u u
  in (test (iU == tU) [u] [iU,tU] "identity is not maximal")
\end{code}
%}}}

For the unit test, we first of all need a totality test:

\index{ded_isTotal@{\texttt{ded\_isTotal}}}%
%{{{ all_isTotal
\begin{code}
ded_isTotal d = all_isTotal (ded_all d)
\end{code}
%}}}

Given a partial unit,
we can test whether it is a unit with the following test:

\index{all_partialUnit_unit_TEST@{\texttt{all\_partialUnit\_unit\_TEST}}}%
\begin{code}
all_partialUnit_unit_TEST :: obj -> Test All obj mor
all_partialUnit_unit_TEST u a =                -- Precondition: u is partial unit
  let objects = all_objects a
      homset = all_homset a
      check s = any (all_isTotal a) (homset s u)
  in ffold $ do
      s <- objects
      [test (check s) [s,u] [] "no total morphism to unit"]
\end{code}

We integrate this test directly into the unit search functions;
because of the different complexity of the partial unit test
we again provide this function both for allegories
and for Dedekind categories:

\index{all_units@{\texttt{all\_units}}}%
\index{ded_units@{\texttt{ded\_units}}}%
\begin{code}
all_units :: All obj mor -> [obj]
all_units a =
  let objects = all_objects a
      homset = all_homset a
      check u s = any (all_isTotal a) (homset s u)
      punit u = noResults (all_partialUnit_TEST u) a
      unit u = punit u && all (check u) objects
  in filter unit objects

ded_units :: Eq mor => Ded obj mor -> [obj]
ded_units a =
  let objects = ded_objects a
      homset = ded_homset a
      check u s = any (ded_isTotal a) (homset s u)
      punit u = noResults (ded_partialUnit_TEST u) a
      unit u = punit u && all (check u) objects
  in filter unit objects
\end{code}

%}}}

%{{{ \subsection{Tabulations}
\subsection{Tabulations}

According to \cite{Freyd-Scedrov-1990},
a pair $f,g$ of maps \emph{tabulates}\index{tabulation} a morphism $R$ iff
\BD
    \rtrans{f}\rcmp g = R
\qquad\mbox{and}\qquad
    f \rcmp \rtrans{f} \reland g \rcmp \rtrans{g} = \RId{}
\enskip.
\ED

Actually, it is sufficient to demand that $f$ and $g$ be univalent,
since the second condition implies their totality.

The heart of the tabulation test therefore has the following precondition:
$R: s \rel t$, $f: p \pfun s$, $g: p \pfun t$.

\index{is_tabulation@{\texttt{is\_tabulation}}}%
%{{{ is_tabulation
\begin{code}
is_tabulation :: (Eq mor) => All obj mor ->
                             obj -> obj -> mor ->
                             obj -> mor -> mor -> TestResult obj mor
is_tabulation a s t r p f g =
  let (^) = all_comp a
      (&&&) = all_meet a
      conv = all_converse a
      os = [s,t,p]
      fC = conv f
      gC = conv g
      fCg = fC ^ g
      cc = (f ^ fC) &&& (g ^ gC)
      ip = all_idmor a p
  in test (fCg == r) os [r,f,g,fCg]   "tabulation not correct"
   . test (cc == ip) os [r,f,g,cc,ip] "tabulation not precise"
\end{code}
%}}}

Since the non-standard algebras we are looking for
are certainly \emph{not} tabular,
we do not provide a test for tabularity.
%}}}

%{{{ \subsection{Direct Products}
\subsection{Direct Products}\sectlabel{DirectProducts}

It is well-known that the self-duality of categories of relations
implies that categorical sums are at the same time
categorical products ---
in relation algebras with sets and concrete relations,
categorical sums are disjoint unions.

However, Cartesian products can be axiomatised appropriately
on the relational level
\cite{Zierer-Schmidt-Berghammer-1986,Schmidt-Stroehlein-1993}:

%{{{ {Def} direct product
\begin{Def}\Deflabel{dirprod}
A \emph{direct product}\index{direct product} for two objects $\objA$ and $\objB$
is a triple $(\objP,\pi,\rho)$\index{pi@{$\pi$}}\index{rho@{$\rho$}}
consisting of an object $\objP$ and two \emph{projections}\index{projection},
i.e., relations
$\pi: \objP \rel \objA$ and $\rho : \objP \rel \objB$
for which the following conditions hold:
\BD
    \piT\rcmp\pi =\RId{}
\enskip,\qquad
    \rhoT\rcmp\rho =\RId{}
\enskip,\qquad
    \piT\rcmp\rho =\RL{}
\enskip,\qquad
    \pi\rcmp\piT \reland \rho\rcmp\rhoT = \RId{}
\enskip.
\EDQ
\end{Def}
%}}}

In our product data type, we explicitly mention all three objects
involved:

\index{Product@{\texttt{Product}}}%
\begin{code}
type Product obj mor = (obj,obj,obj,mor,mor)
\end{code}

The last two conditions for direct products
are equivalent to saying that
the projections tabulate $\RL{\objA,\objB}$,
and we use this in our test:

\index{ded_isNonemptyProduct@{\texttt{ded\_isNonemptyProduct}}}%
%{{{ ded_isNonemptyProduct
\begin{code}
ded_isNonemptyProduct :: (Eq obj, Eq mor) =>
    obj -> obj -> obj -> mor -> mor ->  Test Ded obj mor
ded_isNonemptyProduct a b p pA pB d =
  let alleg = ded_all d
      source = ded_source d
      target = ded_target d
      top = ded_top d
      (^) = ded_comp d
--      (&&&) = ded_meet d
      conv = ded_converse d
      pAC = conv pA
      pBC = conv pB
      idmor = ded_idmor d
      iA = idmor a
      iB = idmor b
--      iP = idmor p
  in
  test (source pA == p) [p,a] [pA] "inconsistent source of first projection" .
  test (target pA == a) [p,a] [pA] "inconsistent target of first projection" .
  test (source pB == p) [p,b] [pB] "inconsistent source of second projection" .
  test (target pB == a) [p,b] [pB] "inconsistent target of second projection" .
  test (pAC ^ pA == iA) [a,p] [pA,pAC,iA]
       "first projection not {univalent and surjective}" .
  test (pBC ^ pB == iB) [b,p] [pB,pBC,iB]
       "second projection not {univalent and surjective}" .
  is_tabulation alleg a b (top a b) p pA pB
--test (pAC ^ pB == top a b) [a,p,b] [pAC,pB] "non-comprehensive product" .
--test ((pA ^ pAC) &&& (pB ^ pBC) == iP) [a,p,b] [pA,pAC,pB,pBC,iP]
--     "product not {univalent and total}"
\end{code}
%}}}

For the sake of speed,
we use an integrated version of these tests
when searching for products;
we also demand an ordering on objects and only return products over
pairs of objects inside that ordering:

\index{ded_NonemptyProducts@{\texttt{ded\_NonemptyProducts}}}%
%{{{ ded_NonemptyProducts
\begin{code}
ded_NonemptyProducts :: (Eq obj, Ord obj, Eq mor) =>
                        Ded obj mor -> [Product obj mor]
ded_NonemptyProducts d =
  let objects = ded_objects d
      homset = ded_homset d
      top = ded_top d
      (^) = ded_comp d
      (&&&) = ded_meet d
      conv = ded_converse d
      idmor = ded_idmor d
  in do p <- objects
        let iP = idmor p
        a <- objects
        let iA = idmor a
        pA <- homset p a
        let pAC = conv pA
        if pAC ^ pA /= iA then []
         else do
          b <- objects
          let iB = idmor b
          pB <- homset p b
          let pBC = conv pB
          if b < a || pBC ^ pB /= iB then []
           else
            if (pAC ^ pB == top a b)
            && ((pA ^ pAC) &&& (pB ^ pBC) == iP)
            then [(a,b,p,pA,pB)]
            else []
\end{code}
%}}}

A simpler variant only checks whether two projections can be found
for a given triple of objects:
 
\index{ded_NonemptyProducts1@{\texttt{ded\_NonemptyProducts1}}}%
%{{{ ded_NonemptyProducts1
\begin{code}
ded_NonemptyProducts1 :: (Eq obj, Eq mor) => obj -> obj -> obj ->
                                             Ded obj mor -> [Product obj mor]
ded_NonemptyProducts1 a b p d =
  let homset = ded_homset d
      top = ded_top d
      (^) = ded_comp d
      (&&&) = ded_meet d
      conv = ded_converse d
      idmor = ded_idmor d
      iP = idmor p
      iA = idmor a
      iB = idmor b
  in do pA <- homset p a
        let pAC = conv pA
        if pAC ^ pA /= iA then []
         else do
          pB <- homset p b
          let pBC = conv pB
          if pBC ^ pB /= iB then []
           else
            if (pAC ^ pB == top a b)
            && ((pA ^ pAC) &&& (pB ^ pBC) == iP)
            then [(a,b,p,pA,pB)]
            else []
\end{code}
%}}}

For all direct products in relation algebras,
the following inclusion holds:
\BD
    P \rcmp R
    \reland
    Q\rcmp S
  \sepB{\rsups}
    (P\rcmp \piT \reland Q\rcmp\rhoT)\rcmp(\pi\rcmp R\reland \rho\rcmp S)  .
\ED
The opposite inclusion
\BD
    P \rcmp R
    \reland
    Q\rcmp S
  \sepB{\rsubs}
    (P\rcmp \piT \reland Q\rcmp\rhoT)\rcmp(\pi\rcmp R\reland \rho\rcmp S)
\ED
does not always hold. It is, however, trivial to prove it
in the context of relations in the classical sense.
The inability to prove it relation-algebraically first came up in 1981, 
when Rodrigo Cardoso prepared his diploma thesis \cite{Cardoso-1982}
under the supervision of the second-named author
who convinced himself that this might indeed be impossible,
who named it the \emph{sharpness condition}\index{sharpness},
and who conjectured that 
there might be ``unsharp''\index{unsharp} models of relation algebra.

For a relation algebra with an unsharp product,
together with its history, see \sectref{Maddux}.
Since the search for computationally relevant models with
unsharp products constitutes a main motivation for our current endeavour,
we need a test whether a given product is unsharp:

\index{ded_unsharp@{\texttt{ded\_unsharp}}}%
%{{{ unsharp
\begin{code}
ded_unsharp ::  (Eq obj, Eq mor) => Product obj mor -> Test Ded obj mor
ded_unsharp (a,b,_,pA,pB) d =
  let objects = ded_objects d
      homset = ded_homset d
      (^) = ded_comp d
      (&&&) = ded_meet d
      conv = ded_converse d
      pAC = conv pA
      pBC = conv pB
  in ffold (do
        x <- objects
        xA <- homset x a
        let xAP = xA ^ pAC
        xB <- homset x b
        let xBP = xB ^ pBC
        let xP = xAP &&& xBP
        y <- objects
        aY <- homset a y
        let xAY = xA ^ aY
        let pAY = pA ^ aY
        bY <- homset b y
        let xBY = xB ^ bY
        let pBY = pB ^ bY
        let pY = pAY &&& pBY
        let xPY = xP ^ pY
        let xY = xAY &&& xBY
        [test (xPY == xY) [x,y] [xA,xB,aY,bY,xPY,xY] "unsharpness example"]
       )        
\end{code}
%}}}
%}}}


%% Local variables:
%% folded-file: t
%% fold-internal-margins: 0
%% eval: (fold-set-marks "%{{{ " "%}}}")
%% eval: (fold-whole-buffer)
%% end: 


