
\section{Data Structures and Tests}\sectlabel{RelAlg}

%{{{ Intro
Although the multi-parameter-class interface presented in the last
section might look quite attractive at first sight,
it has several drawbacks.
Most obviously, the use of a non-standard extension to Haskell
brings about portability problems.
Also, since there is not yet a universally accepted design
for multi-parameter classes,
their use is always prone to future changes of supporting implementations.

%%% The second, perhaps even more serious drawback is
%%% that there may well be different categories using the same
%%% representations,
%%% so that the instance mechanism of Haskell is too coarse
%%% to differentiate properly.

The natural solution would be an ML-style module system.
Since this is not available in Haskell,
we resort to a translation of module types into record data types,
where types contained in the module become type parameters of the
record type constructor.

This approach makes the translation of the classes of the last section
into explicit \emph{dictionary records} straightforward.
We use prefixes to separate the name spaces,
and to ease a class-like use,
we explicitly import superclass members into subclasses
by straightforward selector composition.

We use abbreviated type names
in order to avoid name conflicts with the class names of the previous section,
since, in \sectref{Instances},
we are going to enable access to the constructions presented here
via those class interfaces.
In implementations
that support multi-parameter type classes with functional dependencies
(such as Hugs),
we can therefore seamlessly  integrate the class view of the last section
and the explicit dictionary view of this section.
%}}}

%{{{ \subsection{Preliminaries}
\subsection{Preliminaries}

This is the central module of our relation algebra library,
and there is nothing to hide here.
We do, however, import a few utilities
from Haskell's standard libraries,
on from our own prelude extensions {\tt ExtPrel} listed in \appref{ExtPrel}:

\begin{code}
module RelAlg where

import qualified IO(hFlush, stdout)
import Maybe(listToMaybe)
import ExtPrel(listEqAsSet)
\end{code}
%}}}

%{{{ \subsection{Testing}
\subsection{Testing}

We shall define numerous tests that allow to check
whether the structures we introduce are well-defined,
or whether certain laws hold or not.

In either case,
a negative result should indicate in which way the test failed,
so we define a uniform test result structure
that can hold all information for a single failure case
in the context of a single category or relation algebra:

\index{Instance@{\texttt{Instance}}}%
\begin{code}
type Instance obj mor = (String,[obj],[mor])
\end{code}

The semantics of such an \verb|Instance| does of course
heavily depend on its production site.
But we find that this is detailed and simple enough
both for the test programmer and for the test user.

We display \verb|Instance|s with every component on a line of its own:

\index{showsInstance@{\texttt{showsInstance}}}%
\index{showInstance@{\texttt{showInstance}}}%
\begin{code}
showsInstance :: (Show obj, Show mor) => Instance obj mor -> ShowS
showsInstance (s,os,ms) r = foldr (\s' r' -> s' ++ '\n' : r') r
   (s : case os of [] -> []
                   [o] -> [" Object: " ++ show o]
                   _ -> " Objects:"   : map (indent . show) os
     ++ case ms of [] -> []
                   [m] -> [" Morphism: " ++ show m]
                   _ -> " Morphisms:" : map (indent . show) ms)
  where indent s' = "  " ++ s'

showInstance :: (Show obj, Show mor) => Instance obj mor -> String
showInstance i = showsInstance i ""
\end{code}

%{{{ diag
%%% To liven up test sequences, we may sometimes use diagonalisation,
%%% in the following integer-less variant:
%%% 
%%% \begin{code}
%%% diag :: [[a]] -> [a]
%%% diag [] = []
%%% diag lists = f id lists where
%%%   f :: ([[a]] -> [[a]]) -> [[a]] -> [a]
%%%   f a [] = split id (a []) []
%%%   f a (l:ls) = split id (a [l]) ls
%%%   split :: ([[a]] -> [[a]]) -> [[a]] -> [[a]] -> [a]
%%%   split a [] [] = diag (a [])
%%%   split a [] r = f a r
%%%   split a ([] : ls) r = split a ls r
%%%   split a ((x:xs) : ls) r = x : split (a . (xs :)) ls r
%%% \end{code}
%%% 
%%% This allows us to easily obtain nicely mixed sequences of tuples:
%%% 
%%% \begin{code}
%%% listProd :: [a] -> [b] -> [[(a,b)]]
%%% listProd as bs = [[(a,b) | b <- bs ] | a <- as]
%%% 
%%% pairs :: [a] -> [(a,a)]
%%% pairs as = diag (listProd as as)
%%% 
%%% triples :: [a] -> [(a,a,a)]
%%% triples as = map (\(a,(b,c)) -> (a,b,c)) $ diag (listProd as (pairs as))
%%% 
%%% quadruples :: [a] -> [(a,a,a,a)]
%%% quadruples as = map (\(a,(b,c,d)) -> (a,b,c,d)) $ diag (listProd as (triples as))
%%% \end{code}
%}}}

A simplistic approach would let individual test cases
produce results of type \verb|[Instance]| and then concatenate these
to the complete test result.
Since concatenation may incur quadratic running time costs,
we use the standard technique
to replace concatenation with function composition
and let individual test cases return results of the following type
(in analogy to the prelude type \verb|ShowS = String -> String|):

\index{TestResult@{\texttt{TestResult}}}%
\begin{code}
type TestResult obj mor = [Instance obj mor] -> [Instance obj mor]
\end{code}

As in the case of \verb|ShowS|,
function composition now acts as a low-cost binary concatenation operator on
expressions of type \verb|TestResult|.

Typically, we shall generate \verb|TestResult|s via the following function:

\index{test@{\texttt{test}}}%
\begin{code}
test :: Bool -> [obj] -> [mor] -> String -> TestResult obj mor
test b os ms s = \ is -> if b then is else (s,os,ms) : is
\end{code}

Sometimes, however, presence of a result is an indication that certain
other tests need not be performed;
for these circumstances we provide a variant operating on
lazy \verb|TestResult| lists:

\index{testX@{\texttt{testX}}}%
\begin{code}
testX :: Bool -> [obj] -> [mor] -> String ->
         [TestResult obj mor] -> [TestResult obj mor]
testX b os ms s = \ crs -> if b then crs else [((s,os,ms):)]
\end{code}

As in the case of \verb|ShowS|,
functions of type \verb|TestResult| never inspect their argument,
but return it with maybe some additional \verb|Instance|s
consed onto its beginning.

For testing any individual property,
usually a whole list of \verb|TestResult|s is produced,
and we concatenate them with the following instance of \verb|foldr|:

\index{ffold@{\texttt{ffold}}}%
\begin{code}
ffold :: [a -> a] -> a -> a
ffold l r = foldr id r l
\end{code}

In our tests, we then use this at the type

\begin{center}
\verb|ffold :: [TestResult obj mor] -> TestResult obj mor|.
\end{center}

For tests that check structures like categories or relation algebras
for consistency or for occurrence of certain special configurations,
we then may use the following type,
where \verb|s| is a binary type constructor:

\index{Test@{\texttt{Test}}}%
\begin{code}
type Test s obj mor = s obj mor -> TestResult obj mor
\end{code}

The most frequent use of tests will be to \emph{perform} them interactively
for inspecting the results:

\index{perform@{\texttt{perform}}}%
\begin{code}
perform :: (Show obj, Show mor) => Test c obj mor -> c obj mor -> IO ()
perform t c = printTestResults (t c)
\end{code}

Since the output for every instance may be quite verbose,
we currently only output the first three test result \verb|Instance|s:

\index{printTestResults@{\texttt{printTestResults}}}%
\begin{code}
printTestResults :: (Show obj, Show mor) => TestResult obj mor -> IO ()
printTestResults t = case map showsInstance $ t [] of
                       [] -> putStrLn "No results."
                       l -> putStr $ ffold (take 3 l) ""
\end{code}

For situations where all results are needed, we also provide:

\index{performAll@{\texttt{performAll}}}%
\index{printAllTestResults@{\texttt{printAllTestResults}}}%
\begin{code}
performAll :: (Show obj, Show mor) => Test c obj mor -> c obj mor -> IO ()
performAll t c = printAllTestResults (t c)

printAllTestResults :: (Show obj, Show mor) => TestResult obj mor -> IO ()
printAllTestResults t = do
  putStrLn "=== Test Start ==="
  mapM_ (putStrFlush . showInstance) (t [])
  putStrLn "=== Test End   ==="

putStrFlush s = putStr s >> IO.hFlush IO.stdout
\end{code}

Sometimes, however, we are only interested whether there are any results
or not:

\index{noResults@{\texttt{noResults}}}%
\begin{code}
noResults :: Test c obj mor -> c obj mor -> Bool
noResults t c = null (t c [])
\end{code}

%}}}

%{{{ \subsection{Categories}
\subsection{Categories}

We now turn to the data structures representing categories etcetera.
We define them as record data types,
using as field labels (i.e.\null{} also as selector functions)
the corresponding method names
prefixed with a lower-case variant of the type name
(which we also use as the constructor name):

\index{Cat@{\texttt{Cat}}}%
\index{cat*@{\texttt{cat\_*}}}%
\begin{code}
data Cat obj mor = Cat
  {cat_isObj   :: obj -> Bool
  ,cat_isMor   :: obj -> obj -> mor -> Bool
  ,cat_objects :: [obj]
  ,cat_homset  :: obj -> obj -> [mor]
  ,cat_source  :: mor -> obj
  ,cat_target  :: mor -> obj
  ,cat_idmor   :: obj -> mor
  ,cat_comp    :: mor -> mor -> mor
  }
\end{code}

We organise the consistency test for categories into four groups:
\begin{enumerate}
\item One object: Consistency of object list and of identity as a morphism
\item Two objects, one morphism:
   Consistency of morphism list, identity properties
\item Three objects, two morphisms: Well-definedness of composition
\item Four objects, three morphisms: Associativity of composition
\end{enumerate}
We generate the result lists via \verb|do| expressions in the list monad;
with respect to list comprehension this has the advantage that local variables
are introduced \emph{before} they are used.
Since \verb|return| in the list monad is just the singleton function,
we usually directly write singletons instead of return
since this saves space and serves as an additional reminder that the
\verb|do| expressions are in the list monad.

Keeping the tests in separate
\verb|do| expressions has the advantage of better readability,
and also the advantage that different failures of one property
are grouped closer together.
However it incurs a slight runtime cost.
Later we will usually join the tests of different complexity
into nested \verb|do| expressions.
Then, failures will be grouped
essentially according to the objects and morphisms involved in them.

All the tests included in this report
are decision procedures\index{decision procedure} for finite categories.
Although it is perfectly possible
to use diagonalisation\index{diagonalisation}
to obtain semi-decision procedures\index{semi-decision procedures}
for countable categories,
the overhead would incur significant running-time and readability costs
for the finite case,
which is the case we are interested in.

\index{cat_TEST@{\texttt{cat\_TEST}}}%
\begin{code}
cat_TEST :: (Eq obj, Eq mor) => Test Cat obj mor
cat_TEST c =
  let isObj = cat_isObj c
      isMor = cat_isMor c
      objs = cat_objects c
      homset = cat_homset c
      source = cat_source c
      target = cat_target c
      idmor = cat_idmor c
      (^) = cat_comp c
  in ffold (let a1 = "identity "
                a2 = a1 ++ "has inconsistent " in
            do o <- objs
               let i = idmor o
               [test (isObj  o     ) [o] [] "object list contains non-object" .
                test (source i == o) [o] [i] (a2 ++ "source") .
                test (target i == o) [o] [i] (a2 ++ "target") .
                test (isMor  o o  i) [o] [i] (a1 ++ "is non-morphism")]
           ) .
     ffold (let a1 = "homset contains "
                a2 = a1 ++ "morphism with inconsistent " in
            do s <- objs
               let sId = idmor s
               t <- objs
               let os = [s,t]
               let tId = idmor t
               m <- homset s t
               [test (source m == s) os [m] (a2 ++ "source") .
                test (target m == t) os [m] (a2 ++ "target") .
                test (isMor s t m  ) os [m] (a1 ++ "non-morphism") .
                test (sId ^ m == m) os [sId,m] "left-identity violated" .
                test (m ^ tId == m) os [m,tId] "right-identity violated"]
           ) .
     ffold (let a1 = "composition yields "
                a2 = a1 ++ "morphism with inconsistent " in
            do o1 <- objs
               o2 <- objs
               f <- homset o1 o2
               o3 <- objs
               let os = [o1,o2,o3]
               g <- homset o2 o3
               let m = f ^ g
               let ms = [f,g,m]
               [test (source m == o1) os ms (a2 ++ "source") .
                test (target m == o3) os ms (a2 ++ "target") .
                test (isMor o1 o3 m ) os ms (a1 ++ "non-morphism")]
           ) .
     ffold (do o1 <- objs
               o2 <- objs
               f <- homset o1 o2
               o3 <- objs
               g <- homset o2 o3
               let fg = f ^ g
               o4 <- objs
               let os = [o1,o2,o3,o4]
               h <- homset o3 o4
               let gh = g ^ h
               let k1 = f ^ gh
               let k2 = fg ^ h
               [test (k1 == k2) os [f,g,h,fg,gh,k1,k2]
                     "non-associative composition"    ]
           )
\end{code}
%}}}

%{{{ \subsection{Functors}
\subsection{Functors}

Functors are category homomorphisms and therefore an important tool
for establishing relations between different categories.
Unfortunately, the prelude defines
\index{Functor@{\texttt{Functor}}}%
``{\tt Functor}'' as class name
for endofunctors in the category of Haskell types and Haskell functions ---
we resolve the name clash
with the prelude class \verb|Functor|
by using the abbreviation {\tt Fun}.

Furthermore, for the time being
we want to work with the \verb|Test| datatype from above,
and we want the objects and morphisms of the source category to appear in the
\verb|TestResult|s ---
this determines the reversed order of the type arguments to the
\verb|Fun| type constructor:

\index{Fun@{\texttt{Fun}}}%
\begin{code}
data Fun obj2 mor2 obj1 mor1 = Fun
  {fun_obj :: obj1 -> obj2
  ,fun_mor :: mor1 -> mor2
  }
\end{code}

Since we align the direction of functor composition
with the direction of our categorical composition,
the twisted type of functors
``recovers'' the usual twisted type of composition:

\index{funcomp@{\texttt{funcomp}}}%
\index{$$$@{\texttt{\$\$\$}}}%
\begin{code}
funcomp :: Fun obj2 mor2 obj1 mor1 ->
           Fun obj3 mor3 obj2 mor2 ->
           Fun obj3 mor3 obj1 mor1
Fun fo1 fm1 `funcomp` Fun fo2 fm2  =  Fun (fo1 $$$ fo2) (fm1 $$$ fm2)

($$$) :: (a -> b) -> (b -> c) -> (a -> c)
f $$$ g = \ x -> g (f x)
\end{code}

Testing whether a functor data structure does indeed represent a functor
is divided into three steps:
\begin{enumerate}
\item One object: Testing well-formedness of the object mapping,
  and preservation of identities
\item Two objects, one morphism: Testing well-formedness of the morphism mapping
\item Three objects, two morphisms: Testing preservation of composition.
\end{enumerate}

\index{functor_TEST@{\texttt{functor\_TEST}}}%
%{{{ functor_TEST
\begin{code}
functor_TEST :: Eq mor2 => Cat obj1 mor1 -> Cat obj2 mor2 ->
                           Test (Fun obj2 mor2) obj1 mor1
functor_TEST c1 c2 fun =
  let objects1 = cat_objects c1
      homset1 = cat_homset c1
      idmor1 = cat_idmor c1
      (^) = cat_comp c1

      isObj2 = cat_isObj c2
      isMor2 = cat_isMor c2
      idmor2 = cat_idmor c2
      (^^) = cat_comp c2

      fo = fun_obj fun
      fm = fun_mor fun
  in ffold (do
   s1 <- objects1
   let s2 = fo s1
   let is1 = idmor1 s1
   let is2 = idmor2 s2
   testX (isObj2 s2) [s1] [] "functor yields non-object" $
    [test (fm is1 == is2) [s1] [is1] "functor does not preserve identity"]
  ) . ffold (do
   s1 <- objects1
   let s2 = fo s1
   t1 <- objects1
   let t2 = fo t1
   f1 <- homset1 s1 t1
   [test (isMor2 s2 t2 (fm f1)) [s1, t1] [f1] "functor yields non-morphism"]
  ) . ffold (do
   s1 <- objects1
   t1 <- objects1
   f1 <- homset1 s1 t1
   let f2 = fm f1
   u1 <- objects1
   g1 <- homset1 t1 u1
   let g2 = fm g1
   let h1 = f1 ^ g1
   [test (fm h1 == (f2 ^^ g2)) [s1,t1,u1] [f1,g1]
         "functor does not preserve composition" ]
  )
\end{code}
%}}}

We also implement a straightforward test
for checking whether some other functor \verb|f2|
is right-inverse with respect to \verb|`funcomp`|
to the test argument \verb|f1|:

\index{functor_rightinv_test@{\texttt{functor\_rightinv\_test}}}%
%{{{ functor_rightinv_test
\begin{code}
functor_rightinv_test :: (Eq mor1, Eq obj1) =>
                         Cat obj1 mor1 -> Cat obj2 mor2 ->
                         Fun obj1 mor1 obj2 mor2 ->
                         Test (Fun obj2 mor2) obj1 mor1
functor_rightinv_test c1 c2 f2 f1 =
  let objects1 = cat_objects c1
      homset1 = cat_homset c1
      fo1 = fun_obj f1
      fo2 = fun_obj f2
      fm1 = fun_mor f1
      fm2 = fun_mor f2
  in ffold (do
   o1 <- objects1
   let o2 = fo1 o1
   let o1a = fo2 o2
   [test (o1 == o1a) [o1,o1a] [] "not right-inverse on objects"]
  ) . ffold (do
   o1 <- objects1
   o2 <- objects1
   f1 <- homset1 o1 o2
   let f2 = fm1 f1
   let f1a = fm2 f2
   [test (f1 == f1a) [o1,o2] [f1,f1a] "not right-inverse on morphisms"]
  )   
\end{code}
%}}}
%}}}

%{{{ \subsection{Allegories}
\subsection{Allegories}

We already mentioned that records like those of the \verb|Cat obj mor|
datatype correspond to method dictionaries.
We now proceed to define the first subclass of categories,
and we include the \verb|Cat| dictionary as first entry
in the subclass dictionary, or, mathematically,
explicitly include the base category as such:

\index{all*@{\texttt{all\_*}}}%
\index{All@{\texttt{All}}}%
\begin{code}
data All obj mor = All
  {all_cat  :: Cat obj mor
  ,all_converse :: mor -> mor
  ,all_meet     :: mor -> mor -> mor
  ,all_incl     :: mor -> mor -> Bool
  }
\end{code}

For transparent access to all parts of the mathematical structure
(corresponding to the flat tuples in the definitions)
we transfer the superclass methods into the subclass
via composition with the superclass dictionary selector:

\index{all*@{\texttt{all\_*}}}%
\begin{code}
all_isObj   = cat_isObj   . all_cat      -- :: obj -> Bool
all_isMor   = cat_isMor   . all_cat      -- :: obj -> obj -> mor -> Bool
all_objects = cat_objects . all_cat      -- :: [obj]
all_homset  = cat_homset  . all_cat      -- :: obj -> obj -> [mor]
all_source  = cat_source  . all_cat      -- :: mor -> obj
all_target  = cat_target  . all_cat      -- :: mor -> obj
all_idmor   = cat_idmor   . all_cat      -- :: obj -> mor
all_comp    = cat_comp    . all_cat      -- :: mor -> mor -> mor
\end{code}

The consistency tests for allegories are organised into two large groups:
\begin{enumerate}
\item Two objects:
  \begin{enumerate}
  \item One morphism: Consistency of converse and idempotency of meet
  \item Two morphisms: Consistency and commutativity of meet,
    monotony of converse and consistency with meet
  \item Three morphisms: Associativity and sub-distributivity of meet
  \end{enumerate}
\item Three objects:
  \begin{enumerate}
  \item Two morphisms: Preservation of composition by converse
  \item Three morphisms: modal rule
  \end{enumerate}
\end{enumerate}
%
The tests that are commented out
are for properties that are implied by the other tests.

\index{all_TEST@{\texttt{all\_TEST}}}%
%{{{ all_TEST
\begin{code}
all_TEST :: (Eq obj, Eq mor) => Test All obj mor
all_TEST c =
  let (^) = all_comp c
      conv = all_converse c
      (<<==) = all_incl c
      (&&&) = all_meet c
      homset = all_homset c
      objs = all_objects c
      idmor = all_idmor c
      convNPres = "converse does not preserve "
  in -- ffold (do s <- objs
     --          let i = idmor s
     --          let i' = conv i
     --          [test (i == i') [s] [i,i'] (convNPres ++ "identity")]
     --       ) .
     ffold (let c1 = "converse yields "
                c2 = c1 ++ "morphism with inconsistent "
                a1 = "meet yields "
                a2 = a1 ++ "morphism with inconsistent "
                a3 = "meet is not " in
            do s <- objs
               t <- objs
               let os = [s,t]
               f <- homset s t
               let fC = conv f
               let ms_C = [f,fC]
               (test (all_source c fC == t) os ms_C (c2 ++ "source") .
                test (all_target c fC == s) os ms_C (c2 ++ "target") .
                test (all_isMor c t s fC  ) os ms_C (c1 ++ "non-morphism") .
                let fCC = conv fC in
                test (fCC == f)    os (ms_C++[fCC]) (c1 ++ "no involution") .
                let f' = f &&& f in
                test (f == f')            os [f,f'] (a3 ++ "idempotent")
                ) : do
                 g <- homset s t
                 let gC = conv g
                 let m = f &&& g
                 let m' = g &&& f
                 let ms = [f,g,m]
                 let mC = conv m
                 let cm = fC &&& gC
                 (test (all_source c m == s) os ms (a2 ++ "source") .
                  test (all_target c m == t) os ms (a2 ++ "target") .
                  test (all_isMor c s t m)   os ms (a1 ++ "non-morphism") . 
                  test (m == m')   os (ms ++ [m']) (a3 ++ "commutative") .
                  test ((f == m) == (f <<== g)) os ms
                       (a1 ++ "inclusion inconsistency") .
                  test (mC == cm) os (ms_C++[g,gC,mC,cm]) (convNPres ++ "meet")
--                test ((fC <<== gC) == (f <<== g)) os (ms_C++[gC])
--                     "non-monotone conversion"
                  ) : do
                   h <- homset s t
                   let m1 = m &&& h
                   let ms1 = [f,g,m,h,m1]
                   [let m2  =       g &&& h
                        m2' = f &&& m2      in
                    test (m2' == m1) os (ms1 ++ [m2,m2']) (a3 ++ "associative")
                    ]
                  ++ do
                   o3 <- objs
                   k <- homset o3 s
                   let kf = k ^ f
                   let kg = k ^ g
                   let km = k ^ m
                   let mk = kf &&& kg
                   [test (km <<== mk) (o3:os) [k,f,g,km,mk]
                         "meet-subdistributivity violated" ]
           ) .
     ffold (do o1 <- objs
               o2 <- objs
               o3 <- objs
               let os = [o1,o2,o3]
               g <- homset o1 o2
               let gC = conv g
               h <- homset o2 o3
               let gh = g ^ h
               let ghC = conv gh
               let hC = conv h
               let hCgC = hC ^ gC
               test (ghC == hCgC) os [g,h,ghC,hCgC] "converse is no functor" : do
                 f <- homset o1 o3
--               [test ((f &&& gh) <<== (((f ^ hC) &&& g) ^ ((gC ^ f) &&& h)))
--                     os [f,g,h] "Dedekind violation"]
                 [test ((f &&& gh) <<== (g ^ ((gC ^ f) &&& h)))
                       os [f,g,h] "violation of modal rule"]
           )
\end{code}
%}}}

\cite{Freyd-Scedrov-1990} define a \emph{representation of allegories}
to be a functor that preserves converse and meet ---
preservation of meet implies monotony.

Other sources, including \cite{Bird-deMoor-1997},
define a \emph{relator} to be
a monotone functor between \emph{tabular} allegories ---
there, monotony implies preservation of converse.

Since we are particularly interested in non-tabular allegories,
we still employ the name of the latter, but define:

\begin{Def}
A \emph{relator}\index{relator}
is a monotone functor between allegories that preserves converse.
A \emph{representation of allegories}\index{representation of allegories}
is a relator that also preserves meets.
\qed
\end{Def}

\index{relator_TEST@{\texttt{relator\_TEST}}}%
%{{{ relator_TEST
\begin{code}
relator_TEST, allrepr_TEST :: Eq mor2 => All obj1 mor1 -> All obj2 mor2 ->
                                         Test (Fun obj2 mor2) obj1 mor1
relator_TEST = relator_TEST_frame False
allrepr_TEST = relator_TEST_frame True

relator_TEST_frame ::  Eq mor2 => Bool -> All obj1 mor1 -> All obj2 mor2 ->
                                          Test (Fun obj2 mor2) obj1 mor1
relator_TEST_frame allrepr c1 c2 fun =
  let objects1 = all_objects c1
      homset1 = all_homset c1
      (&&&) = all_meet c1
      (&&&&) = all_meet c2
      conv1 = all_converse c1
      conv2 = all_converse c2
      fo = fun_obj fun
      fm = fun_mor fun
      ident = if allrepr then "allegory representation" else "relator"
      message s = ident ++ " does not preserve " ++ s
  in ffold $ do
   s1 <- objects1
   t1 <- objects1
   let os = [s1,t1]
   f1 <- homset1 s1 t1
   let f2 = fm f1
   let f1C = conv1 f1
   let f2C = conv2 f2
   test (f2C == fm f1C) os [f1,f1C] (message "converse")
    : (do g1 <- homset1 s1 t1
          let g2 = fm g1
          if allrepr
           then let h1 = f1 &&& g1
                in [test (fm h1 == (f2 &&&& g2)) os [f1,g1,h1] (message "meet")]
           else let b = all_incl c2 f2 g2 || not (all_incl c1 f1 g1)
                in [test b os [f1,g1] (message "inclusion")]
      )
\end{code}
%}}}

We shall sometimes need to test whether two allegories are equivalent;
for this we assemble all relevant tests,
creating a pair of \verb|TestResult|s of different types:

\index{all_equiv_TESTS@{\texttt{all\_equiv\_TESTS}}}%
\begin{code}
all_equiv_TESTS :: (Eq obj1, Eq mor1, Eq obj2, Eq mor2) =>
   All obj1 mor1 -> All obj2 mor2 ->
   Fun obj2 mor2 obj1 mor1 -> Fun obj1 mor1 obj2 mor2 ->
   (TestResult obj1  mor1, TestResult obj2 mor2)
all_equiv_TESTS a1 a2 f1 f2 =
  let c1 = all_cat a1
      c2 = all_cat a2
  in (functor_TEST c1 c2 f1 .
      allrepr_TEST a1 a2 f1 .
      functor_rightinv_test c1 c2 f2 f1
     ,functor_TEST c2 c1 f2 .
      allrepr_TEST a2 a1 f2 .
      functor_rightinv_test c2 c1 f1 f2
     )
\end{code}

This nave procedure tends, however, to bind too much space;
therefore we also define the corresponding sequence of \verb|perform| actions:

\index{all_equiv_perform@{\texttt{all\_equiv\_perform}}}%
\begin{code}
all_equiv_perform a1 a2 f1 f2 =
  let c1 = all_cat a1
      c2 = all_cat a2
  in do perform (functor_TEST c1 c2) f1
        perform (functor_TEST c2 c1) f2
        perform (allrepr_TEST a1 a2) f1
        perform (allrepr_TEST a2 a1) f2
        perform (\ f1 -> functor_rightinv_test c1 c2 f2 f1) f1
        perform (\ f2 -> functor_rightinv_test c2 c1 f1 f2) f2
\end{code}
%}}}

%{{{ \subsection{Distributive Allegories}
\subsection{Distributive Allegories}

Since all finite partial orders with a least element contain atoms,
we include access to the atoms
already in the interface of distributive allegories.
Since this is a derived concept,
we shall provide default definitions below.

\index{DistrAll@{\texttt{DistrAll}}}%
\index{distrAll*@{\texttt{distrAll\_*}}}%
\index{atomset@{\texttt{atomset}}|see{\texttt{distrAll\_atomset}}}%
\index{distrAll_atomset@{\texttt{distrAll\_atomset}}}%
\index{atoms@{\texttt{atoms}}|see{\texttt{distrAll\_atoms}}}%
\index{distrAll_atoms@{\texttt{distrAll\_atoms}}}%
\begin{code}
data DistrAll obj mor = DistrAll
  {distrAll_all     :: All obj mor
  ,distrAll_bottom  :: obj -> obj -> mor
  ,distrAll_join    :: mor -> mor -> mor
  ,distrAll_atomset :: obj -> obj -> [mor]
  ,distrAll_atoms   :: mor -> [mor]
  }
\end{code}

We introduce an abbreviation that allows to directly access
the bottom relation from the homset of a given morphism:

\index{bot@{\texttt{bot}}|see{\texttt{distrAll\_bot}}}%
\index{distrAll_bot@{\texttt{distrAll\_bot}}}%
\begin{code}
distrAll_bot da f = let s = distrAll_source da f
                        t = distrAll_target da f
                    in distrAll_bottom da s t
\end{code}

\index{distrAll*@{\texttt{distrAll\_*}}}%
\begin{code}
distrAll_isObj    = cat_isObj    . distrAll_cat   -- :: obj -> Bool
distrAll_isMor    = cat_isMor    . distrAll_cat   -- :: obj -> obj -> mor -> Bool
distrAll_objects  = cat_objects  . distrAll_cat   -- :: [obj]
distrAll_homset   = cat_homset   . distrAll_cat   -- :: obj -> obj -> [mor]
distrAll_source   = cat_source   . distrAll_cat   -- :: mor -> obj
distrAll_target   = cat_target   . distrAll_cat   -- :: mor -> obj
distrAll_idmor    = cat_idmor    . distrAll_cat   -- :: obj -> mor
distrAll_comp     = cat_comp     . distrAll_cat   -- :: mor -> mor -> mor

distrAll_cat      = all_cat      . distrAll_all
distrAll_converse = all_converse . distrAll_all   -- :: mor -> mor
distrAll_meet     = all_meet     . distrAll_all   -- :: mor -> mor -> mor
distrAll_incl     = all_incl     . distrAll_all   -- :: mor -> mor -> Bool
\end{code}

Testing distributive allegories is organised in the following way:
\begin{enumerate}
\item Two objects, one morphism:
   Bottom consistency inside one homset, idempotency of join
\item Three objects one morphism: Zero law
\item Two objects, two morphisms: Homset closed under join,
  commutativity and absorption laws
\item Two objects, three morphisms: Associativity of join, lattice distributivity
\item Three objects, two morphisms: Distributivity of composition over join
\end{enumerate}

\index{distrAll_TEST@{\texttt{distrAll\_TEST}}}%
%{{{ distrAll_TEST
\begin{code}
distrAll_TEST :: (Eq obj, Eq mor) => Test DistrAll obj mor
distrAll_TEST c = 
  let objects = distrAll_objects c
      (&&&) = distrAll_meet c
      (|||) = distrAll_join c
      (<<==) = distrAll_incl c
      (^) = distrAll_comp c
      homset = distrAll_homset c
      isMor = distrAll_isMor c
      bottom = distrAll_bottom c
      a1 = "join yields "
      a2 = a1 ++ "morphism with inconsistent "
      a3 = "join is not "
  in ffold $ do
      s <- objects
      t <- objects
      let os = [s,t]
      let bot = bottom s t
      testX (isMor s t bot) [s,t] [bot] "bottom is non-morphism"
       $ do
        f <- homset s t
        let f' = f ||| f
        (test (bot <<== f) os [bot,f] "inconsistency of bottom wrt. inclusion" .
         test ((bot ||| f) == f) os [bot,f] "bottom not a unit for join" .
         test (f == f') os [f,f'] (a3 ++ "idempotent")
         ) : do
          u <- objects
          let botTU = bottom t u
          let botSU = bottom s u
          let fbot = f ^ botTU
          [test (fbot == botSU) os [f,botTU,fbot,botSU]
                "zero-law violated"                    ]
         ++ do
         g <- homset s t
         let j = f ||| g
         let j' = g ||| f
         let ms = [f,g,j]
         (test (distrAll_source c j == s) os ms (a2 ++ "source") .
          test (distrAll_target c j == t) os ms (a2 ++ "target") .
          test (isMor s t j)   os ms (a1 ++ "non-morphism") . 
          test (j == j')   os (ms ++ [j']) (a3 ++ "commutative") .
          test (f &&& j == f) os ms "meet is not absorbing" .
          test (f ||| (f &&& g) == f) os ms (a3 ++ "absorbing")
          ) : do
           h <- homset s t
           let j1 = j ||| h
           let ms1 = [f,g,j,h,j1]
           let m1 = j &&& h
           let m2 = (f &&& h) ||| (g &&& h)
           [let j2  =       g ||| h
                j2' = f ||| j2      in
            test (j2' == j1) os (ms1++[j2,j2']) (a3 ++ "associative") .
            test (m1 == m2) os [f,g,h,m1,m2] "lattice not distributive"
            ]
          ++ do
           o3 <- objects
           k <- homset o3 s
           let kf = k ^ f
           let kg = k ^ g
           let kj = k ^ j
           let jk = kf ||| kg
           [test (kj == jk) (o3:os) [k,f,g,kj,jk] "join-distributivity violated"]
\end{code}
%}}}

From the above test, together with the allegory test,
it follows that $F \rsubs G \zeq F \relor G = G$,
but this can also be tested separately:

\index{distrAll_join_incl_TEST@{\texttt{distrAll\_join\_incl\_TEST}}}%
%{{{ distrAll_join_incl_TEST
\begin{code}
distrAll_join_incl_TEST :: Eq mor => Test DistrAll obj mor
distrAll_join_incl_TEST c =
  let objects = distrAll_objects c
      (|||) = distrAll_join c
      (<<==) = distrAll_incl c
      homset = distrAll_homset c
  in ffold $ do
      s <- objects
      t <- objects
      f <- homset s t
      g <- homset s t
      let j = f ||| g
      [test ((j == g) == (f <<== g)) [s,t] [f,g]
            "inconsistency of join wrt. inclusion"]
\end{code}
%}}}

A test for atomicity of a morphism
only has to check all morphisms from
the homset of the morphism in question for inclusion in that morphism:

\index{distrAll_isAtom@{\texttt{distrAll\_isAtom}}}%
%{{{ distrAll_isAtom
\begin{code}
distrAll_isAtom :: Eq mor => DistrAll obj mor -> obj -> obj -> mor -> Bool
distrAll_isAtom da s t m =
  let b = distrAll_bottom da s t
      homs = distrAll_homset da s t
      (<<==) = distrAll_incl da
  in distrAll_isMor da s t m &&
     m /= b &&
     all (\ m' -> (m' == m) || (m' == b) || not (m' <<== m)) homs
\end{code}
%}}}

Filtering homsets with this test
provides the default definition for atom lists:

\index{distrAll_atomset_default@{\texttt{distrAll\_atomset\_default}}}%
\index{distrAll_atoms_default@{\texttt{distrAll\_atoms\_default}}}%
%{{{ distrAll_atomset_default, distrAll_atoms_default
\begin{code}
distrAll_atomset_default all s t = filter (distrAll_isAtom all s t)
                                          (distrAll_homset all s t)
\end{code}

It is a fact that every finite Boolean lattice is atomic,
i.e., every lattice element is the join of all atoms below it.
Therefore, in a relation algebra
the atom lists obtained by filtering the global atom sets with inclusion
are, when considered as sets,
a unique representation of the morphism in question.
We provide these atom lists already here:

\begin{code}
distrAll_atoms_default all m =
  let s = distrAll_source all m
      t = distrAll_target all m
  in filter (\ at -> distrAll_incl all at m) $ distrAll_atomset all s t
\end{code}
%}}}

Our separate test for the atom components
assumes an ordering on morphisms
for being able to use a more efficient comparison:

\index{distrAll_atomTEST@{\texttt{distrAll\_atomTEST}}}%
\begin{code}
distrAll_atomTEST :: Ord mor => Test DistrAll obj mor
distrAll_atomTEST da =
 let objects = distrAll_objects da
 in
 ffold $ do
   s <- objects
   t <- objects
   let os = [s,t]
   let atoms = distrAll_atomset_default da s t
   let atoms' = distrAll_atomset da s t
   test (atoms `listEqAsSet` atoms') os atoms' "inconsistent atom set" :
     do f <- distrAll_homset da s t
        let ats = distrAll_atoms_default da f
        let ats' = distrAll_atoms da f
        [test (ats `listEqAsSet` ats') os (f : ats')
              "inconsistent atom representation"]
\end{code}
%}}}

%{{{ \subsection{Division Allegories}
\subsection{Division Allegories}

Division allegories only add three division operators:

\index{DivAll@{\texttt{DivAll}}}%
\index{divAll*@{\texttt{divAll\_*}}}%
\begin{code}
data DivAll obj mor = DivAll
  {divAll_distrAll :: DistrAll obj mor
  ,divAll_rres :: mor -> mor -> mor
  ,divAll_lres :: mor -> mor -> mor
  ,divAll_syq  :: mor -> mor -> mor
  }
\end{code}

The symmetric quotient is defined on top of the residuals,
which gives us the default definition:

\index{divAll_syq_default@{\texttt{divAll\_syq\_default}}}%
\begin{code}
divAll_syq_default :: DivAll obj mor -> mor -> mor -> mor
divAll_syq_default da f g = let conv = divAll_converse da in
   divAll_meet da (divAll_rres da f g)
                  (divAll_lres da (conv f) (conv g))
\end{code}

Each residual may be defined in terms of the other,
i.e., $f \rres g = \rtrans{(\rtrans{g} \lres \rtrans{f})}$ and
$g \lres h = \rtrans{(\rtrans{h} \rres \rtrans{g})}$:

\index{divAll_rres_lresDefault@{\texttt{divAll\_rres\_lresDefault}}}%
\index{divAll_lres_rresDefault@{\texttt{divAll\_lres\_rresDefault}}}%
\begin{code}
divAll_rres_lresDefault :: DivAll obj mor -> mor -> mor -> mor
divAll_rres_lresDefault da f g =
   let conv = divAll_converse da
   in conv (divAll_lres da (conv g) (conv f))

divAll_lres_rresDefault :: DivAll obj mor -> mor -> mor -> mor
divAll_lres_rresDefault da g h =
   let conv = divAll_converse da
   in conv (divAll_rres da (conv h) (conv g))
\end{code}

For giving a default definition based on the residual properties,
we need to be able to find
the inclusion-maximal element of a set of morphisms;
for this purpose we define the following auxiliary function:

\index{poMax@{\texttt{poMax}}}%
%{{{ poMax
\begin{code}
poMax :: (a -> a -> Bool) -> a -> [a] -> a
--  Preconditions: `po` is a partial order
--                 bot  is the least element wrt. `po`
--                 the list contains a maximal element
poMax po bot [] = bot
poMax po bot [x] = x
poMax po bot (x:y:ys)
  | x `po` y   =  poMax po y ys
  | y `po` x   =  poMax po x ys
  | otherwise  =  poMax po x ys -- Neither of x or y is the maximum,
                                -- but we do not try to catch errors.
                                -- Alternatively, we could use join here.
\end{code}
%}}}

Translation of the residual specifications is now straightforward:

\index{divAll_rres_inclDefault@{\texttt{divAll\_rres\_inclDefault}}}%
\index{divAll_lres_inclDefault@{\texttt{divAll\_lres\_inclDefault}}}%
%{{{ divAll_rres_inclDefault, divAll_lres_inclDefault
\begin{code}
divAll_rres_inclDefault da f g =
   let (^) = divAll_comp da
       (<<==) = divAll_incl da
       target = divAll_target da
       s = target f
       t = target g
       ms = divAll_homset da s t
       check m = (f ^ m) <<== g
   in poMax (<<==) (divAll_bottom da s t) (filter check ms)
divAll_lres_inclDefault da g h =
   let (^) = divAll_comp da
       (<<==) = divAll_incl da
       source = divAll_source da
       s = source g
       t = source h
       ms = divAll_homset da s t
       check m = (m ^ h) <<== g
   in poMax (<<==) (divAll_bottom da s t) (filter check ms)
\end{code}
%}}}

We provide separate tests for the different components,
so that one may selectively test
only those not defined via default definitions:

\index{divAll_rres_TEST@{\texttt{divAll\_rres\_TEST}}}%
%{{{ divAll_rres_TEST
\begin{code}
divAll_rres_TEST :: Test DivAll obj mor
divAll_rres_TEST da =
  let objects = divAll_objects da
      homset = divAll_homset da
      isMor = divAll_isMor da
      (^) = divAll_comp da
      (<<==) = divAll_incl da
      rres = divAll_rres da
  in ffold $ do
      s <- objects
      t <- objects
      g <- homset s t
      m <- objects
      f <- homset s m
      let r = f `rres` g
      testX (isMor m t r) [s,m,t] [f,g,r] "right residual yields non-morphism" $
       do
        h <- homset m t
        let fh = f ^ h
        [test ((h <<== r) == (fh <<== g)) [s,m,t] [g,f,r,h,fh]
              "right residual property violated"]
\end{code}
%}}}

Although the corresponding test for left residuals
textually differs only in minor points,
the fact that these minor points affect the dependencies of the
inner-most quantification and several places depending on it
implies that factoring out the common parts
would incur unreasonable costs in at least one of running time
and readability.

\index{divAll_lres_TEST@{\texttt{divAll\_lres\_TEST}}}%
%{{{ divAll_lres_TEST
\begin{code}
divAll_lres_TEST :: Test DivAll obj mor
divAll_lres_TEST da =
  let objects = divAll_objects da
      homset = divAll_homset da
      isMor = divAll_isMor da
      (^) = divAll_comp da
      (<<==) = divAll_incl da
      lres = divAll_lres da
  in ffold $ do
      s <- objects
      t <- objects
      g <- homset s t
      m <- objects
      h <- homset m t
      let r = g `lres` h
      testX (isMor s m r) [s,m,t] [r,g,h] "left residual yields non-morphism" $
       do
        f <- homset s m
        let fh = f ^ h
        [test ((f <<== r) == (fh <<== g)) [s,m,t] [g,f,r,h,fh]
              "left residual property violated"]
\end{code}
%}}}

The obvious and fast way to check the symmetric quotient is by
verifying that its results correspond to the definition via residuals:

\index{divAll_syq_resTEST@{\texttt{divAll\_syq\_resTEST}}}%
%{{{ divAll_syq_resTEST
\begin{code}
divAll_syq_resTEST :: Eq mor => Test DivAll obj mor
divAll_syq_resTEST da =
  let objects = divAll_objects da
      homset = divAll_homset da
      isMor = divAll_isMor da
      conv = divAll_converse da
      (&&&) = divAll_meet da
      lres = divAll_lres da
      rres = divAll_rres da
      syq = divAll_syq da
  in ffold $ do
      s <- objects
      m <- objects
      f <- homset m s
      t <- objects
      let os = [s,m,t]
      g <- homset m t
      let q = syq f g
      let l = rres f g
      let r = lres (conv f) (conv g)
      let ms = [f,g,q,l,r]
      testX (isMor s t q) os ms "syQ yields non-morphism"
        [test (q == (l &&& r)) os ms "syQ is not meet of residuals"]
\end{code}
%}}}

However, the following definition of symmetric quotients
also makes sense in the absence of residuals
(see \cite{Furusawa-Kahl-1998}):

%{{{ {Def}\Deflabel{DistrAllSyq}
\begin{Def}\Deflabel{DistrAllSyq}
In an allegory,
the \emph{symmetric quotient}\index{symmetric quotient}
$\syq{Q}{S}:\objB\rel \objC$ of two relations
$Q : \objA \rel \objB$ and $S : \objA \rel \objC$ is defined by
\BD
    X \rsubs \syq Q S
  \iff
    Q\rcmp X \rsubs S \mbox{~~and~~}
    X \rcmp \rtrans{S} \rsubs \rtrans{Q}
  \qquad
    \mbox{for all $X:\objB\rel \objC$}
\enskip.
\EDQ
\end{Def}
%}}}

This kind of symmetric quotient is of course usually a partial operation
(it is obviously univalent).
The following function calculates this operation:

\index{all_syq@{\texttt{all\_syq}}}%
\begin{code}
all_syq :: All obj mor -> mor -> mor -> Maybe mor
all_syq a f g =
  let objects = all_objects a
      homset = all_homset a
      isMor = all_isMor a
      conv = all_converse a
      (^) = all_comp a
      (<<==) = all_incl a
      target = all_target a
      s = target f
      t = target g
      check q x = (x <<== q) == (((f ^ x) <<== g) && ((g ^ conv x) <<== f))
      syq q = all (check q) (homset s t)
   in listToMaybe $ filter syq $ homset s t
\end{code}

A given binary partial operation on morphisms
can be tested for inclusion in this symmetric quotient by the following function:

\index{all_syq_directTEST@{\texttt{all\_syq\_directTEST}}}%
%{{{ all_syq_directTEST
\begin{code}
all_syq_directTEST :: (mor -> mor -> Maybe mor) -> Test All obj mor
all_syq_directTEST syq a =
  let objects = all_objects a
      homset = all_homset a
      isMor = all_isMor a
      conv = all_converse a
      (^) = all_comp a
      (<<==) = all_incl a
  in ffold $ do
      s <- objects
      m <- objects
      f <- homset m s
      t <- objects
      let os = [s,m,t]
      g <- homset m t
      case syq f g of
        Nothing -> []
        Just q -> do
          let ms = [f,g,q]
          testX (isMor s t q) os ms "syQ yields non-morphism" $
           do x <- homset s  t
              let l = f ^      x <<==      g
              let r = x ^ conv g <<== conv f
              [test ((x <<== q) == (l && r)) os ms "syQ property violated"]
\end{code}
%}}}

Finally, here is the definition of the expanded interface:

\index{divAll*@{\texttt{divAll\_*}}}%
%{{{ expanded interface
\begin{code}
divAll_isObj    = cat_isObj        . divAll_cat   -- :: obj -> Bool
divAll_isMor    = cat_isMor        . divAll_cat   -- :: obj -> obj -> mor -> Bool
divAll_objects  = cat_objects      . divAll_cat   -- :: [obj]
divAll_homset   = cat_homset       . divAll_cat   -- :: obj -> obj -> [mor]
divAll_source   = cat_source       . divAll_cat   -- :: mor -> obj
divAll_target   = cat_target       . divAll_cat   -- :: mor -> obj
divAll_idmor    = cat_idmor        . divAll_cat   -- :: obj -> mor
divAll_comp     = cat_comp         . divAll_cat   -- :: mor -> mor -> mor

divAll_cat      = all_cat          . divAll_all
divAll_converse = all_converse     . divAll_all      -- :: mor -> mor
divAll_meet     = all_meet         . divAll_all      -- :: mor -> mor -> mor
divAll_incl     = all_incl         . divAll_all      -- :: mor -> mor -> Bool

divAll_all      = distrAll_all     . divAll_distrAll
divAll_bottom   = distrAll_bottom  . divAll_distrAll -- :: obj -> obj -> mor
divAll_bot      = distrAll_bot     . divAll_distrAll -- :: mor -> mor
divAll_join     = distrAll_join    . divAll_distrAll -- :: mor -> mor -> mor
divAll_atomset  = distrAll_atomset . divAll_distrAll -- :: obj -> obj -> [mor]
divAll_atoms    = distrAll_atoms   . divAll_distrAll -- :: mor -> [mor]
\end{code}
%}}}
%}}}

%{{{ \subsection{Dedekind Categories}
\subsection{Dedekind Categories}

\index{Ded@{\texttt{Ded}}}%
\index{ded*@{\texttt{ded\_*}}}%
\begin{code}
data Ded obj mor = Ded
  {ded_divAll :: DivAll obj mor
  ,ded_top  :: obj -> obj -> mor
  }
\end{code}

Note that, as mentioned above,
every finite distributive allegory is already a Dedekind category,
so we can provide a default definition for \verb|top|:

\index{@{\texttt{}}}%
\begin{code}
ded_top_default :: Ded obj mor -> obj -> obj -> mor
ded_top_default d s t = poMax (ded_incl d) (ded_bottom d s t) (ded_homset d s t)
\end{code}

In the same way as for bottom,
we introduce an abbreviation that allows to directly access
the top relation from the homset of a given morphism:

\index{tp@{\texttt{tp}}|see{\texttt{ded\_tp}}}%
\index{ded_tp@{\texttt{ded\_tp}}}%
\begin{code}
ded_tp ded f = let s = ded_source ded f
                   t = ded_target ded f
               in ded_top ded s t
\end{code}

The only item to test here is whether every morphism is indeed
included in the top element of its homset:

\index{ded_top_incl_TEST@{\texttt{ded\_top\_incl\_TEST}}}%
\begin{code}
ded_top_incl_TEST :: Eq mor => Test Ded obj mor
ded_top_incl_TEST c =
  let objects = ded_objects c
      top = ded_top c
      (<<==) = ded_incl c
      homset = ded_homset c
  in ffold $ do
      s <- objects
      t <- objects
      let tp = top s t
      test (ded_isMor c s t tp) [s,t] [tp] "top is non-morphism" : do
        f <- homset s t
        [test (f <<== tp) [s,t] [f,tp] "inconsistency of top wrt. inclusion"]
\end{code}

Finally, here is the expanded interface:

\index{ded*@{\texttt{ded\_*}}}%
%{{{ expanded interface
\begin{code}
ded_isObj    = cat_isObj    . ded_cat       -- :: obj -> Bool
ded_isMor    = cat_isMor    . ded_cat       -- :: obj -> obj -> mor -> Bool
ded_objects  = cat_objects  . ded_cat       -- :: [obj]
ded_homset   = cat_homset   . ded_cat       -- :: obj -> obj -> [mor]
ded_source   = cat_source   . ded_cat       -- :: mor -> obj
ded_target   = cat_target   . ded_cat       -- :: mor -> obj
ded_idmor    = cat_idmor    . ded_cat       -- :: obj -> mor
ded_comp     = cat_comp     . ded_cat       -- :: mor -> mor -> mor
                             
ded_cat      = all_cat      . ded_all
ded_converse = all_converse . ded_all       -- :: mor -> mor
ded_meet     = all_meet     . ded_all       -- :: mor -> mor -> mor
ded_incl     = all_incl     . ded_all       -- :: mor -> mor -> Bool

ded_distrAll = divAll_distrAll . ded_divAll

ded_all      = divAll_all      . ded_divAll
ded_bottom   = divAll_bottom   . ded_divAll -- :: obj -> obj -> mor
ded_bot      = divAll_bot      . ded_divAll -- :: mor -> mor
ded_join     = divAll_join     . ded_divAll -- :: mor -> mor -> mor
ded_atomset  = divAll_atomset  . ded_divAll -- :: obj -> obj -> [mor]
ded_atoms    = divAll_atoms    . ded_divAll -- :: mor -> [mor]
ded_rres     = divAll_rres     . ded_divAll -- :: mor -> mor -> mor
ded_lres     = divAll_lres     . ded_divAll -- :: mor -> mor -> mor
ded_syq      = divAll_syq      . ded_divAll -- :: mor -> mor -> mor
\end{code}
%}}}
%}}}

%{{{ \subsection{Relation Algebras}
\subsection{Relation Algebras}

Not even every finite Dedekind category is a relation algebra,
so the introduction of the complement is again a real step in advance:

\index{RA@{\texttt{RA}}}%
\index{ra*@{\texttt{ra\_*}}}%
\begin{code}
data RA obj mor = RA
  {ra_ded   :: Ded obj mor
  ,ra_compl :: mor -> mor
  }
\end{code}

The presence of the complement allows more concise default definitions for the residuals:

\index{ra_rres_default@{\texttt{ra\_rres\_default}}}%
\index{ra_lres_default@{\texttt{ra\_lres\_default}}}%
\begin{code}
ra_rres_default ra f g =
       let compl = ra_compl ra in compl (ra_comp ra (ra_converse ra f) (compl g))
ra_lres_default ra f g =
       let compl = ra_compl ra in compl (ra_comp ra (compl f) (ra_converse ra g))
\end{code}

Testing the complement is straightforward testing of the properties
$F \reland \relnot{F} = \RO{}$ and $F \relor \relnot{F} = \RL{}$:

\index{ra_compl_TEST@{\texttt{ra\_compl\_TEST}}}%
\begin{code}
ra_compl_TEST :: Eq mor => Test RA obj mor
ra_compl_TEST c =
  let objects = ra_objects c
      homset = ra_homset c
      bot = ra_bottom c
      top = ra_top c
      not = ra_compl c
      (&&&) = ra_meet c
      (|||) = ra_join c
  in ffold $ do
      s <- objects
      t <- objects
      let tp = top s t
      let bt = bot s t
      f <- homset s t
      let fN = not f
      testX (ra_isMor c s t fN) [s,t] [fN] "complement yields non-morphism"
       (let m = f &&& fN
     	    j = f ||| fN in
     	[test (m == bt) [s,t] [f,fN,m,bt] "meet with complement is not bottom" .
     	 test (j == tp) [s,t] [f,fN,j,tp] "join with complement is not top"
     	]
       )
\end{code}

For relation algebras with default definitions for all division operators
the following is sufficient:

\index{ra_TEST@{\texttt{ra\_TEST}}}%
\begin{code}
ra_TEST :: (Eq obj,Eq mor) => Test RA obj mor
ra_TEST ra =
  cat_TEST (ra_cat ra) .
  all_TEST (ra_all ra) .
  distrAll_TEST (ra_distrAll ra) .
  ded_top_incl_TEST (ra_ded ra) .
  ra_compl_TEST ra
\end{code}

Otherwise, there is also a variant
with the atom tests (which require \verb|Ord mor|)
and the division tests included:

\index{ra_TEST_ALL@{\texttt{ra\_TEST\_ALL}}}%
\begin{code}
ra_TEST_ALL :: (Eq obj,Ord mor) => Test RA obj mor
ra_TEST_ALL ra =
  cat_TEST (ra_cat ra) .
  all_TEST (ra_all ra) .
  distrAll_TEST (ra_distrAll ra) .
  distrAll_atomTEST (ra_distrAll ra) .
  let da = ra_divAll ra in
  divAll_rres_TEST da .
  divAll_lres_TEST da .
  divAll_syq_resTEST da .
  ded_top_incl_TEST (ra_ded ra) .
  ra_compl_TEST ra
\end{code}

Finally, here is the expanded interface:

\index{ra*@{\texttt{ra\_*}}}%
%{{{ expanded interface
\begin{code}
ra_isObj    = cat_isObj    . ra_cat      -- :: obj -> Bool
ra_isMor    = cat_isMor    . ra_cat      -- :: obj -> obj -> mor -> Bool
ra_objects  = cat_objects  . ra_cat      -- :: [obj]
ra_homset   = cat_homset   . ra_cat      -- :: obj -> obj -> [mor]
ra_source   = cat_source   . ra_cat      -- :: mor -> obj
ra_target   = cat_target   . ra_cat      -- :: mor -> obj
ra_idmor    = cat_idmor    . ra_cat      -- :: obj -> mor
ra_comp     = cat_comp     . ra_cat      -- :: mor -> mor -> mor

ra_cat      = all_cat      . ra_all
ra_converse = all_converse . ra_all      -- :: mor -> mor
ra_meet     = all_meet     . ra_all      -- :: mor -> mor -> mor
ra_incl     = all_incl     . ra_all      -- :: mor -> mor -> Bool

ra_all      = ded_all      . ra_ded
ra_bottom   = ded_bottom   . ra_ded      -- :: obj -> obj -> mor
ra_bot      = ded_bot      . ra_ded      -- :: mor -> mor
ra_join     = ded_join     . ra_ded      -- :: mor -> mor -> mor
ra_atomset  = ded_atomset  . ra_ded      -- :: obj -> obj -> [mor]
ra_atoms    = ded_atoms    . ra_ded      -- :: mor -> [mor]
ra_top      = ded_top      . ra_ded      -- :: obj -> obj -> mor
ra_tp       = ded_tp       . ra_ded      -- :: mor -> mor
ra_rres     = ded_rres     . ra_ded      -- :: mor -> mor -> mor
ra_lres     = ded_lres     . ra_ded      -- :: mor -> mor -> mor
ra_syq      = ded_syq      . ra_ded      -- :: mor -> mor -> mor

ra_divAll   = ded_divAll   . ra_ded
ra_distrAll = ded_distrAll . ra_ded
\end{code}
%}}}
%}}}

%{{{ \subsection{Simple Example Algebras}
\subsection{Simple Example Algebras}

The following four algebras are not studied 
because they are interesting themselves.
Rather, we need them as coefficients of matrix algebra constructions.
So they are defined in a very detailed way so as to be able
to proceed smoothly to more complex structures.

%{{{ \subsubsection{Trivial Relation Algebras}
\subsubsection{Trivial Relation Algebras}

The smallest relation algebra has just one object and one morphism.
Sometimes the definition of relation algebras
requires that homsets be non-trivial Boolean lattices,
but, as already mentioned, we do not follow this here.

Since we want to be able to talk about \emph{embedded} relation algebras,
we do not fix the object and morphism type.
Instead, we build a trivial relation algebra
from arbitrary objects and morphisms,
as long as their types allow equality to be tested.

\index{cat1@{\texttt{cat1}}}%
\begin{code}
cat1 :: (Eq obj, Eq mor) => obj -> mor -> Cat obj mor
cat1 obj mor = Cat
  {cat_isObj   = (obj ==)
  ,cat_isMor   = const $ const $ (mor ==)
  ,cat_objects = [obj]
  ,cat_homset  = const $ const [mor]
  ,cat_source  = const obj
  ,cat_target  = const obj
  ,cat_idmor   = const mor
  ,cat_comp    = const $ const mor
  }
\end{code}

\index{all1@{\texttt{all1}}}%
\begin{code}
all1 :: (Eq obj, Eq mor) => obj -> mor -> All obj mor
all1 obj mor = All
  {all_cat = cat1 obj mor
  ,all_converse = id
  ,all_meet = const $ const mor
  ,all_incl = const $ const True
  }
\end{code}

\index{distrAll1@{\texttt{distrAll1}}}%
\begin{code}
distrAll1 :: (Eq obj, Eq mor) => obj -> mor -> DistrAll obj mor
distrAll1 obj mor = DistrAll
  {distrAll_all  = all1 obj mor
  ,distrAll_bottom = const $ const mor
  ,distrAll_join = const $ const mor
  ,distrAll_atomset = const $ const []
  ,distrAll_atoms = const []
  }
\end{code}

\index{divAll1@{\texttt{divAll1}}}%
\begin{code}
divAll1 :: (Eq obj, Eq mor) => obj -> mor -> DivAll obj mor
divAll1 obj mor = DivAll
  {divAll_distrAll  = distrAll1 obj mor
  ,divAll_rres = const $ const mor
  ,divAll_lres = const $ const mor
  ,divAll_syq  = const $ const mor
  }
\end{code}

\index{ded1@{\texttt{ded1}}}%
\begin{code}
ded1 :: (Eq obj, Eq mor) => obj -> mor -> Ded obj mor
ded1 obj mor = Ded
  {ded_divAll  = divAll1 obj mor
  ,ded_top  = const $ const mor
  }
\end{code}

\index{ra1@{\texttt{ra1}}}%
\begin{code}
ra1 :: (Eq obj, Eq mor) => obj -> mor -> RA obj mor
ra1 obj mor = RA
  {ra_ded = ded1 obj mor
  ,ra_compl = id
  }
\end{code}
%}}}

%{{{ \subsubsection{Two-Element Relation Algebras}
\subsubsection{Two-Element Relation Algebras}

In the same way, we may define two-element relation algebras,
where one morphism is bottom
and the other morphism is identity and top at the same time:

\index{cat2@{\texttt{cat2}}}%
\begin{code}
cat2 :: (Eq obj, Eq mor) => obj -> mor -> mor -> Cat obj mor
cat2 obj bot id = Cat
  {cat_isObj   = (obj ==)
  ,cat_isMor   = const $ const $ (\ mor -> bot == mor || id == mor)
  ,cat_objects = [obj]
  ,cat_homset  = const $ const [bot,id]
  ,cat_source  = const obj
  ,cat_target  = const obj
  ,cat_idmor   = const id
  ,cat_comp    = (\ f g -> if f == id then g else bot)
  }
\end{code}

\index{all2@{\texttt{all2}}}%
\begin{code}
all2 :: (Eq obj, Eq mor) => obj -> mor -> mor -> All obj mor
all2 obj bot i = let c2 = cat2 obj bot i
 in All
  {all_cat = c2
  ,all_converse = id
  ,all_meet = cat_comp c2
  ,all_incl   = (\ f g -> f == bot || g == i)
  }
\end{code}

\index{distrAll2@{\texttt{distrAll2}}}%
\begin{code}
distrAll2 :: (Eq obj, Eq mor) => obj -> mor -> mor -> DistrAll obj mor
distrAll2 obj bot i = DistrAll
  {distrAll_all  = all2 obj bot i
  ,distrAll_bottom = const $ const bot
  ,distrAll_join = (\ f g -> if f == bot then g else i)
  ,distrAll_atomset = const $ const [i]
  ,distrAll_atoms   = (\ f -> if f == i then [i] else [])
  }
\end{code}

\index{divAll2@{\texttt{divAll2}}}%
\begin{code}
divAll2 :: (Eq obj, Eq mor) => obj -> mor -> mor -> DivAll obj mor
divAll2 obj bot i = DivAll
  {divAll_distrAll = distrAll2 obj bot i
  ,divAll_rres = (\ f g -> if f == bot || g == i   then i else bot)
  ,divAll_lres = (\ f g -> if f == i   || g == bot then i else bot)
  ,divAll_syq  = (\ f g -> if f == g               then i else bot)
  }
\end{code}


\index{ded2@{\texttt{ded2}}}%
\begin{code}
ded2 :: (Eq obj, Eq mor) => obj -> mor -> mor -> Ded obj mor
ded2 obj bot i = Ded
  {ded_divAll  = divAll2 obj bot i
  ,ded_top  = const $ const i
  }
\end{code}


\index{ra2@{\texttt{ra2}}}%
\begin{code}
ra2 :: (Eq obj, Eq mor) => obj -> mor -> mor -> RA obj mor
ra2 obj bot i = RA
  {ra_ded = ded2 obj bot i
  ,ra_compl = (\ f -> if f == bot then i else bot)
  }
\end{code}
%}}}

%{{{ \subsubsection{The Relation Algebra $\mathbb{B}$}
\subsubsection{The Relation Algebra $\mathbb{B}$}

The relation algebra $\mathbb{B}$ of truth values might now be defined
as \verb|cat2 () False True|.
For efficiency's sake, we also give a direct definition:

\index{catB@{\texttt{catB}}}%
\begin{code}
catB :: Cat () Bool
catB = Cat
  {cat_isObj   = const True
  ,cat_isMor   = const $ const $ const True
  ,cat_objects = [()]
  ,cat_homset  = const $ const [False, True]
  ,cat_source  = const ()
  ,cat_target  = const ()
  ,cat_idmor   = const True
  ,cat_comp    = (&&)
  }
\end{code}

\index{allB@{\texttt{allB}}}%
\begin{code}
allB :: All () Bool
allB = All
  {all_cat = catB
  ,all_converse = id
  ,all_meet = (&&)
  ,all_incl   = (\ f g -> g || not f)
  }
\end{code}

\index{distrAllB@{\texttt{distrAllB}}}%
\begin{code}
distrAllB :: DistrAll () Bool
distrAllB = DistrAll
  {distrAll_all  = allB
  ,distrAll_bottom = const $ const False
  ,distrAll_join = (||)
  ,distrAll_atomset = const $ const [True]
  ,distrAll_atoms   = (\ f -> if f then [True] else [])
  }
\end{code}

\index{divAllB@{\texttt{divAllB}}}%
\begin{code}
divAllB :: DivAll () Bool
divAllB = DivAll
  {divAll_distrAll = distrAllB
  ,divAll_rres = (\ f g -> not f || g)
  ,divAll_lres = (\ f g -> not g || f)
  ,divAll_syq  = (==)
  }
\end{code}


\index{dedB@{\texttt{dedB}}}%
\begin{code}
dedB :: Ded () Bool
dedB = Ded
  {ded_divAll  = divAllB
  ,ded_top  = const $ const True
  }
\end{code}


\index{raB@{\texttt{raB}}}%
\begin{code}
raB :: RA () Bool
raB = RA
  {ra_ded = dedB
  ,ra_compl = not
  }
\end{code}
%}}}

%{{{ \subsubsection{$(n+1)$-Element Linearly Ordered Dedekind Categories}
\subsubsection{$(n+1)$-Element Linearly Ordered Dedekind Categories}

We now give a set of examples of ``discretely fuzzy'' Dedekind categories
which are not relation algebras.
As in $\mathbb{B}$, there is only one object
and the identity is the maximum morphism,
but there is a linearly ordered set of morphisms below the identity.
For simplicity,
we use initial segments \verb|[0 .. n]| of the natural numbers as homsets.

Composition coincides with meet and is the minimum:

\index{catN@{\texttt{catN}}}%
\begin{code}
catN :: Eq obj => obj -> Int -> Cat obj Int
catN obj n = Cat
  {cat_isObj   = (obj ==)
  ,cat_isMor   = const $ const $ (\ k -> 0 <= k && k <= n)
  ,cat_objects = [obj]
  ,cat_homset  = const $ const [0 .. n]
  ,cat_source  = const obj
  ,cat_target  = const obj
  ,cat_idmor   = const n
  ,cat_comp    = min
  }
\end{code}

Conversion is the identity function on morphisms:

\index{allN@{\texttt{allN}}}%
\begin{code}
allN :: Eq obj => obj -> Int -> All obj Int
allN obj n = All
  {all_cat = catN obj n
  ,all_converse = id
  ,all_meet = min
  ,all_incl   = (<=)
  }
\end{code}

Join is of course maximum,
and the only atom is \verb|1|:

\index{distrAllN@{\texttt{distrAllN}}}%
\begin{code}
distrAllN :: Eq obj => obj -> Int -> DistrAll obj Int
distrAllN obj n = da where
 da = DistrAll
  {distrAll_all  = allN obj n
  ,distrAll_bottom = const $ const 0
  ,distrAll_join = max
  ,distrAll_atomset = (\ s t -> if n > 0 then [1] else [])
  ,distrAll_atoms   = (\ f   -> if f > 0 then [1] else [])
  }
\end{code}

For residuals, we use the defaults:

\index{divAllN@{\texttt{divAllN}}}%
\begin{code}
divAllN :: Eq obj => obj -> Int -> DivAll obj Int
divAllN obj n = da where
 da = DivAll
  {divAll_distrAll = distrAllN obj n
  ,divAll_rres = divAll_rres_inclDefault da
  ,divAll_lres = divAll_lres_inclDefault da
  ,divAll_syq  = divAll_syq_default da
  }
\end{code}

Maximum morphisms are trivial again:

\index{dedN@{\texttt{dedN}}}%
\begin{code}
dedN :: Eq obj => obj -> Int -> Ded obj Int
dedN obj n = Ded
  {ded_divAll  = divAllN obj n
  ,ded_top  = const $ const n
  }
\end{code}

Just for fun,
we also define a relation algebra constructor with a bogus
complement function
which will only work for $n \in \{0,1\}$.

\index{raN@{\texttt{raN}}}%
\begin{code}
raN :: Eq obj => obj -> Int -> RA obj Int
raN obj n = RA
  {ra_ded = dedN obj n
  ,ra_compl = (n -)
  }
\end{code}

Consequently, \verb|ra_TEST_ALL (raN () 2)| fails in \verb|ra_compl_TEST|,
exhibiting the middle morphism which has no complement.
%}}}
%}}}


%% Local variables:
%% folded-file: t
%% fold-internal-margins: 0
%% eval: (fold-set-marks "%{{{ " "%}}}")
%% eval: (fold-whole-buffer)
%% end:
