
\section{From Categories to Relation Algebras}\sectlabel{RelAlgClasses}

%{{{ Intro
In this section we review the necessary definitions
and a few of their properties; the notation we use
is that agreed upon for the book \cite{Brink-Kahl-Schmidt-1997}.

Alongside,
we introduce Haskell identifiers that shall serve to access
the different components of these mathematical structures
in our Haskell-based exploration system.
As most sections of this report,
the file containing the present section is therefore at the same time
a \emph{literate} Haskell module.

This present module offers a uniform interface to categories of all levels,
but at the cost of employing a non-standard extension to Haskell 98,
namely \emph{multi-parameter type classes}
with \emph{functional dependencies} between parameters
\cite{Jones-2000} ---
these are currently supported by the Haskell interpreter Hugs%
\footnote{To load this module, Hugs needs to be started with the
command line option ``\texttt{-98}'' which enables Hugs-specific extensions;
this option cannot be changed while the interpreter is running.
}%footnote
.

For Haskell, this module begins with the following heading:

\begin{code}
module RelAlgClasses where
\end{code}

The absence of an explicit export list before the keyword \verb|where|
implies that everything defined in this module is also exported.
%}}}

%{{{ \subsection{Categories}
\subsection{Categories}


We recall the definition of a category, the construct that we have chosen to
model heterogeneity of relation algebras.

%{{{ {Def}{category}
\begin{Def}
A \emph{category} ${\catC}$
is a tuple $(Obj_{\catC}, Mor_{\catC}, \_ : \_ \tfun \_, \RId{}, \rcmp)$
where
\begin{itemize}
\item $Obj_{\catC}$ is a collection of \emph{objects}.
\item $Mor_{\catC}$ is a collection of \emph{arrows} or \emph{morphisms}.
\item ``$\_ : \_ \tfun \_$'' is ternary relation
  relating every morphism $f$ univalently with two objects $\objA$ and $\objB$,
  written $f : \objA \tfun \objB$,
  where $\objA$ is called the \emph{source} of $f$,
  and $\objB$ the \emph{target} of $f$.

  The collection of all morphisms $f$ with $f : \objA \tfun \objB$
  is denoted as $\CThom{\catC}{\objA}{\objB}$ and also called a \emph{homset}.
\item ``$\rcmp$'' is the binary \emph{composition} operator,
  and composition of two morphisms $f : \objA \tfun \objB$ and
  $g : \objB' \tfun \objC$ is defined iff $\objB = \objB'$,
  and then $(f \rcmp g) : \objA \tfun \objC$;
  composition is associative.
\item $\RId{}$ associates with every object $\objA$ a morphism $\RId{\objA}$
  which is both a right and left unit for composition.
\qed
\end{itemize}
\end{Def}
%}}}

\noindent
Composition operators like ``$\rcmp$''
will bind with a higher priority than all other binary operators.

\medskip
For being able to manipulate categories as data in Haskell programs,
we define a multi-parameter class \texttt{Category}
with three parameters:
The type variable \texttt{cat} stands for the type of categories-as-data,
and we do not parameterise this type for the time being.
Next,
\texttt{obj} is the type of objects, but this type need not
exclusively
comprise objects of the categories in question,
so we add a member predicate \texttt{isObj}
that checks whether some item of the object type is an object of the
category in question.
Last, \texttt{mor} is the type of morphisms,
and the test whether some item of the morphism type
is in fact a morphism is specialised to
directly check membership
in the homset spanned by two objects,
i.e., membership in the relation $\_ : \_ \tfun \_$.
This is more useful than a global morphism test
(which could be defined using the {\tt source} and {\tt target} functions),
and the obligation to provide the additional source and target arguments
seems not to be molesting in our experience.

Since we do exhaustive exploration rather than symbolic proofs,
we want to treat only finite categories,
i.e., categories where both $Obj_{\catC}$ and $Mor_{\catC}$ are finite sets.
Therefore we demand an enumeration \texttt{objects} of the object set and,
for every two objects \verb|s| and \verb|t|,
an enumeration \verb|homset s t| of the corresponding homset.

For theoretical purposes,
the important restriction here is only that homsets should be finite,
so we define:

\begin{Def}
When all homsets of a category are restricted to be sets,
the category is called \emph{locally small}.
A locally small category is called \emph{locally finite}
if every homset is a finite set.
\qed
\end{Def}

The remaining two class members are in direct correspondence to items of the
mathematical definition;
given the enumerations of objects and homsets
it is of course possible to derive the identities from the
other information,
but we shall generally postpone such decisions to the implementation.
The interface is much easier to use if certain derived components are
included directly in the interface,
and this way,
also the implementation has more freedom to use more efficient definitions.

\index{Category@{\texttt{Category} (class)}}%
\index{isObj@{\texttt{isObj}}}%
\index{isMor@{\texttt{isMor}}}%
\index{objects@{\texttt{objects}}}%
\index{homset@{\texttt{homset}}}%
\index{source@{\texttt{source}}}%
\index{target@{\texttt{target}}}%
\index{idmor@{\texttt{idmor}}}%
\index{comp@{\texttt{comp}}}%
\begin{code}
class Category cat obj mor | cat -> obj, cat -> mor where
  isObj   :: cat -> obj -> Bool
  isMor   :: cat -> obj -> obj -> mor -> Bool
  objects :: cat -> [obj]
  homset  :: cat -> obj -> obj -> [mor]
  source  :: cat -> mor -> obj
  target  :: cat -> mor -> obj
  idmor   :: cat -> obj -> mor
  comp    :: cat -> mor -> mor -> mor
\end{code}

The functional dependencies ``\verb|cat -> obj, cat -> mor|''
correspond to the fact that every category ${\catC}$
brings with it the type of its objects and morphisms.
These functional dependencies are necessary
because the parameter variable \verb|obj| does not occur in the type of \verb|comp|,
and \verb|mor| does not occur in the type of \verb|objects|.
%}}}

%{{{ \subsection{Allegories}
\subsection{Allegories}

The simplest abstraction of the behaviour of relations
among those presented in \cite{Freyd-Scedrov-1990}
only reflects transposition (converse) and
intersection (meet) (and therewith also inclusion) of relations
on top of the category structure:

%{{{ {Def}{allegory}
\begin{Def}
An \emph{allegory}\index{allegory}
is a tuple
$\catC = (Obj_{\catC}, Mor_{\catC}, \_ : \_ \rel \_, \RId{}, \rcmp, \RAconverse{}, \reland)$
where:
\begin{enumerate}
\renewcommand{\labelenumi}{\roman{enumi})}
\item The tuple $(Obj_{\catC}, Mor_{\catC}, \_ : \_ \rel \_, \RId{}, \rcmp)$
  is a category, the so-called \emph{underlying category} of $\catC$.%
\footnote{$Mor_{\catC}$ may be a class in \cite{Freyd-Scedrov-1990},
meaning that there, allegories are not restricted to be locally small.
The price of this generality, however, is that join, meet, etc.
need to be characterised at a more elementary level,
while we can introduce these as lattice operators.
Since we mostly have finite categories in mind, anyway,
we may sacrifice that generality for the sake of brevity and readability.
} %footnote
The morphisms are usually called {\it relations}\index{relation}.
\item Every homset $\CThom{\cal C}{\objA}{\objB}$ carries the structure
of a lower semi-lattice with
$\RAmeet_{\objA,\objB}$ for \emph{meet}\index{meet},
and inclusion ordering $\rsubs_{\objA,\objB}$,
all usually written without indices.
\item $\RAconverse{}$ is the total unary operation
of \emph{conversion}\index{converse} of morphisms,
where for $R:\objA\rel \objB$ we have $\rtrans{R}:\objB\rel \objA$,
and the following properties hold:

(a) $\rtrans{(\rtrans{R})}=R\enskip$, \\
(b) $\rtrans{(Q\rcmp R)}=\rtrans{R} \rcmp \rtrans{Q}\enskip$,\\
(c) $\rtrans{(Q\RAmeet Q')} = \rtrans{Q} \RAmeet \rtrans{Q'}\enskip$.

\item For all $Q:\objA\rel \objB$ and $R, R':\objB\rel \objC$,
{\em meet-subdistributivity}\index{meet-subdistributivity} holds:
$$ 
    Q\rcmp (R \reland R') 
   \rsubs
    Q\rcmp R \reland Q\rcmp R'
\enskip.
$$

\item For all $Q:\objA\rel \objB$, $R:\objB\rel \objC$, and $S: \objA\rel \objC$,
the {\em modal rule}\index{modal rule} holds:
\BD
    Q\rcmp R\sqcap S
  \rsubs
    (Q \sqcap S\rcmp\rtrans{R})
    \rcmp
    R 
\enskip.
\EDQ
\end{enumerate}
\end{Def}
%}}}

We define the type class \texttt{Allegory}
as a sub-class of \texttt{Category},
adding the converse and meet operators,
and the inclusion relation between morphisms:

\index{Allegory@{\texttt{Allegory} (class)}}%
\index{converse@{\texttt{converse}}}%
\index{meet@{\texttt{meet}}}%
\index{incl@{\texttt{incl}}}%
\begin{code}
class Category all obj mor => Allegory all obj mor | all -> obj, all -> mor where
  converse :: all -> mor -> mor
  meet     :: all -> mor -> mor -> mor
  incl     :: all -> mor -> mor -> Bool
\end{code}

%{{{ basic properties
The following basic properties are easily deduced
from the definition of allegories:
\begin{itemize}
\item Conversion is an isotone and involutive contravariant functor:
  In addition to the properties from the definition, this comprises also
  $\rtrans{\RId{A}}=\RId{A}$ and 
  $Q \rsubs Q' \zeq \rtrans{Q} \rsubs \rtrans{Q'}$.
\item Composition is monotonic:
   If $Q \rsubs Q'$ and $R \rsubs R'$, then
      $Q\rcmp R \rsubs Q'\rcmp R'$.
\end{itemize}
%}}}

\noindent
%{{{ Dedekind rule, modal rules
From the modal rule listed among the allegory axioms,
we may --- 
using properties of conversion --- obtain the other modal rule 
$$
    Q\rcmp R \reland S
  \rsubs
    Q
    \rcmp
    (R \sqcap \rtrans{Q}\rcmp S)
\enskip,
$$
which is used by Olivier and Serrato for their axiomatisation
of Dedekind categories \cite{Olivier-Serrato-1980,Olivier-Serrato-1995}
(see also the next section)
and there called Dedekind formula\index{Dedekind formula} ---
by Jacques Riguet, however,
this name had much earlier been attached 
to the formula proved in the next proposition \cite{Riguet-1948}.
Paul Lorenzen called it {\it Bund-Axiom} \cite{Lorenzen-1954}.

%{{{ {Prop} \Proplabel{Dedekind}
\begin{Prop} \Proplabel{Dedekind}
Both \emph{modal rules}\index{modal rule}

\vskip0.5ex
$\phantom{.}$ \hskip4.3cm \hfill $
\begin{array}{lcl@{\hskip3.5cm}r}
Q\rcmp R \reland S &\rsubs& Q \rcmp (R \sqcap \rtrans{Q}\rcmp S) & (m1)
\\
Q\rcmp R \reland S &\rsubs& (Q \sqcap S\rcmp \rtrans{R})\rcmp R & (m2)
\end{array}
$

\vskip0.5ex
\noindent
together are equivalent to the {\em Dedekind rule}\index{Dedekind rule}
\BD
        Q\rcmp R\sqcap S
         \rsubs
        (Q \sqcap S\rcmp \rtrans{R})
          \rcmp (R \sqcap \rtrans{Q}\rcmp S)
\enskip.
\ED
\end{Prop}
\Proof{The modal rules follow immediately from the Dedekind rule: 
$$
    Q\rcmp R\sqcap S
  \rsubs 
    (Q \sqcap S\rcmp \rtrans{R}) \rcmp (R \sqcap \rtrans{Q}\rcmp S)
  \rsubs
    \left\{\begin{array}{l}
      (Q \sqcap S\rcmp \rtrans{R})\rcmp R 
    \\
      Q \rcmp (R \sqcap \rtrans{Q}\rcmp S)
    \end{array}\right.
$$
Conversely, 
assume that the modal rules hold.
Then we have
\BDAR
    Q\rcmp R \reland S
  \sepAR{\rsubs}{\mbox{(m1)}}
    Q \rcmp (R \sqcap \rtrans{Q}\rcmp S) \sqcap S
  \sepBR{\rsubs}{\mbox{(m2)}}
    (Q \sqcap S\rcmp \rtrans{(R \sqcap \rtrans{Q}\rcmp S)})
    \rcmp (R \sqcap \rtrans{Q}\rcmp S)
  \sepBR{\rsubs}{\forall U,V: U\reland V \rsubs V}
    (Q \sqcap S\rcmp \rtrans{R}) \rcmp (R \sqcap \rtrans{Q}\rcmp S) 
\enskip.
\EDARQ
}%Proof
%}}}
%}}}
%}}}

%{{{ \subsection{Distributive Allegories}
\subsection{Distributive Allegories}

To the structure presented so far, we now
add the possibility of finding joins and a zero
together with distributivity of composition over joins.

%{{{ {Def}{distributive allegory}
\begin{Def}
A \emph{distributive allegory}\index{allegory}\index{distributive allegory}
is a tuple\newline
$\catC = (Obj_{\catC}, Mor_{\catC}, \_ : \_ \rel \_, \RId{}, \rcmp, \RAconverse{}, \reland, \relor, \RO{})$
where the following hold:
\begin{enumerate}
\renewcommand{\labelenumi}{\roman{enumi})}
\item The tuple $(Obj_{\catC}, Mor_{\catC}, \_ : \_ \rel \_, \RId{}, \rcmp, \RAconverse{}, \reland)$
  is an allegory, the so-called \emph{underlying allegory} of $\catC$.
\item Every homset $\CThom{\cal C}{\objA}{\objB}$ carries the structure
of a distributive lattice\index{distributive lattice}\index{lattice,
distributive} with
$\RAjoin_{\objA,\objB}$ for \emph{join}\index{join},
and zero element $\RO{\objA,\objB}$.

\item For all objects $\objA$, $\objB$ and $\objC$ and
 all morphisms $Q:\objA\rel \objB$,
the {\em zero law}\index{zero law} holds:
$$
   Q \rcmp \RO{\objB,\objC} = \RO{\objA,\objC}
\enskip.
$$

\item For all $Q:\objA\rel \objB$ and $R, R':\objB\rel \objC$,
{\em join-distributivity}\index{join-distributivity} holds:
\BD 
    Q\rcmp (R \relor R') 
   = 
    Q\rcmp R \relor Q\rcmp R' 
\enskip.
\EDQ
\end{enumerate}
\end{Def}
%}}}

We mention a few easily derivable facts.

%{{{ {Prop}\Proplabel{DistrAllProps}
\begin{Prop}\Proplabel{DistrAllProps}
Let $Q,Q':\objA \rel \objB$ and $R:\objB \rel \objC$
be morphisms in a distributive allegory.
Then:
\begin{enumerate}
\item $\rtrans{\RO{A,B}}=\RO{B,A}$.
\item $\RO{A,B}\rcmp R=\RO{A,C}$.
\item $\rtrans{(Q\relor Q')} = \rtrans{Q}\relor \rtrans{Q'}$.
\qed
\end{enumerate}
\end{Prop}
%}}}

For our Haskell module,
the new class \texttt{DistribAllegory} only needs to add two components:

\index{DistribAllegory@{\texttt{DistribAllegory} (class)}}%
\index{join@{\texttt{join}}}%
\index{bottom@{\texttt{bottom}}}%
\begin{code}
class        Allegory all obj mor =>
      DistribAllegory all obj mor | all -> obj, all -> mor where
  join   :: all -> mor -> mor -> mor
  bottom :: all -> obj -> obj -> mor
\end{code}

Distributive allegories with only finite homsets are locally complete,
according to the definition of \cite[2.22]{Freyd-Scedrov-1990}:

\begin{Def}
A distributive allegory is \emph{locally complete}
if every homset is a complete lattice,
and if composition and finite intersection distribute over arbitrary unions:
that is,
given $R : \objA \rel \objB$ and $\{S_i : \objB \rel \objC\}_{i \in I}$
one has
\BM
    R\rcmp(\bigrelor_{i\in I} S_i)
  =
    \bigrelor_{i\in I} (R \rcmp S_i)
\EM.
For empty $I$ we understand this to mean $R \rcmp \RO{} = \RO{}$.
\qed
\end{Def}
%}}}

%{{{ \subsection{Division Allegories}
\subsection{Division Allegories}

Demanding properties usually attributed to a division
operation characterises division allegories
among distributive allegories.

%{{{ {Def}\Deflabel{DivAll}
\begin{Def}\Deflabel{DivAll}
\strut\cite{Freyd-Scedrov-1990}
A {\it division allegory\/}\index{division allegory}
is a distributive allegory 
where for arbitrary relations 
$S:\objA\rel \objC$ and $R:\objB\rel \objC$,
the \emph{left residual} $S\lres R$ exists, defined by 
\BD
     Q\rcmp R \rsubs S
  \iff
     Q \rsubs S \lres R
  \qquad
     \mbox{for all $Q:\objA\rel \objB$}
  \enskip.
\EDQ
\end{Def}

On top of the left residual we may continue to define:

\begin{Def}
In a division allegory,
the \emph{right residual} may be defined via the left residual:
\BD
    Q \rres S
  \defeq
    \rtrans{(\rtrans{S} \lres \rtrans{Q})}
\ED
and fulfils a corresponding specification:
\BD
     Q\rcmp R \rsubs S
  \iff
     R \rsubs Q \rres S
  \qquad
     \mbox{for all $R:\objB\rel \objC$}
\ED
The \emph{symmetric quotient} is defined as the intersection of two residuals:
For $P : \objA \rel \objB$ and $Q : \objA \rel \objC$
we have $\syq{P}{Q} : \objB \rel \objC$ with
\BD
    \syq{R}{S} = R \rres S \reland \rtrans{R} \lres \rtrans{S}
  \enskip.
\EDQ
\end{Def}

This symmetric quotient has originally been defined
in the context of heterogeneous relation algebras
\cite{Berghammer-Schmidt-Zierer-1986,Berghammer-Schmidt-Zierer-1989}
and is --- modulo conversion of the arguments ---
exactly the \emph{symmetric division}\index{symmetric division}
as introduced by Freyd and Scedrov
for division allegories \cite[2.35]{Freyd-Scedrov-1990}.
%}}}

For concrete relations $R$ and $S$,
the symmetric quotient relates
elements $r$ from the range of $R$
with elements $s$ from the range of $S$
exactly if the inverse image of $r$ under $R$
is the same as the inverse image of $s$ under $S$,
or, in the language of predicate logic:
$$
    (r,s) \in \syq{R}{S}
  \qquad\iff\qquad
    \forall x : (x,r) \in R \leftrightarrow (x,s) \in S
$$
(Riguet had introduced the unary operation of ``noyeau''
in the homogeneous setting, which can now be seen as defined by
${\sf noy}(R) = \syq{R}{R}$, in \cite{Riguet-1948}.)

\index{DivisionAllegory@{\texttt{DivisionAllegory} (class)}}%
\index{rres@{\texttt{rres}}}%
\index{lres@{\texttt{lres}}}%
\index{syq@{\texttt{syq}}}%
\begin{code}
class  DistribAllegory all obj mor =>
      DivisionAllegory all obj mor | all -> obj, all -> mor where
  rres :: all -> mor -> mor -> mor
  lres :: all -> mor -> mor -> mor
  syq  :: all -> mor -> mor -> mor
\end{code}

\noindent 
The conditions 
of meet-subdistributivity, join-distributivity and zero law
listed for distributive allegories
are not required in the axiomatisation of division allegories,
since here they can be deduced using the residuals.

On the other hand,
residuals always exist in a locally complete distributive allegory,
so every locally finite distributive allegory is a division allegory.
%}}}

%{{{ \subsection{Dedekind Categories}
\subsection{Dedekind Categories}

Independent of Freyd and Scedrov,
Olivier and Serrato defined a kind of relation categories in
\cite{Olivier-Serrato-1980} which differs from division allegories
precisely by being
what is called ``locally complete''\index{locally complete}
in \cite[2.22]{Freyd-Scedrov-1990}:

%{{{ {Def}\Deflabel{DedCat}
\begin{Def}\Deflabel{DedCat}
\strut\cite{Olivier-Serrato-1980}
A {\it Dedekind category\/}\index{Dedekind category}
 is a division allegory ${\cal C}$
where every homset $\CThom{\cal C}{A}{B}$ is a
\emph{complete}\index{complete lattice}\index{lattice, complete} lattice with greatest element
  $\RL{A,B}$, called \emph{universal relation}\index{universal relation}.
\qed
\end{Def}
%}}}

\index{DedCat@{\texttt{DedCat} (class)}}%
\index{top@{\texttt{top}}}%
\begin{code}
class DivisionAllegory ded obj mor =>
      DedCat           ded obj mor | ded -> obj, ded -> mor where
  top  :: ded -> obj -> obj -> mor
\end{code}

In contrast to \cite[2.22]{Freyd-Scedrov-1990},
the infinite variants of
meet-subdistributivity and join-distribu\-tivity,
which form part of the definition of local completeness,
need not be listed here,
since they follow from the complete lattice structure
via the presence of residuals. On the other hand,
the full definition of local completeness implies the existence
of residuals \cite[2.315]{Freyd-Scedrov-1990},
such that a Dedekind category is just a locally complete
distributive allegory.

We still separate the Haskell definitions of
distributive allegories, division allegories and Dedekind categories
since these Haskell definitions themselves are equally adequate
to deal with infinite structures, and may also prove useful for that purpose.
It is only our tests that rely on finiteness.
%}}}

%{{{ \subsection{Relation Algebras}
\subsection{Relation Algebras}

\medskip
\noindent
If all morphisms of a Dedekind category have complements\index{complement}, 
the Dedekind category is equivalent to a Schr\"oder category:

%{{{ {Def}{Schr\"oder category}
\begin{Def}
A \emph{Schr\"oder category}\index{Schroder category@Schr\"oder category} \cite{Olivier-Serrato-1980,Jonsson-1988}
is a Dedekind category
where every homset is a Boolean lattice\index{Boolean lattice}\index{lattice, Boolean}.
\qed
\end{Def}

The complement of a relation $R$ is written $\relnot{R}$.
%}}}

%{{{ Dedekind rule is equivalent to the Schr\"oder equivalences
It is well-known that in a distributive allegory with Boolean lattices
as homsets, the Dedekind rule\index{Dedekind rule} is equivalent
to the \emph{Schr\"oder equivalences}\index{Schroder equivalences@Schr\"oder equivalences}:
$$
    Q\rcmp R\rsubs S 
  \qquad\iff\qquad
    \rtrans{Q}\rcmp\relnot S \rsubs \relnot R
  \qquad\iff\qquad
    \relnot S\rcmp\rtrans{R} \rsubs \relnot Q
$$
for all relations $Q:\objA \rel \objB$, $R: \objB \rel \objC$ and $S: \objA \rel \objC$.
For the first direction, it is sufficient
to show that with the Dedekind rule,
$Q\rcmp R\rsubs S$ implies
$\rtrans{Q}\rcmp\relnot S \rsubs \relnot R$:
assume $Q\rcmp R\rsubs S$, then that is equivalent to 
$Q\rcmp R\sqcap\relnot S = \RAbot$, and we have
$$
   \rtrans{Q}\rcmp\relnot S \sqcap R 
        \rsubs 
          \rtrans{Q}\rcmp(\relnot S \sqcap Q\rcmp R) 
        =
          \RAbot
\enskip.
$$
Conversely, assume that the Schr\"oder equivalences hold.
Then \cite{Schmidt-Stroehlein-1985} shows:
\BDAR {}\sepA{}
    Q\rcmp R 
  \sepBR{=}{\mbox{Boolean lattice}}
    ((Q \sqcap S \rcmp \rtrans{R}) \sqcup (Q \sqcap \relnot{S\rcmp
    \rtrans{R}}))\rcmp ((R \sqcap \rtrans{Q}\rcmp S) \sqcup  (R \sqcap 
    \relnot{\rtrans{Q}\rcmp S}))
  \sepBR{=}{\mbox{join-distributivity}}
    (Q \sqcap S \rcmp \rtrans{R})\rcmp(R \sqcap \rtrans{Q}\rcmp S)
    \sqcup
    (Q \sqcap S \rcmp \rtrans{R})\rcmp(R \sqcap \relnot{\rtrans{Q}\rcmp S})
  \sepX{}
    \relor\ {} 
    (Q \sqcap \relnot{S\rcmp\rtrans{R}})\rcmp(R \sqcap \rtrans{Q}\rcmp S)
    \sqcup
    (Q \sqcap \relnot{S\rcmp\rtrans{R}})\rcmp
                        (R \sqcap \relnot{\rtrans{Q}\rcmp S})
  \sepBR{\rsubs}{\forall U,V\,: U \reland V \rsubs U}
    (Q \sqcap S \rcmp \rtrans{R})\rcmp(R \sqcap \rtrans{Q}\rcmp S)
    \sqcup Q\rcmp\relnot{\rtrans{Q}\rcmp S}
    \sqcup \relnot{S\rcmp\rtrans{R}}\rcmp R
  \sepBR{\rsubs}{\mbox{Schr\"oder}}
    (Q \sqcap S \rcmp \rtrans{R})\rcmp(R \sqcap \rtrans{Q}\rcmp S)
    \sqcup \relnot{S}
%\enskip .
\EDAR
yielding the Dedekind rule
$
    Q\rcmp R \sqcap S
  \rsubs 
    (Q \sqcap S \rcmp \rtrans{R})\rcmp(R \sqcap \rtrans{Q}\rcmp S)
$ via Boolean lattice properties.
%}}}

\medskip
%{{{ in Schr\"oder categories the residual is defined {\sl a priori}
Furthermore, the Schr\"oder equivalences allow us to calculate:
$$
    Q\rcmp R\rsubs S
  \qquad\iff\qquad
    \relnot S\rcmp\rtrans{R} \rsubs \relnot Q
  \qquad\iff\qquad
    Q \rsubs \relnot{ \relnot S\rcmp\rtrans{R}}
$$
Therefore, we have $S/R = \relnot{ \relnot S\rcmp\rtrans{R}}$,
so that in Schr\"oder categories the residual is defined {\sl a priori}
and need not be listed in the axiomatisation.
%}}}

\medskip
The concept of Schr\"oder categories
can be considered as a slightly relaxed variant
of the following, older, concept of heterogeneous relation algebras:

%{{{ {Def}\Deflabel{HetRelAlg}
\begin{Def}\Deflabel{HetRelAlg}
A \emph{heterogeneous relation algebra}
\cite{Schmidt-1977,Schmidt-1981a,Schmidt-Stroehlein-1989,Schmidt-Stroehlein-1993}
is a\linebreak Schr\"oder category where every homset
is an \emph{atomic}\index{atomic lattice}\index{lattice, atomic}
and complete Boolean lattice.
\qed
\end{Def}
%}}}

In many contexts, non-triviality of the Boolean lattices is also demanded,
namely $\RL{\objA,\objB} \neq \RO{\objA,\objB}$ for all objects $\objA$ and
$\objB$,
and also the following rule:

%{{{ {Def}{Tarski rule}
\begin{Def}
The \emph{Tarski rule}\index{Tarski rule} holds in a
heterogeneous relation algebra
iff
\BD
    R \neq \RO{\objA,\objB}
  \;\iff\;
    \RL{\objC,\objA}\rcmp R\rcmp\RL{\objB,\objD} = \RL{\objC,\objD}
\ED
holds for all $\objA,\objB,\objC,\objD : \CTobj{\cal R}$
and $R : \objA \rel \objB$.
\qed
\end{Def}
%}}}

Both of these constraints, however,
are inappropriate for our search for computationally relevant
non-standard relation algebras, so they are not included in the definition here.

Obviously, a Schr\"oder category with finite homsets
is always a heterogeneous relation algebra,
so we directly introduce an interface for the latter:

\index{RelAlg@{\texttt{RelAlg} (class)}}%
\index{compl@{\texttt{compl}}}%
\begin{code}
class DedCat ra obj mor => RelAlg ra obj mor | ra -> obj, ra -> mor where
  compl :: ra -> mor -> mor
\end{code}
%}}}


%% Local variables:
%% folded-file: t
%% fold-internal-margins: 0
%% eval: (fold-set-marks "%{{{ " "%}}}")
%% eval: (fold-whole-buffer)
%% end:
