
\section{Interoperability With the Class Interface}\sectlabel{Instances}

Although the class interface of \sectref{RelAlgClasses}
and the dictionary records of \sectref{RelAlg}
are completely independent  of each other,
it is easy to obtain interoperability between the two interfaces.

In this section we first instantiate the classes of \sectref{RelAlgClasses}
for the dictionary types of \sectref{RelAlg} in \subsectref{Instances}.
We then show in \subsectref{ReverseInstances} how to obtain explicit
dictionaries from class interfaces,
and apply this to transfer the test functions defined in \sectref{RelAlg}
from the dictionary setting to the class setting in
\subsectref{TestTransfer}.

\begin{code}
module RelAlgInstances where

import RelAlg
import RelAlgClasses
import Properties
import Atomset
\end{code}

%{{{ \subsection{Instantiating the Class Interface}\subsectlabel{Instances}
\subsection{Instantiating the Class Interface}\subsectlabel{Instances}

%{{{ \subsubsection*{Categories}
\subsubsection*{Categories}

\begin{code}
instance Category (Cat obj mor) obj mor where
  isObj   = cat_isObj
  isMor   = cat_isMor
  objects = cat_objects
  homset  = cat_homset
  source  = cat_source
  target  = cat_target
  idmor   = cat_idmor
  comp    = cat_comp

instance Category (All obj mor) obj mor where
  isObj   = all_isObj
  isMor   = all_isMor
  objects = all_objects
  homset  = all_homset
  source  = all_source
  target  = all_target
  idmor   = all_idmor
  comp    = all_comp

instance Category (DistrAll obj mor) obj mor where
  isObj   = distrAll_isObj
  isMor   = distrAll_isMor
  objects = distrAll_objects
  homset  = distrAll_homset
  source  = distrAll_source
  target  = distrAll_target
  idmor   = distrAll_idmor
  comp    = distrAll_comp

instance Category (DivAll obj mor) obj mor where
  isObj   = divAll_isObj
  isMor   = divAll_isMor
  objects = divAll_objects
  homset  = divAll_homset
  source  = divAll_source
  target  = divAll_target
  idmor   = divAll_idmor
  comp    = divAll_comp

instance Category (Ded obj mor) obj mor where
  isObj   = ded_isObj
  isMor   = ded_isMor
  objects = ded_objects
  homset  = ded_homset
  source  = ded_source
  target  = ded_target
  idmor   = ded_idmor
  comp    = ded_comp

instance Category (RA obj mor) obj mor where
  isObj   = ra_isObj
  isMor   = ra_isMor
  objects = ra_objects
  homset  = ra_homset
  source  = ra_source
  target  = ra_target
  idmor   = ra_idmor
  comp    = ra_comp
\end{code}
%}}}

%{{{ \subsubsection*{Allegories}
\subsubsection*{Allegories}

\begin{code}
instance Allegory (All obj mor) obj mor where
  converse = all_converse
  meet     = all_meet
  incl     = all_incl

instance Allegory (DistrAll obj mor) obj mor where
  converse = distrAll_converse
  meet     = distrAll_meet
  incl     = distrAll_incl

instance Allegory (DivAll obj mor) obj mor where
  converse = divAll_converse
  meet     = divAll_meet
  incl     = divAll_incl

instance Allegory (Ded obj mor) obj mor where
  converse = ded_converse
  meet     = ded_meet
  incl     = ded_incl

instance Allegory (RA obj mor) obj mor where
  converse = ra_converse
  meet     = ra_meet
  incl     = ra_incl
\end{code}
%}}}

%{{{ \subsubsection*{Distributive Allegories}
\subsubsection*{Distributive Allegories}

\begin{code}
instance DistribAllegory (DistrAll obj mor) obj mor where
  join   = distrAll_join
  bottom = distrAll_bottom

instance DistribAllegory (DivAll obj mor) obj mor where
  join   = divAll_join
  bottom = divAll_bottom

instance DistribAllegory (Ded obj mor) obj mor where
  join   = ded_join
  bottom = ded_bottom

instance DistribAllegory (RA obj mor) obj mor where
  join   = ra_join
  bottom = ra_bottom
\end{code}
%}}}

%{{{ \subsubsection*{Division Allegories}
\subsubsection*{Division Allegories}


\begin{code}
instance DivisionAllegory (DivAll obj mor) obj mor where
  rres = divAll_rres
  lres = divAll_lres
  syq  = divAll_syq

instance DivisionAllegory (Ded obj mor) obj mor where
  rres = ded_rres
  lres = ded_lres
  syq  = ded_syq

instance DivisionAllegory (RA obj mor) obj mor where
  rres = ra_rres
  lres = ra_lres
  syq  = ra_syq
\end{code}
%}}}

%{{{ \subsubsection*{Dedekind Categories}
\subsubsection*{Dedekind Categories}

\begin{code}
instance DedCat (Ded obj mor) obj mor where
  top  = ded_top

instance DedCat (RA obj mor) obj mor where
  top  = ra_top
\end{code}
%}}}

%{{{ \subsubsection*{Relation Algebras}
\subsubsection*{Relation Algebras}

\begin{code}
instance RelAlg (RA obj mor) obj mor where
  compl = ra_compl
\end{code}
%}}}}
%}}}

%{{{ \subsection{Reverse Instances}\subsectlabel{ReverseInstances}
\subsection{Reverse Instances}\subsectlabel{ReverseInstances}

\index{catDict@{\texttt{catDict}}}%
\begin{code}
catDict :: Category cat obj mor => cat -> Cat obj mor
catDict c = Cat
  {cat_isObj   = isObj c
  ,cat_isMor   = isMor c
  ,cat_objects = objects c
  ,cat_homset  = homset c
  ,cat_source  = source c
  ,cat_target  = target c
  ,cat_idmor   = idmor c
  ,cat_comp    = comp c
  }
\end{code}

\index{allDict@{\texttt{allDict}}}%
\begin{code}
allDict :: Allegory all obj mor => all -> All obj mor
allDict a = All
  {all_cat = catDict a
  ,all_converse = converse a
  ,all_meet = meet a
  ,all_incl = incl a
  }
\end{code}

\index{distrAllDict@{\texttt{distrAllDict}}}%
\begin{code}
distrAllDict :: (DistribAllegory da obj mor, Eq mor) => da -> DistrAll obj mor
distrAllDict da = da' where
 da' = DistrAll
  {distrAll_all  = allDict da
  ,distrAll_bottom = bottom da
  ,distrAll_join = join da
  ,distrAll_atomset = distrAll_atomset_default da'
  ,distrAll_atoms   = distrAll_atoms_default da'
  }
\end{code}

\index{divAllDict@{\texttt{divAllDict}}}%
\begin{code}
divAllDict :: (DivisionAllegory da obj mor, Eq mor) => da -> DivAll obj mor
divAllDict da = DivAll
  {divAll_distrAll = distrAllDict da
  ,divAll_rres = rres da
  ,divAll_lres = lres da
  ,divAll_syq  = syq da
  }
\end{code}

\index{dedDict@{\texttt{dedDict}}}%
\begin{code}
dedDict	:: (DedCat ded obj mor, Eq mor) => ded -> Ded obj mor
dedDict ded = Ded
  {ded_divAll  = divAllDict ded
  ,ded_top  = top ded
  }
\end{code}

\index{raDict@{\texttt{raDict}}}%
\begin{code}
raDict :: (RelAlg ra obj mor, Eq mor) => ra -> RA obj mor
raDict ra = RA
  {ra_ded = dedDict ra
  ,ra_compl = compl ra
  }
\end{code}

\index{acatDict@{\texttt{acatDict}}}%
\index{aallDict@{\texttt{aallDict}}}%
\begin{code}
acatDict ::(DistribAllegory da obj mor, Ord obj, Eq mor) => da -> ACat obj mor
acatDict = distrAll_acat . distrAllDict

aallDict ::(DistribAllegory da obj mor, Ord obj, Eq mor) => da -> AAll obj mor
aallDict = distrAll_aall . distrAllDict
\end{code}
%}}}

%{{{ \subsection{Transfer of Tests}\subsectlabel{TestTransfer}
\subsection{Transfer of Tests}\subsectlabel{TestTransfer}

Using these ``dictionary explicators'',
we can lift our testing machinery to the class setting:

\index{category_TEST@{\texttt{category\_TEST}}}%
\index{allegory_TEST@{\texttt{allegory\_TEST}}}%
\index{distribAllegory_TEST@{\texttt{distribAllegory\_TEST}}}%
\index{distribAllegory_join_incl_TEST@{\texttt{distribAllegory\_join\_incl\_TEST}}}%
\index{divisionAllegory_rres_TEST@{\texttt{divisionAllegory\_rres\_TEST}}}%
\index{divisionAllegory_lres_TEST@{\texttt{divisionAllegory\_lres\_TEST}}}%
\index{divisionAllegory_syq_resTEST@{\texttt{divisionAllegory\_syq\_resTEST}}}%
\index{allegory_syq_directTEST@{\texttt{allegory\_syq\_directTEST}}}%
\index{dedCat_top_incl_TEST@{\texttt{dedCat\_top\_incl\_TEST}}}%
\index{relAlg_compl_TEST@{\texttt{relAlg\_compl\_TEST}}}%
\index{relAlg_TEST@{\texttt{relAlg\_TEST}}}%
\index{relAlg_TEST_ALL@{\texttt{relAlg\_TEST\_ALL}}}%
\begin{code}
category_TEST ::
  (Category cat obj mor, Eq obj, Eq mor) => cat -> TestResult obj mor
category_TEST = cat_TEST . catDict

allegory_TEST ::
  (Allegory all obj mor, Eq obj, Eq mor) => all -> TestResult obj mor
allegory_TEST = all_TEST . allDict

distribAllegory_TEST ::
  (DistribAllegory da obj mor, Eq obj, Eq mor) => da -> TestResult obj mor
distribAllegory_TEST = distrAll_TEST . distrAllDict

distribAllegory_join_incl_TEST ::
  (DistribAllegory da obj mor, Eq obj, Eq mor) => da -> TestResult obj mor
distribAllegory_join_incl_TEST = distrAll_join_incl_TEST . distrAllDict

divisionAllegory_rres_TEST, divisionAllegory_lres_TEST
 , divisionAllegory_syq_resTEST
   :: (DivisionAllegory da obj mor, Eq obj, Eq mor) => da -> TestResult obj mor
divisionAllegory_rres_TEST   = divAll_rres_TEST   . divAllDict
divisionAllegory_lres_TEST   = divAll_lres_TEST   . divAllDict
divisionAllegory_syq_resTEST = divAll_syq_resTEST . divAllDict

allegory_syq_directTEST :: (Allegory all obj mor, Eq obj, Eq mor) =>
   (mor -> mor -> Maybe mor) -> all -> TestResult obj mor
allegory_syq_directTEST syq = all_syq_directTEST syq . allDict

dedCat_top_incl_TEST ::
  (DedCat ded obj mor, Eq obj, Eq mor) => ded -> TestResult obj mor
dedCat_top_incl_TEST = ded_top_incl_TEST . dedDict

relAlg_compl_TEST, relAlg_TEST
   :: (RelAlg ra obj mor, Eq obj, Eq mor) => ra -> TestResult obj mor
relAlg_compl_TEST = ra_compl_TEST . raDict
relAlg_TEST       = ra_TEST       . raDict

relAlg_TEST_ALL
   :: (RelAlg ra obj mor, Eq obj, Ord mor) => ra -> TestResult obj mor
relAlg_TEST_ALL   = ra_TEST_ALL   . raDict
\end{code}

This makes the following direct queries possible:
\begin{session}
HugsMain> perform category_TEST   ra_McKenzie
No results.

HugsMain> perform allegory_TEST   ra_LRNnoc
No results.

HugsMain> perform relAlg_TEST_ALL ra_Winter
No results.
\end{session}
%%%
%%% perform category_TEST   ra_McKenzie
%%% perform allegory_TEST   ra_LRNnoc
%%% perform relAlg_TEST_ALL ra_Winter

We also transfer other tests:

\index{uniform_TEST@{\texttt{uniform\_TEST}}}%
\index{allegory_partialUnit_TEST@{\texttt{allegory\_partialUnit\_TEST}}}%
\index{dedCat_partialUnit_TEST@{\texttt{dedCat\_partialUnit\_TEST}}}%
\index{allegory_partialUnit_unit_TEST@{\texttt{allegory\_partialUnit\_unit\_TEST}}}%
\index{dedCat_units@{\texttt{dedCat\_units}}}%
\index{isNonemptyProduct@{\texttt{isNonemptyProduct}}}%
\index{nonemptyProducts@{\texttt{nonemptyProducts}}}%
\index{nonemptyProducts1@{\texttt{nonemptyProducts1}}}%
\index{unsharp@{\texttt{unsharp}}}%
\begin{code}
uniform_TEST :: (DedCat ded obj mor, Eq obj, Eq mor) => ded -> TestResult obj mor
uniform_TEST = ded_uniform_TEST . dedDict

allegory_partialUnit_TEST :: (Allegory all obj mor, Eq obj, Eq mor) =>
  obj -> all -> TestResult obj mor
allegory_partialUnit_TEST u = all_partialUnit_TEST u . allDict

dedCat_partialUnit_TEST :: (DedCat all obj mor, Eq obj, Eq mor) =>
  obj -> all -> TestResult obj mor
dedCat_partialUnit_TEST u = ded_partialUnit_TEST u . dedDict

allegory_partialUnit_unit_TEST :: (Allegory all obj mor, Eq obj, Eq mor) =>
  obj -> all -> TestResult obj mor
allegory_partialUnit_unit_TEST u = all_partialUnit_unit_TEST u . allDict

dedCat_units :: (DedCat ded obj mor, Eq obj, Eq mor) => ded -> [obj]
dedCat_units = ded_units . dedDict

isNonemptyProduct :: (DedCat ded obj mor, Eq obj, Eq mor) =>
                     obj -> obj -> obj -> mor -> mor -> ded -> TestResult obj mor
isNonemptyProduct oA oB oP pA pB = ded_isNonemptyProduct oA oB oP pA pB . dedDict

nonemptyProducts ::
  (DedCat ded obj mor, Ord obj, Eq mor) => ded -> [Product obj mor]
nonemptyProducts = ded_NonemptyProducts . dedDict

nonemptyProducts1 :: (DedCat ded obj mor, Eq obj, Eq mor) =>
                     obj -> obj -> obj -> ded -> [Product obj mor]
nonemptyProducts1 oA oB oP = ded_NonemptyProducts1 oA oB oP . dedDict

unsharp :: (DedCat ded obj mor, Eq obj, Eq mor) =>
           Product obj mor -> ded -> TestResult obj mor
unsharp p = ded_unsharp p . dedDict
\end{code}

Now we can directly formulate queries such as in the following session:

\begin{session}
HugsMain> nonemptyProducts ra_Maddux
[(B,C,A,SetMor ({At1},A,B),SetMor ({At1},A,C))]
HugsMain> performAll (unsharp (head $ nonemptyProducts ra_Maddux)) ra_Maddux
=== Test Start ===
unsharpness example
 Objects:
  D
  E
 Morphisms:
  SetMor ({At1},D,B)
  SetMor ({At1},D,C)
  SetMor ({At1},B,E)
  SetMor ({At1},C,E)
  SetMor ({At2},D,E)
  SetMor ({At1, At2},D,E)
unsharpness example
 Objects:
  E
  D
 Morphisms:
  SetMor ({At1},E,B)
  SetMor ({At1},E,C)
  SetMor ({At1},B,D)
  SetMor ({At1},C,D)
  SetMor ({At2},E,D)
  SetMor ({At1, At2},E,D)
=== Test End   ===
\end{session}
%}}}


%% Local variables:
%% folded-file: t
%% fold-internal-margins: 0
%% eval: (fold-set-marks "%{{{ " "%}}}")
%% eval: (fold-whole-buffer)
%% end:
