\section{Sub-Algebras}

Forming sub-algebras\index{sub-algebras} is another standard algebra construction mechanism.
It is particularly promising in connection with relation algebras,
as it is known that one may sometimes take a subset 
of all the available relations
and will still maintain the basic structure.


\begin{code}
module SubAlg where

import FiniteMaps
import Sets
import ExtPrel
import RelAlg
\end{code}

Our approach is to use an auxiliary \texttt{SubCat} data structure
to contain the information necessary to define a sub-algebra
of a given algebra.
From the mathematical point of view,
this additional information consists of the object set
and of the homsets of the sub-algebra;
all operations are preserved.

%{{{ \subsection*{Using Sub-Algebras}
\subsection*{Using Sub-Algebras}

This information can then be used to obtain a sub-algebra from an algebra
for all kinds of algebras under consideration in this report:

\index{@{\texttt{}}}%
\index{@{\texttt{}}}%
\index{@{\texttt{}}}%
\index{@{\texttt{}}}%
\index{@{\texttt{}}}%
\index{@{\texttt{}}}%
\begin{code}
sub_cat      :: (Ord o, Ord m) => SubCat o m -> Cat      o m -> Cat      o m
sub_all      :: (Ord o, Ord m) => SubCat o m -> All      o m -> All      o m
sub_distrAll :: (Ord o, Ord m) => SubCat o m -> DistrAll o m -> DistrAll o m
sub_divAll   :: (Ord o, Ord m) => SubCat o m -> DivAll   o m -> DivAll   o m
sub_ded      :: (Ord o, Ord m) => SubCat o m -> Ded      o m -> Ded      o m
sub_ra       :: (Ord o, Ord m) => SubCat o m -> RA       o m -> RA       o m
\end{code}

Higher-level structures only add operations,
so the only difference is in the underlying structure of the next lower level.
In the following functions we always assume the
\texttt{SubCat} structure to have been checked for closedness under the relevant operations and therefore omit tests:

\begin{code}
sub_ra       s c = c {ra_ded          = sub_ded      s (ra_ded          c)}
sub_ded      s c = c {ded_divAll      = sub_divAll   s (ded_divAll      c)}
sub_divAll   s c = c {divAll_distrAll = sub_distrAll s (divAll_distrAll c)}
sub_distrAll s c = c {distrAll_all    = sub_all      s (distrAll_all    c)}
sub_all      s c = c {all_cat         = sub_cat      s (all_cat         c)}
\end{code}

For categories,
the sub-category obtains redefined object and homset components,
but inherits the unchanged operations:

\begin{code}
sub_cat s@(SubCat objs hs) c = c
  {cat_isObj   = (\ o -> o `elemSet` objs)
  ,cat_isMor   = (\ a b m -> m `elemSet` lookupDftFM hs zeroSet (a,b))
  ,cat_objects = toListSet objs
  ,cat_homset  = curry (toListSet . lookupDftFM hs zeroSet)
  }
\end{code}

The preferred interface, however,
takes an arbitrary \texttt{SubCat} data structure,
closes it under the relevant operations,
and applies the above functions to obtain the corresponding sub-algebra:

\index{subCat@{\texttt{subCat}}}%
\index{subAll@{\texttt{subAll}}}%
\index{subDistrAll@{\texttt{subDistrAll}}}%
\index{subDivAll@{\texttt{subDivAll}}}%
\index{subDed@{\texttt{subDed}}}%
\index{subRA@{\texttt{subRA}}}%
\begin{code}
subCat      :: (Ord o, Ord m) => SubCat o m -> Cat      o m -> Cat      o m
subAll      :: (Ord o, Ord m) => SubCat o m -> All      o m -> All      o m
subDistrAll :: (Ord o, Ord m) => SubCat o m -> DistrAll o m -> DistrAll o m
subDivAll   :: (Ord o, Ord m) => SubCat o m -> DivAll   o m -> DivAll   o m
subDed      :: (Ord o, Ord m) => SubCat o m -> Ded      o m -> Ded      o m
subRA       :: (Ord o, Ord m) => SubCat o m -> RA       o m -> RA       o m
\end{code}

These functions are defined below,
after introduction of the necessary machinery.
%}}}

%{{{ \subsection*{Sub-Algebra Closure Machinery}
\subsection*{Sub-Algebra Closure Machinery}

By introducing a very abstract and comprehensive interface
to the \texttt{SubCat} data type,
one might be able to define sub-algebras without resorting
to even \texttt{Eq} instances for the object
and morphism data types.
However, we think this is not worth the effort.
For the sake of efficiency,
we even demand \texttt{Ord} instances
and do not consider this as a serious restriction
for the kind of uses we have in mind.

For \texttt{obj} and \texttt{mor} types in the \texttt{Ord} class
we can directly implement the \texttt{SubCat} data type
via standard set and finite map data structures%
\footnote{These are imported from the modules
\texttt{Sets} and \texttt{FiniteMaps}
taken from Manuel Chakravarty's compiler toolkit
and slightly modified for our purposes.
We prefer this variant over those provided by GHC
(from which they are derived)
for portability reasons since they work with other Haskell implementations
as well and do not give rise to name clashes with GHC.}:

\index{SubCat@{\texttt{SubCat}}}%
\index{sub_objects@{\texttt{sub\_objects}}}%
\index{sub_homset@{\texttt{sub\_homset}}}%
\begin{code}
data SubCat obj mor = SubCat
  {sub_objects :: Set obj
  ,sub_homset :: FiniteMap (obj,obj) (Set mor)
  }
\end{code}

Simple lookup functions:

\index{sub_isMor@{\texttt{sub\_isMor}}}%
\index{sub_isEmpty@{\texttt{sub\_isEmpty}}}%
\begin{code}
sub_isMor :: (Ord obj, Ord mor) => SubCat obj mor -> obj -> obj -> mor -> Bool
sub_isMor (SubCat objs mors) a b m =
   case lookupFM mors (a,b) of
     Nothing -> False
     Just mors -> m `elemSet` mors

sub_isEmpty (SubCat objs hs) = isZeroSet objs && isZeroFM hs
\end{code}

Adding a single morphism to some homset:

\begin{code}
addToHomset :: (Ord obj, Ord mor) => obj -> obj -> mor 
                                         -> SubCat obj mor -> SubCat obj mor
addToHomset a b m (SubCat objs hs) = SubCat objs $ addToFM (a,b) mors' hs
 where mors' = case lookupFM hs (a,b) of
                 Nothing -> unitSet m
                 Just mors -> addToSet m mors
\end{code}

Joining two (intermediate) \texttt{SubCat} data structures:

\index{subcat_join@{\texttt{subcat\_join}}}%
\begin{code}
subcat_join (SubCat objs1 hs1) (SubCat objs2 hs2) =
  SubCat (objs1 `joinSet` objs2) (foldFM f hs1 hs2)
 where f p ms hs = addToFM p ms' hs
        where ms' = case lookupFM hs p of
                      Nothing -> ms
                      Just mors -> mors `joinSet` ms
\end{code}

\index{SubCatDiff@{\texttt{SubCatDiff}}}%
\index{SubCatClosure@{\texttt{SubCatClosure}}}%
\begin{code}
type SubCatDiff obj mor = SubCat obj mor

type SubCatClosure obj mor = STFun (SubCat obj mor) Bool
\end{code}

\texttt{SubCatClosure}s can be composed, yielding the conjunction of the
intermediate results:

\index{scComp@{\texttt{scComp}}}%
\begin{code}
scComp :: (Ord obj, Ord mor) =>
           SubCatClosure obj mor -> SubCatClosure obj mor -> 
           SubCatClosure obj mor
scComp f g = do b1 <- f
                b2 <- g
                return (b1 && b2)
\end{code}

\texttt{SubCatClosure}s will usually be created via \texttt{scStep}
from a function
calculating an incremental \texttt{SubCatDiff} from an intermediate \texttt{SubCat}:

\begin{code}
scStep :: (Ord obj, Ord mor) =>
          (SubCat obj mor -> SubCatDiff obj mor) -> SubCatClosure obj mor
scStep f = STFun (\ s -> let d = f s
                             b = sub_isEmpty d
                             s' = subcat_join s d
                         in (s', b))
\end{code}

Applying a \texttt{SubCatClosure} means iterating it until the
incremental difference is empty:

\index{scClose@{\texttt{scClose}}}%
\begin{code}
scClose :: (Ord obj, Ord mor) =>
           SubCatClosure obj mor -> SubCat obj mor -> SubCat obj mor
scClose step s = fst $ applySTFun iter s
  where iter = do b <- step
                  if b then return () else iter
\end{code}
%}}}

%{{{ \subsection*{Sub-Algebra Closure Functions}
\subsection*{Sub-Algebra Closure Functions}

After thus establishing the machinery,
we now present the individual difference creation functions;
these are then used by the sub-algebra generators.

\medskip
The simplest closure is creating a sub-category induced by a set of objects;
this only has to take all morphisms between those objects
and needs not be iterated:

\index{cat_homset_close@{\texttt{cat\_homset\_close}}}%
\begin{code}
cat_homset_closeSubCatDiff :: (Ord obj, Ord mor) => 
                          Cat obj mor -> SubCat obj mor -> SubCatDiff obj mor
cat_homset_closeSubCatDiff c s =
  let objects = toListSet $ sub_objects s
      idmor = cat_idmor c
      homset = cat_homset c
      adds = do a <- objects
                b <- objects
                f <- homset a b
                if sub_isMor s a b f then []
                                     else [addToHomset a b f]
  in foldr id (SubCat zeroSet zeroFM) adds

cat_homset_close :: (Ord obj, Ord mor) => 
                          Cat obj mor -> SubCat obj mor -> SubCat obj mor
cat_homset_close c s = s `subcat_join` cat_homset_closeSubCatDiff c s
\end{code}

A more dedicated function could eliminate the
cost of morphism lookup,
which is logarithmic in the sizes of the object set and of the homset in question.
However, we postpone this until it is felt to be a bottle neck.

For turning arbitrary \texttt{SubCat} data structures
into legal sub-category descriptions,
we first of all have to make sure that all identities are present:

\begin{code}
cat_id_closeSubCatDiff :: (Ord obj, Ord mor) => 
                          Cat obj mor -> SubCat obj mor -> SubCatDiff obj mor
cat_id_closeSubCatDiff c s =
  let objects = toListSet $ sub_objects s
      idmor = cat_idmor c
      notthere o i = not (sub_isMor s o o i)
  in foldSet (\ o r -> let i = idmor o in if notthere o i
                       then addToHomset o o i r
                       else r)
             (SubCat zeroSet zeroFM)
             (sub_objects s)
\end{code}

Next we close the homsets under composition:

\begin{code}
cat_comp_closeSubCatDiff :: (Ord obj, Ord mor) => 
                          Cat obj mor -> SubCat obj mor -> SubCatDiff obj mor
cat_comp_closeSubCatDiff c s@(SubCat objs hs) =
  let objects = toListSet $ sub_objects s
      homset a b = toListSet $ lookupDftFM hs zeroSet (a,b)
      (^) = cat_comp c
      comps = do a <- objects
                 b <- objects
                 f <- homset a b
                 c <- objects
                 g <- homset b c
                 let h = f ^ g
                 if sub_isMor s a c h then []
                                      else [addToHomset a c h]
  in foldr id (SubCat zeroSet zeroFM) comps
\end{code}

These two are sufficient for sub-categories:

\index{subCat@{\texttt{subCat}}}%
\begin{code}
cat_closeStep :: (Ord obj, Ord mor) => Cat obj mor -> SubCatClosure obj mor
cat_closeStep c = scStep (cat_id_closeSubCatDiff c) `scComp`
                  scStep (cat_comp_closeSubCatDiff c)

subCat s c = sub_cat (scClose (cat_closeStep c) s) c
\end{code}

For allegories, we have to close under conversion and meet:

\begin{code}
all_conv_closeSubCatDiff :: (Ord obj, Ord mor) => 
                          All obj mor -> SubCat obj mor -> SubCatDiff obj mor
all_conv_closeSubCatDiff c s@(SubCat objs hs) =
  let objects = toListSet $ sub_objects s
      homset a b = toListSet $ lookupDftFM hs zeroSet (a,b)
      conv = all_converse c
      (&&&) = all_meet c
      convs = do a <- objects
                 b <- objects
                 f <- homset a b
                 let g = conv f
                 (if sub_isMor s b a g then id
                                       else ((addToHomset b a g)  :))
                  $ do
                   g <- homset a b
                   let h = f &&& g
                   if sub_isMor s a b h then []
                                        else [addToHomset a b h]
  in foldr id (SubCat zeroSet zeroFM) convs
\end{code}

\index{subAll@{\texttt{subAll}}}%
\begin{code}
all_closeStep :: (Ord obj, Ord mor) => All obj mor -> SubCatClosure obj mor
all_closeStep c = cat_closeStep (all_cat c) `scComp`
                  scStep (all_conv_closeSubCatDiff c)

subAll s c = sub_all (scClose (all_closeStep c) s) c
\end{code}

For distributive allegories,
we simultaneously add bottom morphisms and close under joins:

\begin{code}
distrAll_closeSubCatDiff :: (Ord obj, Ord mor) => 
                         DistrAll obj mor -> SubCat obj mor -> SubCatDiff obj mor
distrAll_closeSubCatDiff c s@(SubCat objs hs) =
  let objects = toListSet $ sub_objects s
      homset a b = toListSet $ lookupDftFM hs zeroSet (a,b)
      bot = distrAll_bottom c
      (|||) = distrAll_join c
      adds = do a <- objects
                b <- objects
                let t = bot a b
                (if sub_isMor s a b t then id else ((addToHomset a b t) :))
                 $ do
                  f <- homset a b
                  g <- homset a b
                  let h = f ||| g
                  if sub_isMor s a b h then []
                                       else [addToHomset a b h]
  in foldr id (SubCat zeroSet zeroFM) adds
\end{code}

\index{subDistrAll@{\texttt{subDistrAll}}}%
\begin{code}
distrAll_closeStep :: (Ord obj, Ord mor) =>
                      DistrAll obj mor -> SubCatClosure obj mor
distrAll_closeStep c = all_closeStep (distrAll_all c) `scComp`
                       scStep (distrAll_closeSubCatDiff c)

subDistrAll s c = sub_distrAll (scClose (distrAll_closeStep c) s) c
\end{code}

For division allegories, we only have to add left and right residuals ---
symmetric quotients are added as intersections of those in the allegory step:

\begin{code}
divAll_closeSubCatDiff :: (Ord obj, Ord mor) => 
                          DivAll obj mor -> SubCat obj mor -> SubCatDiff obj mor
divAll_closeSubCatDiff c s@(SubCat objs hs) =
  let objects = toListSet $ sub_objects s
      homset a b = toListSet $ lookupDftFM hs zeroSet (a,b)
      lres = divAll_lres c
      rres = divAll_rres c
      comps = do a <- objects
                 b <- objects
                 f <- homset a b
                 c <- objects
                 (do g <- homset a c
                     let h = g `rres` f
                     if sub_isMor s b c h then []
                                          else [addToHomset b c h]
                  ) ++ (do
                     g <- homset c b
                     let h = f `lres` g
                     if sub_isMor s a b h then []
                                          else [addToHomset a b h]
                  )
  in foldr id (SubCat zeroSet zeroFM) comps
\end{code}

\index{subDivAll@{\texttt{subDivAll}}}%
\begin{code}
divAll_closeStep :: (Ord obj, Ord mor) => DivAll obj mor -> SubCatClosure obj mor
divAll_closeStep c = distrAll_closeStep (divAll_distrAll c) `scComp`
                     scStep (divAll_closeSubCatDiff c)

subDivAll s c = sub_divAll (scClose (divAll_closeStep c) s) c
\end{code}

For Dedekind categories,
we only need to add top morphisms:

\begin{code}
ded_closeSubCatDiff :: (Ord obj, Ord mor) => 
                       Ded obj mor -> SubCat obj mor -> SubCatDiff obj mor
ded_closeSubCatDiff c s@(SubCat objs hs) =
  let objects = toListSet $ sub_objects s
      homset a b = toListSet $ lookupDftFM hs zeroSet (a,b)
      top = ded_top c
      adds = do a <- objects
                b <- objects
                let t = top a b
                if sub_isMor s a b t then [] else [addToHomset a b t]
  in foldr id (SubCat zeroSet zeroFM) adds
\end{code}

\index{subDed@{\texttt{subDed}}}%
\begin{code}
ded_closeStep :: (Ord obj, Ord mor) => Ded obj mor -> SubCatClosure obj mor
ded_closeStep c = divAll_closeStep (ded_divAll c) `scComp`
                  scStep (ded_closeSubCatDiff c)

subDed s c = sub_ded (scClose (ded_closeStep c) s) c
\end{code}

Complementation is the only operation we have to check for relation algebras:

\begin{code}
ra_compl_closeSubCatDiff :: (Ord obj, Ord mor) => 
                          RA obj mor -> SubCat obj mor -> SubCatDiff obj mor
ra_compl_closeSubCatDiff c s@(SubCat objs hs) =
  let objects = toListSet $ sub_objects s
      homset a b = toListSet $ lookupDftFM hs zeroSet (a,b)
      compl = ra_compl c
      adds = do a <- objects
                b <- objects
                f <- homset a b
                let g = compl f
                if sub_isMor s a b g then []
                                     else [addToHomset a b g]
  in foldr id (SubCat zeroSet zeroFM) adds
\end{code}

For relation algebra closure,
we may skip the separate closure operators
for division allegories and Dedekind categories:

\index{subRA@{\texttt{subRA}}}%orphisms are all implied by complementation:
\begin{code}
ra_closeStep :: (Ord obj, Ord mor) => RA obj mor -> SubCatClosure obj mor
ra_closeStep c = distrAll_closeStep (ra_distrAll c) `scComp`
                 scStep (ra_compl_closeSubCatDiff c)

subRA s c = sub_ra (scClose (ra_closeStep c) s) c
\end{code}
%}}}


%% Local variables:
%% folded-file: t
%% fold-internal-margins: 0
%% eval: (fold-set-marks "%{{{ " "%}}}")
%% eval: (fold-whole-buffer)
%% end:
