
\ignore{
\begin{code}
{-# LANGUAGE 
    MultiParamTypeClasses
  , FunctionalDependencies
  , FlexibleInstances
  , TypeFamilies
  , DataKinds
  , PolyKinds
  , TypeOperators
  , ConstraintKinds
  , UndecidableInstances
  #-}

module TypeComputation.Class 
  ( module TypeComputation.Class
  , module TypeComputation.Boolean
  , module TypeComputation.TypeLits 
  ) where 

import TypeComputation.Proxy 
import TypeComputation.Boolean
import TypeComputation.TypeLits 

\end{code}
}

\section{Definitions of arithmetic classes}\label{sec:Class}
This module contains definitions of classes which used for arithmetic
with different numeric types, as well as type families used for the same purpose.
The type families used in arithmetic are open type families to allow implementations
to define type family instances.
This module also defines some basic types. 

For actual implementations, see FloatingPoint~\ref{sec:FloatingPoint}, Integer~\ref{sec:Integer},
and Natural~\ref{sec:Natural}.

\subsection{Comparator}\label{subsec:Comparator}

The \cod{Ordering} datatype is lifted to a kind to produce three types to represent the results of comparisons:
\cod{LT}, \cod{GT}, and \cod{EQ}.
A class is defined to convert the type level values back to data level values. 
\begin{code}
class Comparator (c :: Ordering) where
    toDataComp :: p c -> Ordering
instance Comparator LT where toDataComp _ = LT
instance Comparator GT where toDataComp _ = GT
instance Comparator EQ where toDataComp _ = EQ

\end{code}

\subsection{Generic number}\label{subsec:GenNum}
Generic number class. Used for various data-level manipulations, mostly for pretty-printing and converting
type level numbers to data for testing. 
\begin{code}
class Show a => GenericNumber a where 
    numToRational :: a -> Rational 
    numToFloating :: Floating b => a -> b

    numToFloating = fromRational . numToRational 

showNum :: GenericNumber a => a -> String 
showNum = show 

\end{code}
\subsection{Testing for zero}\label{subsec:ZeroTest}
Determine if a number is zero.
This type family remains open so it can have instances in other modules. 
\begin{code}
class IsZero a 

\end{code}
\cod{NonZero a} is equivalent to \cod{IsZero a == False}.
The function \cod{nonZero} is a proof that a number is non-zero 
Knowing a number is non-zero allows us to determine any two terms in a multiplication;
this is encoded in the \hask{MultNZ} class (see ~\ref{subsec:ClassMultNZ}). 
\begin{code}
class NonZero a 

nonZero :: NonZero a => a -> ()
nonZero = const ()
\end{code}
\subsection{Negation}\label{subsec:ClassNeg}
The negation class and type family.
\begin{code}
class Negate n n' | n -> n' where negate' :: n -> n' 

type family Negate' (a :: k) :: k 

\end{code}
\subsection{Absolute Value}\label{subsec:ClassAbs}
The absolute value class and type family.
\begin{code}
class Abs n n' | n -> n' where abs' :: n -> n'

type family Abs' (a :: k) :: k 

\end{code}

\subsection{Comparison and ordering}\label{subsec:ClassComp}
Includes an open type family for use with sets and lists.
\begin{code}
class Compare a b (c :: Ordering) | a b -> c where 
  compare' :: a -> b -> Proxy c
  compare' _ _ = Proxy 

type family Compare' (a :: k) (b :: k) :: Ordering 

type instance Compare' (n :: Nat) (m :: Nat) = If (NatLTEQ n m) (If (n ==? m) EQ LT) GT

compareD :: (Compare a b c, Comparator c) => a -> b -> Ordering
compareD a b = toDataComp (compare' a b)

\end{code}
Every comparison function comes in two flavours: $\odot$? which returns a boolean value,
 and $\odot$ which is a constraint, defined as
$a \odot b = a \odot$? $b ==$ True. The actual implementation is differenent in order to produce
better errors. 

The single function \cod{Compare'} is sufficient to define all of these. 
\begin{code}
infixl 4 <?, >=?, >?, <=?

type family (<=?) (a :: k) (b :: k) :: Bool where 
  a <=? a = True
  a <=? b = If (Compare' a b ==? LT) True False

type family (<?)  (a :: k) (b :: k) :: Bool where 
  a <? a = False
  a <? b = a <=? b 

type family (>=?) (a :: k) (b :: k) :: Bool where 
  a >=? a = True
  a >=? b = a >? b

type family (>?)  (a :: k) (b :: k) :: Bool where 
  a >? a = False
  a >? b = Not (a <=? b)

(>?)   :: a -> b -> Proxy (a  >? b)
(>=?)  :: a -> b -> Proxy (a >=? b)
(<?)   :: a -> b -> Proxy (a  <? b)
(<=?)  :: a -> b -> Proxy (a <=? b)

((>?), (>=?), (<?), (<=?)) = (\_ _ -> Proxy,\_ _ -> Proxy,\_ _ -> Proxy,\_ _ -> Proxy)

infixl 4 <, >=, >, <=

type (<)  a b = If (a  <?      b) Always (Less_Than             a b)
type (>=) a b = If (a >=?      b) Always (Greater_Than_Or_Equal a b)
type (<=) a b = If (a <=?      b) Always (Less_Than_Or_Equal    a b) 
type (>)  a b = If (a  >?      b) Always (Greater_Than          a b)

assertLT   :: (a <  b)  => a -> b -> ()
assertLTEQ :: (a <= b)  => a -> b -> ()
assertGT   :: (a >  b)  => a -> b -> ()
assertGTEQ :: (a >= b)  => a -> b -> ()

(assertGTEQ, assertGT, assertLT, assertLTEQ) = 
   (\_ _ -> (),\_ _ -> (),\_ _ -> (),\_ _ -> ())

\end{code}
%%\subsection{Addition}\label{subsec:ClassAdd}
\subsection{Arithmetic}\label{subsec:ClassArith}

The basic arithmetic functions, add, subtract, and multiply, have a three way functional
dependency. This asserts that any one type can be determined from the other two. This makes
writing code more intuitive.

\begin{code}
infixl 6 +
type family (+) (a :: k) (b :: k) :: k
type instance (+) (n :: Nat) (m :: Nat) = NatPlus n m

class Add a b c | a b -> c, b c -> a, a c -> b where 
  add :: a -> b -> c 

(+) :: Add a b c => a -> b -> c
(+) = add 

\end{code}
%%\subsection{Subtraction}\label{subsec:ClassSub}
\begin{code}
infixl 6 -
type family (-) (a :: k) (b :: k) :: k
type instance (-) (n :: Nat) (m :: Nat) = NatMinus n m

class Subtract a b c | a b -> c, b c -> a, a c -> b where 
  sub :: a -> b -> c

(-) :: Subtract a b c => a -> b -> c
(-) = sub

\end{code}

%%\subsection{Multiplication}\label{subsec:ClassMult}

\begin{code}
infixl 7 *
type family (*) (a :: k) (b :: k) :: k
type instance (*) (n :: Nat) (m :: Nat) = NatTimes n m

class Mult a b c | a b -> c where 
  mult :: a -> b -> c

(*) :: Mult a b c => a -> b -> c
(*) = mult

\end{code}

\subsection{Exponentiation}\label{subsec:ClassExp}

Only the built in \cod{Nat} uses this so far, since we haven't found it useful for other numbers. 

\begin{code}
infixr 8 ^
type family (^) (a :: k) (b :: k) :: k
type instance (^) (n :: Nat) (m :: Nat) = NatPow n m 

class Pow a b c | a b -> c where 
  pow :: a -> b -> c

(^) :: Pow a b c => a -> b -> c
(^) = pow

\end{code}

\subsection{Non-zero multiplication}\label{subsec:ClassMultNZ}

Multiplication of non zero multiplicands allows us to have the extra functional dependencies 
which assert that any single number may be infered from the other two.

\begin{code}
class (NonZero a, NonZero b, NonZero c) => MultNZ a b c | a b -> c, b c -> a, a c -> b where 
  multNZ :: a -> b -> c 

\end{code}

\subsection{Extremums}\label{subsec:ClassExtremums}

Min-max - it can implemented generally in terms of comparison, but for speed, we implement it individually for each numeric type.
\begin{code}
class Min a b c | a b -> c where 
  min' :: a -> b -> c 

class Max a b c | a b -> c where 
  max' :: a -> b -> c 

minMax :: (Min a b c, Max a b d) => a -> b -> (c, d) 
minMax a b = (min' a b, max' a b)

\end{code}

\subsection{Modular integer division}\label{subsec:ClassDivMod}

Unfortunatley this is too slow to be practical, or else we could reduce fractions to standard form quite easily.
\begin{code}
class DivMod a b c d | a b -> c d, b c d -> a where 
  divMod' :: a -> b -> (c, d) 

\end{code}
