\ignore{
\begin{code}
{-# LANGUAGE 
    MultiParamTypeClasses
  , FunctionalDependencies
  , UndecidableInstances
  , FlexibleInstances
  , RankNTypes
  , FlexibleContexts
  , TypeFamilies
  , GADTs
  , ScopedTypeVariables
  , Rank2Types
  , ConstraintKinds
  , TemplateHaskell
  , TypeOperators
  , QuasiQuotes
  , DefaultSignatures 
  , StandaloneDeriving 
  , PolyKinds
  , DataKinds 
  #-}

module TypeComputation.Examples 
  ( module TypeComputation.Examples 
  , module TypeComputation.Numeric.FloatingPoint
  , module TypeComputation.UnitDefinitions
  ) where

import TypeComputation.Numeric.FloatingPoint
import TypeComputation.UnitDefinitions
import TypeComputation.FFT (Discretization (..), CompositeFrame (..))
import qualified TypeComputation.FFT as FFT 

import Data.List (transpose)
import Data.Complex hiding ((:+))
import qualified Numeric.FFT as FFT
\end{code}
}

\section{Examples}\label{sec:Examples}

This module contains examples of applications of code in the other modules. 

\subsection{Frames}\label{subsec:Frames}

The examples below demonstrate an application of type level numbers and units to the Fast Fourier Transform.\\ 
The basic class defintion for a frame of reference, 
containing a function to show the name of a frame, 
and a type alias for the base unit of the frame. The type of \cod{BaseUnit} should be \cod{SIUnit}~\ref{subsec:SIUnit}.
\begin{code}

data F (s :: Symbol) = F 

class (KnownUnit (BaseUnit frame)) => Frame frame where
  type BaseUnit frame :: * -> *
  name :: frame -> String
  default name :: (KnownSymbol x, frame ~ F x) => frame -> String
  name = symbolVal 

\end{code}
One application of frames of reference is to the Fast Fourier Transform, which has an inverse operation. 
The implementation of FFT is provided by the package pure-fft. The package contains two functions useful to us;
\cod{fft}, which is the FFT and \cod{ifft}, which is the inverse operation. 
We make the assertions that inverse FFT applied to the result of FFT has the same
type as the original arguement - \cod{ ( (ifft . fft) (a :: t) ) :: t }. 
We let the user assert that two frames are dual to each other, and check that the units match.
The functional depency is asserting that the dual of the dual frame is the original frame.
This has to be argued outside the type system, with a definition of what the dual frame is.
Maybe we need to argue it in the context of discretizations, 
if there is no notion of dual coordinate frames, but there is a notion of dual vector spaces.

\subsection{Dual frames and units}\label{subsec:DualUF}

Before dual frames can be formalized, we first formalize dual units; these are units whose dimensions are reciprocals,
 or whose product is unitless.
Instances of the class which defines dual frames of reference must contain two frames whose units are dual.
\begin{code}

type AssertDualUnits a b = Assert (a :*: b ==? Unitless) (DualUnits a b)

class (Frame a, Frame b, AssertDualUnits (BaseUnit a) (BaseUnit b) 
      ) => AssertDualFrames a b | a -> b, b -> a 

\end{code}

The user must then define instances of \cod{Frame} for each frame of reference. 
They are responsible for defining \cod{BaseUnit}s which are consistent; 
they then must create an instance of \cod{AssertDualFrames} to guarantee that the frames are in fact dual to each other.
\begin{code}
instance Frame (F "LabFrame") where
  type BaseUnit (F "LabFrame") = Meter 
  
instance Frame (F "LabFrameT") where
  type BaseUnit (F "LabFrameT") = Recip Meter 
  name _ = "LabFrameDual"

instance AssertDualFrames (F "LabFrame") (F "LabFrameT") where

\end{code}

If the user defines a frame improperly and attempts to assert the duality of the improperly defined frames:
\begin{code}
instance Frame (F "BadFrame0") where
  type BaseUnit (F "BadFrame0") = Unitless 

instance Frame (F "BadFrame1") where
  type BaseUnit (F "BadFrame1") = Meter 

\end{code}
\begin{fakecode}
instance AssertDualFrames (F "BadFrame1") (F "BadFrame0") where
\end{fakecode}
they will meet this compile time error
\begin{fakecode}
    No instance for (DualUnits
                       (SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0))
                       (SIUnit ('M 0) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0)))
      arising from the superclasses of an instance declaration
    In the instance declaration for
      `AssertDualFrames (F "BadFrame1") (F "BadFrame0")'
\end{fakecode}
The source of the problem is clear from the error.

We define another sample frame.
\begin{code}

instance Frame (F "CanalFrame") where
  type BaseUnit (F "CanalFrame") = Meter 

\end{code}

\subsection{Discretization}\label{subsec:Disc}

We are now ready to define a datatype to represent a series of values belonging to one dimensional discretization. 
The discretization has several physical properties which are all present in the constraints of the datatype.
\begin{code}
data Discretization1D frame numSamples stepSize rangeU val where
    Discretization1D :: (Frame frame, IsNat numSamples, IsFloat stepSize,
                         rangeU ~ SIUnit m1 s1 kg1 a1 mol1 k1
                         ) => val -> Discretization1D frame numSamples stepSize rangeU val

deriving instance Show val => Show (Discretization1D a b c d val)

\end{code}
Instances for addition and subtraction of discretization are implemented.
\begin{code}

instance Num val => Add 
      (Discretization1D frame numSamples stepSize rangeU val) 
      (Discretization1D frame numSamples stepSize rangeU val)
      (Discretization1D frame numSamples stepSize rangeU val) where
  (Discretization1D a) `add` (Discretization1D b) = Discretization1D (a + b)

instance Num val => Subtract 
      (Discretization1D frame numSamples stepSize rangeU val) 
      (Discretization1D frame numSamples stepSize rangeU val)
      (Discretization1D frame numSamples stepSize rangeU val) where
  (Discretization1D a) `sub` (Discretization1D b) = Discretization1D (a - b)

instance Num val => Negate 
      (Discretization1D frame numSamples stepSize rangeU val) 
      (Discretization1D frame numSamples stepSize rangeU val) where 
  negate' (Discretization1D a) = Discretization1D (negate a)

\end{code}


Using examples below, we can see that addition between discretizations which do not share all of the same properties produces errors.
\begin{code}
meas1 :: Discretization1D  (F "LabFrame")
                           (NAT 4) 
                           [float|0.002|]
                           Meter 
                           [Complex Double]
meas1 = Discretization1D [0,1,2,3]

meas2 :: Discretization1D  (F "LabFrame")
                           (NAT 5)
                           [float|0.002|]
                           Second 
                           [Complex Double]
meas2 = Discretization1D [0.4,0.8,1.2,1.6]

canalSample1 :: Discretization1D (F "CanalFrame") 
                                 (NAT 12)
                                 [float|0.01|]
                                 Meter  
                                 [Double]
canalSample1 = Discretization1D [1, 1.2, 1.3, 1.12, 1.23, 1.12, 1.15, 1.25, 1.18, 1.20, 1.24, 1.28]

canalSample2 :: Discretization1D (F "CanalFrame") 
                                 (NAT 12)
                                 [float|0.02|]
                                 Meter 
                                 [Double]
canalSample2 = Discretization1D [1, 1.21, 1.2, 1.42, 1.3, 1.32, 1.12, 1.25, 1.23, 1.20, 1.12, 1.28]
\end{code}

\begin{fakecode}
  > add meas1 canalSample1
                
  > add canalSample2 canalSample1
    
  > meas1 U.+ meas2
\end{fakecode}

All of the above return an error of the form:
\begin{fakecode}
    No instance for (Add (Discretization1D ...)  (Discretization1D ...) ...)
\end{fakecode}

\subsection{Fourier Transform}\label{subsec:FTFixed}


We can now use the above defintions to define a class \cod{FT} which will be used to describe
the relationship between a set of discrete data and the Fourier Transform of that data. 
The functional dependencies indicate that the intial data uniquely determines the Fourier
transform, and the result of the Fourier transform uniquely determines the initial data.
\begin{code}
class FT a b | a -> b, b -> a where
  ft :: a -> b 
  invFt :: b -> a
\end{code}

The instance of \cod{FT} for one dimensional discretizations constrains the frames of reference of the original data 
and the result as being dual frames. 
One requirement of Fourier transform is the product of the step sizes of input and output data 
and the number of samples must be one. The \cod{MultNZ} class is used to apply this constraint. 
\begin{code}

instance (
    AssertDualFrames frame1 frame2, Frame frame1, Frame frame2, 
    IsFloat stepSize2, IsFloat stepSize1,

    ToFloat numSamp ~ numSampF,

    MultNZ stepSize1 numSampF t0, MultNZ t0 stepSize2 t1, t1 ~ FLOAT Pos 1 (E 0)

    ) => 
    FT (Discretization1D frame1 numSamp stepSize1 rangeU [Complex Double]) 
       (Discretization1D frame2 numSamp stepSize2 rangeU [Complex Double]) 
        where
            ft (Discretization1D x) = (Discretization1D $ FFT.fft x)
            invFt (Discretization1D x) = (Discretization1D $ FFT.ifft x)

\end{code}

We can now apply Fourier transform to a data set with type safety. The type system infers the type of the result; the result itself is not of particular interest, however.
\begin{fakecode}

  >:t meas1
  meas1
    :: Discretization1D
         (F "LabFrame")
         (NAT 4)
         (FLOAT 'Pos 2 ('E_ 3))
         Meter
         [Complex Double]
  
  >:t ft meas1
  ft meas1
    :: Discretization1D
         (F "LabFrameT")
         (NAT 4)
         (FLOAT 'Pos 125 ('E 0))
         (SIUnit ('M_ 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0))
         [Complex Double]
  
  >:t invFt (ft meas1)
  invFt (ft meas1)
    :: Discretization1D
         (F "LabFrame")
         (NAT 4)
         (FLOAT 'Pos 2 ('E_ 3))
         (SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0))
         [Complex Double]
  

\end{fakecode}
The step sizes of \cod{meas1} and \cod{ft meas1} are $1/500 = 0.002$ and $500/4 = 125$, respectively. 
We indeed have that $0.002 \times 125 \times 4 = 1$, 
as the constraint on the Fourier transform requires. Moreover, we also witness the identity \cod{invFt . ft == id}

Similarly, we can define frames for a two dimensional Fourier Transform:
\begin{code}

instance Frame (F "LabFrame2D") where
  type BaseUnit (F "LabFrame2D") = Meter 
  name _ = "lab2"
  
instance Frame (F "LabFrame2DT") where
  type BaseUnit (F "LabFrame2DT") = Recip Meter 
  name _ = "lab2t"
  
instance AssertDualFrames (F "LabFrame2D") (F "LabFrame2DT")


\end{code}
We can define a similair rectangular discretization in two dimensions. 
In this case, we need to indicate the number of samples in both perpendicular basis directions, as well as the step size. 
\begin{code}
data Discretization2D a b c d e f g where
    Discretization2D :: (Frame frame, IsNat numSamplesX, IsNat numSamplesY, 
                         IsFloat stepSizeX, IsFloat stepSizeY,
                         rangeU ~ SIUnit m1 s1 kg1 a1 mol1 k1
                         ) => val -> Discretization2D frame numSamplesX numSamplesY stepSizeX stepSizeY rangeU val

deriving instance Show val => Show (Discretization2D a b c d e f val)

\end{code}
Instances for addition and subtraction of discretization are implemented.
\begin{code}

instance Num val => Add 
      (Discretization2D frame numSamplesX numSamplesY stepSizeX stepSizeY rangeU val) 
      (Discretization2D frame numSamplesX numSamplesY stepSizeX stepSizeY rangeU val)
      (Discretization2D frame numSamplesX numSamplesY stepSizeX stepSizeY rangeU val) where
  (Discretization2D a) `add` (Discretization2D b) = Discretization2D (a + b)

instance Num val => Subtract 
      (Discretization2D frame numSamplesX numSamplesY stepSizeX stepSizeY rangeU val) 
      (Discretization2D frame numSamplesX numSamplesY stepSizeX stepSizeY rangeU val)
      (Discretization2D frame numSamplesX numSamplesY stepSizeX stepSizeY rangeU val) where
  (Discretization2D a) `sub` (Discretization2D b) = Discretization2D (a - b)

instance Num val => Negate 
      (Discretization2D frame numSamplesX numSamplesY stepSizeX stepSizeY rangeU val) 
      (Discretization2D frame numSamplesX numSamplesY stepSizeX stepSizeY rangeU val) where
  negate' (Discretization2D a) = Discretization2D (negate a)


labSample2D :: Discretization2D (F "LabFrame2D")  
                                (NAT 8)        
                                (NAT 4)   
                                [float|5e-2|]           
                                [float|1e-2|]
                                Meter                     
                                [[Complex Double]] 
labSample2D = Discretization2D
                    [[-0.2 ,-0.26 ,-0.32 ,-0.38 ,-0.44 ,-0.5 ,-0.56 ,-0.63]
                    ,[0.2  ,0.26  ,0.32  ,0.38  ,0.44  ,0.5  ,0.56  ,0.76 ]
                    ,[0.6  ,0.78  ,0.96  ,1.14  ,1.32  ,1.5  ,1.68  ,1.98 ]
                    ,[1.0  ,1.3   ,1.6   ,1.90  ,2.2   ,2.5  ,2.8   ,2.94 ]]

labSample2DX :: Discretization1D (F "LabFrame2D")  
                                (NAT 8)    
                                [float|5e-2|] 
                                Meter                     
                                [Complex Double] 
labSample2DX = Discretization1D [-0.2 ,-0.26 ,-0.32 ,-0.38 ,-0.44 ,-0.5 ,-0.56 ,-0.63]



labSample2DY :: Discretization1D (F "LabFrame2D")  
                                (NAT 4)   
                                [float|1e-2|]
                                Meter                     
                                [Complex Double]
labSample2DY = Discretization1D []

instance (
    AssertDualFrames frame1 frame2,
    Frame frame1, Frame frame2, 

    ToFloat numSampX ~ numSampXF, ToFloat numSampY ~ numSampYF,

    IsFloat stepSizeX2, IsFloat stepSizeX1,
    IsFloat stepSizeY2, IsFloat stepSizeY1,

    MultNZ stepSizeX1 numSampXF t0, MultNZ t0 stepSizeX2 t1, t1 ~ FLOAT Pos 1 (E 0),
    MultNZ stepSizeY1 numSampYF q0, MultNZ q0 stepSizeY2 q1, q1 ~ FLOAT Pos 1 (E 0)

    ) => 
    FT (Discretization2D frame1 numSampX numSampY 
                         stepSizeX1 stepSizeY1 rangeU [[Complex Double]]) 
       (Discretization2D frame2 numSampX numSampY 
                         stepSizeX2 stepSizeY2 rangeU [[Complex Double]]) 
        where
          ft    (Discretization2D x) = (Discretization2D . transpose . (map FFT.fft ) . transpose . (map FFT.fft )) x
          invFt (Discretization2D x) = (Discretization2D . transpose . (map FFT.ifft) . transpose . (map FFT.ifft)) x

\end{code}
\begin{fakecode}

  >:t labSample2D
  labSample2D
    :: Discretization2D
         (F "LabFrame2D")
         (NAT 8)
         (NAT 4)
         (FLOAT 'Pos 5 ('E_ 2))
         (FLOAT 'Pos 1 ('E_ 2))
         Meter
         [[Complex Double]]
  
  >:t ft labSample2D
  ft labSample2D
    :: Discretization2D
         (F "LabFrame2DT")
         (NAT 8)
         (NAT 4)
         (FLOAT 'Pos 25 ('E_ 1))
         (FLOAT 'Pos 25 ('E 0))
         (SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0))
         [[Complex Double]]
  
  >:t invFt (ft labSample2D)
  invFt (ft labSample2D)
    :: Discretization2D
         (F "LabFrame2D")
         (NAT 8)
         (NAT 4)
         (FLOAT 'Pos 5 ('E_ 2))
         (FLOAT 'Pos 1 ('E_ 2))
         (SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0))
         [[Complex Double]]

\end{fakecode}

A simple function for calculation potential energy in the Earth's gravity given change in height and mass.
\begin{code}

potentialEnergy :: Kilogram Double -> Meter Double -> Joule Double
potentialEnergy mass height = mass `mult` g `mult` height
    where g = SIUnit 9.81 :: (Meter :/: Second :/: Second) Double


\end{code}
The following works correctly: 
\begin{fakecode}
potentialEnergy (kilogram 3.0) (meter 25.0)

735.75 m^2 s^-2 kg :: Joule Double
\end{fakecode}
While an incorrectly typed arguement will yield: 
\begin{fakecode}
potentialEnergy (second 1.0) (unitless 1.0)

    Couldn't match type `0' with `1'
    Expected type: Kilogram Double
      Actual type: Second Double

    Couldn't match type `1' with `0'
    Expected type: Meter Double
      Actual type: Unitless Double

\end{fakecode}
Examples of the multidimensional FT found in the FFT module. First some example frames, then some sample computations
on them. 
\begin{code}

instance FFT.Frame "LabFrameX" where
  type BaseUnit "LabFrameX" = Meter 

instance FFT.Frame "LabFrameY" where
  type BaseUnit "LabFrameY" = Meter 
  
instance FFT.Frame "LabFrameXT" where
  type BaseUnit "LabFrameXT" = Recip Meter
  
instance FFT.Frame "LabFrameYT" where
  type BaseUnit "LabFrameYT" = Recip Meter

instance FFT.AssertDualFrames "LabFrameX" "LabFrameXT"
instance FFT.AssertDualFrames "LabFrameY" "LabFrameYT"

labFrame :: 
  FFT.Discretization2D
      '("LabFrameX", NAT 8, [float|5e-2|], Meter)
      '("LabFrameY", NAT 4, [float|1e-2|], Meter)
      []
      (Complex Double)

labFrame = FFT.discretization2D

                    [[-0.2 ,-0.26 ,-0.32 ,-0.38 ,-0.44 ,-0.5 ,-0.56 ,-0.63]
                    ,[0.2  ,0.26  ,0.32  ,0.38  ,0.44  ,0.5  ,0.56  ,0.76 ]
                    ,[0.6  ,0.78  ,0.96  ,1.14  ,1.32  ,1.5  ,1.68  ,1.98 ]
                    ,[1.0  ,1.3   ,1.6   ,1.90  ,2.2   ,2.5  ,2.8   ,2.94 ]]


labFrame2 :: 
  FFT.Discretization1D
      '("LabFrameX", NAT 8, [float|5e-2|], Meter)
      []
      (Complex Double)

labFrame2 = FFT.discretization1D [-0.2 ,-0.26 ,-0.32 ,-0.38 ,-0.44 ,-0.5 ,-0.56 ,-0.63]

\end{code}
\begin{fakecode}

  >:t forwardFT1  (F :: F "LabFrameY") labFrame
  forwardFT1  (F :: F "LabFrameY") labFrame
    :: Discretization
         '['("LabFrameX",
             NAT 8,
             FLOAT 'Pos 5 ('E_ 2),
             SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0)),
           '("LabFrameYT",
             NAT 4,
             FLOAT 'Pos 25 ('E 0),
             SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0))]
         []
         (Complex Double)
\end{fakecode}

\begin{fakecode}
  >:t forwardFT1  (F :: F "LabFrameX") labFrame
  forwardFT1  (F :: F "LabFrameX") labFrame
    :: Discretization
         '['("LabFrameXT",
             NAT 8,
             FLOAT 'Pos 25 ('E_ 1),
             SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0)),
           '("LabFrameY",
             NAT 4,
             FLOAT 'Pos 1 ('E_ 2),
             SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0))]
         []
         (Complex Double)
  >:t forwardFT ((F :: F "LabFrameX") :& (F :: F "LabFrameY") :& Nil) labFrame
\end{fakecode}

\begin{fakecode}
  forwardFT (Cons (F :: F "LabFrameX") $ Cons (F :: F "LabFrameY") Nil) labFrame
    :: Discretization
         '['("LabFrameXT",
             NAT 8,
             FLOAT 'Pos 25 ('E_ 1),
             SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0)),
           '("LabFrameYT",
             NAT 4,
             FLOAT 'Pos 25 ('E 0),
             SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0))]
         []
         (Complex Double)
\end{fakecode}

\begin{fakecode}
  >:t inverseFT1 (F :: F "LabFrameXT") (forwardFT1  (F :: F "LabFrameX") labFrame)
  inverseFT1 (F :: F "LabFrameXT") (forwardFT1  (F :: F "LabFrameX") labFrame)
    :: Discretization
         '['("LabFrameX",
             NAT 8,
             FLOAT 'Pos 5 ('E_ 2),
             SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0)),
           '("LabFrameY",
             NAT 4,
             FLOAT 'Pos 1 ('E_ 2),
             SIUnit ('M 1) ('S 0) ('Kg 0) ('A 0) ('Mol 0) ('K 0))]
         []
         (Complex Double)
\end{fakecode}
