\ignore{
\begin{code}
{-# LANGUAGE 
    MultiParamTypeClasses
  , FunctionalDependencies
  , UndecidableInstances
  , FlexibleInstances
  , RankNTypes
  , FlexibleContexts
  , TypeFamilies
  , GADTs
  , ScopedTypeVariables
  , Rank2Types
  , ConstraintKinds
  , TemplateHaskell
  , TypeOperators
  , QuasiQuotes
  , DefaultSignatures 
  , StandaloneDeriving 
  , PolyKinds
  , DataKinds 
  , OverlappingInstances
  , AllowAmbiguousTypes
  #-}

module TypeComputation.FFT 
  ( module TypeComputation.FFT
  , module TypeComputation.FFT.Frame 
  , module TypeComputation.Numeric.FloatingPoint
  , module TypeComputation.TH.TypeGen
  , module TypeComputation.UnitDefinitions
  ) where

import TypeComputation.FFT.Frame 
import TypeComputation.Numeric.FloatingPoint
import TypeComputation.TH.TypeGen
import TypeComputation.UnitDefinitions
import TypeComputation.List

import Data.Complex 
import Numeric.FFT (fft, ifft)
import Data.List
\end{code}
}

\section{Discrete Fourier Transform}\label{sec:FFT}

This module implements the typed FFT in arbitrary dimension, as opposed to the version in fixed 
dimension. ~\ref{subsec:FTFixed}

\input{tex/Frame}

\subsection{Choice of direction}\label{subsec:FFTDir}

We can generalize the code for computing the FT; it is either forward or inverse. 
\begin{code}

data Dir = Forward | Inverse 

data Direction (d :: Dir) where 
  ForwardDir :: Direction Forward
  InverseDir :: Direction Inverse 

class DualFrameOf dir a b | dir a -> b 
instance AssertDualFrames a b => DualFrameOf Forward a b
instance AssertDualFrames b a => DualFrameOf Inverse a b 

dualFrameOf :: DualFrameOf d a b => Proxy (d :: Dir) -> F a -> F b
dualFrameOf _ _ = F

ftFunc :: Direction dir -> [Complex Double] -> [Complex Double]
ftFunc ForwardDir = fft
ftFunc InverseDir = ifft

\end{code}

\subsection{Computing the correct FFT type}\label{subsec:FFTSearch}

In order to perform FFT on an arbitrary dimension of an $n$-dimensional matrix, we have to count how dimensions
'inside` the desired dimension is.

In order to do this counting, we store a \cod{CompositeFrame} which will later be used to determine
the correct 'depth`. 

Two classes are required to implement the recursive search of the required frame.
\begin{code}

class DirectionFT1' dir db frame dim    frame' dim' 
                  | dir db frame dim -> frame' dim' where 
  
  directionFT1' :: Direction dir -> F db -> CompositeFrame frame -> CompositeFrame dim -> 
                 (CompositeFrame frame', CompositeFrame dim')

class DirectionFT1'' bool dir db frame dim    frame' dim' 
                   | bool dir db frame dim -> frame' dim' where 

  directionFT1'' :: Proxy bool -> Direction dir -> F db -> CompositeFrame frame 
                 -> CompositeFrame dim -> 
                  (CompositeFrame frame', CompositeFrame dim')


instance (bool ~ (fr ==? fr'), DirectionFT1'' bool dir fr' ('(fr, n, s, ru) ': xs) dim (y ': ys) dim'
         ) => DirectionFT1' dir fr' ('(fr, n, s, ru) ': xs) dim (y ': ys) dim' where 
  directionFT1' = directionFT1'' (Proxy :: Proxy bool)


\end{code}
The recursive case for when the target frame and head of the composite frame of the 
\cod{Discretization} are the same. In this case, return the current counter and compute the new
type of the frame.
\begin{code}

instance (IsNat numSamples, DualFrameOf dir frame frameD,
          stepSizeD ~ FLOAT s0 n0 e0,

          ToFloat numSamples ~ numSamplesF,
          stepSizeD ~ RecipFloat (numSamplesF * stepSize)

         ) => DirectionFT1'' True dir fr ( '(frame , numSamples, stepSize , rangeU ) ': xs) dim 
                                         ( '(frameD, numSamples, stepSizeD, rangeU ) ': xs) dim  
  where directionFT1'' _ _ _ (_ :> xs) n = ((F, NAT, FLOAT, Proxy) :> xs, n) 


\end{code}
The recursive case for when the target frame and head of the composite frame of the 
\cod{Discretization} are different. In this case, recurse on the rest of the composite frame
and increment the counter by one. 
\begin{code}

instance (DirectionFT1' dir fr xs dim ys dim', (1 + Length xs) == (1 + Length ys),
          d ~ '(frame, numSamples, stepSize, rangeU )
         ) => DirectionFT1'' False dir fr ( '(frame, numSamples, stepSize, rangeU ) ': xs) dim
                                        ( '(frame, numSamples, stepSize, rangeU ) ': ys) (d ': dim')  
  where directionFT1'' _ dir f (x :> xs) dim = 
          let (ys, dim') = directionFT1' dir f xs dim in (x :> ys, x :> dim')


\end{code}

\subsection{Applying the FFT}\label{subsec:FFTApp}

In order to actually perform the FFT, we need to know the depth of the target
dimension and the overall dimension of the matrix. Then there is a pattern 
to compute the FFT from these values: 

\begin{fakecode}
  Dim 1 : x :  map^0 transpose . fft^1 . map^0 transpose
  
  Dim 2 : x :  map^0 transpose . fft^2 . map^0 transpose
          y :  map^1 transpose . fft^2 . map^1 transpose

  Dim 3 : x :  map^0 transpose . fft^3 . map^0 transpose
          y :  map^1 transpose . fft^3 . map^1 transpose
          z :  map^2 transpose . fft^3 . map^2 transpose
\end{fakecode}
Where \cod{f^0} is \cod{id}, \cod{f^1} is \cod{f}, \cod{f^2} is \cod{f . f}, etc.
This function exponentiation is provided by \cod{dimension} and \cod{dimensionPred}.

In order to actually compute the FFT relative to a frame, that frame must be in the 
composite frame. We check this precondition.
\begin{code}

type AssertAtomicSubframe x xs = Assert (IsAtomicSubframe x xs) (AtomicSubframe x xs)

type family IsAtomicSubframe (x :: Symbol) (xs :: [ (Symbol, k0, k1, k2) ]) :: Bool where 
  IsAtomicSubframe x '[]                       = False
  IsAtomicSubframe x ( '(x, k0, k1, k2) ': xs) = True 
  IsAtomicSubframe x ( '(y, k0, k1, k2) ': xs) = IsAtomicSubframe x xs

atomicSubFrame :: AssertAtomicSubframe db xs => F db -> CompositeFrame xs -> ()
atomicSubFrame _ _ = ()

\end{code}
The top level FFT class encodes the base case of a one dimensional discretization, and the 
general case. 
\begin{code}

class (AssertAtomicSubframe db frame
      ) => DirectionFT1 dir db frame    frame' 
                    | dir db frame -> frame' where 
  directionFT1 :: Direction dir -> F db 
               -> Discretization frame [] (Complex Double) 
               -> Discretization frame' [] (Complex Double)

\end{code}
The base case. The target frame and the single frame inside the composite frame must match. 
\begin{code}

instance (DualFrameOf dir frame frameD, 
          stepSizeD ~ FLOAT s0 n0 e0, numSamples ~ NAT n1, rangeU ~ SIUnit m1 s1 kg1 amp1 mol1 k1,

          ToFloat numSamples ~ numSamplesF,
          stepSizeD ~ RecipFloat (numSamplesF * stepSize)

  ) => DirectionFT1 dir frame '[ '(frame , numSamples, stepSize , rangeU)  ] 
                              '[ '(frameD, numSamples, stepSizeD, rangeU) ] where 

  directionFT1 _ _ (Discretization _ val) = Discretization ((F, NAT, FLOAT, Proxy) :> F0) (fft val)

\end{code}
The general case. This computes the depth of the target frame, then creates the 
transposition and FFT which apply to the proper dimensions. 

At this point we compute the actual function to be applied - \cod{fft} or \cod{ifft}
from the direction parameter. 
\begin{code}

instance (frame ~ (x ': xs), frame' ~ (y ': ys), AssertAtomicSubframe db (x ': xs),
          DirectionFT1' dir db frame '[] frame' dim',

          DimensionPred dim' [] [[a]] ~ [Dimension xs [] (Complex Double)],
          Dimension ys [] (Complex Double) ~ Dimension xs [] (Complex Double),
          Dimension xs [] [Complex Double] ~ [Dimension xs [] (Complex Double)]
  ) => DirectionFT1 dir db (x ': xs) 
                           (y ': ys) where 

  directionFT1 dir db (Discretization frame val) = Discretization frame' (tp . ft . tp $ val)
    where (frame', (dim :: CompositeFrame dim')) = directionFT1' dir db frame F0 
                          
          tp = dimensionPred (Proxy :: Proxy []) dim (transpose :: [[a]] -> [[a]])
          ft = dimensionPred (Proxy :: Proxy []) frame (ftFunc dir) 

   
\end{code}
Once the base case exists, we can write the more general case of taking the FFT with respect
to multiple dimensions. 
\begin{code}

class DirectionFT dir xs frame frame' | dir xs frame -> frame' where
  directionFT :: Direction dir 
              -> List xs 
              -> Discretization frame [] (Complex Double) 
              -> Discretization frame' [] (Complex Double) 

instance DirectionFT dir '[] frame frame where 
  directionFT _ _ = id 

instance (DirectionFT1 dir x frame frame', DirectionFT dir xs frame' frame''
         ) => DirectionFT dir (F x ': xs) frame frame'' where 
  directionFT dir (x :& xs) d = directionFT dir xs (directionFT1 dir x d)

   
\end{code}
Once there is a general form of the FT, getting the inverse or forward FT just requires
passing the correct argument to the general function.
\begin{code}

forwardFT1 :: DirectionFT1 Forward db frame frame' =>
  F db -> Discretization frame  [] (Complex Double) 
       -> Discretization frame' [] (Complex Double)
forwardFT1 = directionFT1 ForwardDir

inverseFT1 :: DirectionFT1 Inverse db frame frame' =>
  F db -> Discretization frame  [] (Complex Double) 
       -> Discretization frame' [] (Complex Double)
inverseFT1 = directionFT1 InverseDir

forwardFT :: DirectionFT Forward db frame frame' =>
  List db -> Discretization frame  [] (Complex Double) 
          -> Discretization frame' [] (Complex Double)
forwardFT = directionFT ForwardDir

inverseFT :: DirectionFT Inverse db frame frame' =>
  List db -> Discretization frame  [] (Complex Double)
          -> Discretization frame' [] (Complex Double)
inverseFT = directionFT InverseDir

\end{code}
