
\ignore{
\begin{code}
{-# LANGUAGE 
      RankNTypes 
    , ScopedTypeVariables 
    , EmptyDataDecls 
    , MultiParamTypeClasses
    , FunctionalDependencies 
    , FlexibleInstances 
    , UndecidableInstances 
    , OverlappingInstances 
    , TypeFamilies  
    , ImpredicativeTypes 
    , FlexibleContexts 
    , DeriveDataTypeable
    , IncoherentInstances
    , ConstraintKinds
    , TypeOperators
    , GADTs
    , TemplateHaskell
    , QuasiQuotes
    , DataKinds
    , PolyKinds
    #-}

module TypeComputation.Numeric.Natural 
  ( module TypeComputation.Numeric.Natural 
  , module TypeComputation.Boolean
  , module TypeComputation.Class
  ) where 

import TypeComputation.Class hiding ((*), (+), (-), (^))
import TypeComputation.Boolean
import TypeComputation.List

import Data.Ratio ((%))

\end{code}
}

\subsection{Naturals}\label{sec:Natural}

This module implements arithmetic for naturals. 

Definition of a natural; since GHC provides type level naturals,
we wrap them in a constructor to allow them to be passed to functions,
for printing, for consistency with \hask{FLOAT} and \hask{INT}, and to prevent spurious overlapping instances.
\begin{code}

data NAT (n :: Nat) = NAT 
instance KnownNat n => Show (NAT n) where show = show . natVal 

\end{code}
Produce data-level values for naturals. 
\begin{code}
instance KnownNat n => GenericNumber (NAT n) where
    numToRational a = (natVal a) % 1
    numToFloating = fromIntegral . natVal

\end{code}
Determine if a type is a \hask{NAT}.
\begin{code}

type family IsNat n where 
  IsNat (NAT n) = Always

isNat :: IsNat n => n -> ()
isNat _ = ()

\end{code}
Is a natural zero?
\begin{code}

instance (If (n ==? 0) Always (Not_Equal_To_Zero (NAT n))) => IsZero  (NAT n) 
instance (If (n /=? 0) Always (Equal_To_Zero     (NAT n))) => NonZero (NAT n)

\end{code}
Is a natural odd or even?
\begin{code}

type family IsOdd a where 
  IsOdd 0 = False
  IsOdd 1 = True 
  IsOdd n = IsOdd (n - 2)

type IsEven n = Not (IsOdd n)

\end{code}
Absoulte value on naturals is the identity. 
\begin{code}
instance Abs (NAT n) (NAT n) where abs' = id 

type instance Abs' (NAT n) = NAT n 

\end{code}
Comparison.
\begin{code}

instance (CompareNat (NAT a) (NAT b) ~ c) => Compare (NAT a) (NAT b) c

type instance Compare' (NAT a) (NAT b) = CompareNat (NAT a) (NAT b)

type family CompareNat a b where 
  CompareNat (NAT n) (NAT n) = EQ
  CompareNat (NAT a) (NAT b) = If (a <=? b) LT GT 

\end{code}
Addition.
\begin{code}

type instance NAT a + NAT b = NAT (a + b)

instance (a + b) ~ c => Add (NAT a) (NAT b) (NAT c) where add _ _ = NAT

\end{code}
Subtraction.
\begin{code}

type instance NAT a - NAT b = NAT (a - b)

instance (a - b) ~ c => Subtract (NAT a) (NAT b) (NAT c) where sub _ _ = NAT 

\end{code}
Multiplication.
\begin{code}

type instance NAT a * NAT b = NAT (a * b)

instance (a * b) ~ c => Mult (NAT a) (NAT b) (NAT c) where mult _ _ = NAT 

\end{code}
Multiplication of non-zero naturals.
\begin{code}
instance (NonZero (NAT a), NonZero (NAT b), NonZero (NAT c), (a * b) ~ c
         ) => MultNZ (NAT a) (NAT b) (NAT c) where multNZ _ _ = NAT 

\end{code}
Minimum and maximum.
\begin{code}
instance (If (a <=? b) a b ~ c) => Min (NAT a) (NAT b) (NAT c) where 
  min' _ _ = NAT 

instance (If (b <=? a) a b ~ c) => Max (NAT a) (NAT b) (NAT c) where 
  max' _ _ = NAT 

type family MinMaxNat a b where 
  MinMaxNat (NAT a) (NAT b) = If (a<=?b) (NAT a, NAT b) (NAT b, NAT a)
type MinNat a b = Fst (MinMaxNat a b)
type MaxNat a b = Snd (MinMaxNat a b)

\end{code}
Division with remainder.
\begin{code}

instance (DivModNat a b ~ (NAT c, NAT d), ((c * b) + d) ~ a
         ) => DivMod (NAT a) (NAT b) (NAT c) (NAT d) where divMod' = divModNat 

type family DivModNat a b where 
  DivModNat a 1 = (NAT a, NAT 0)
  DivModNat 1 a = (NAT 0, NAT 1)
  DivModNat a b = DivModNat' a b 0 0 

type family DivModNat' dividend divisor quot sum where 
  DivModNat' dividend divisor quot sum = 
    DivModNat'' ((sum + divisor) <=? dividend) dividend divisor quot sum

type family DivModNat'' b dividend divisor quot sum where 
  DivModNat'' True  dividend divisor quot sum = 
    DivModNat' dividend divisor (quot + 1) (sum + divisor)
  DivModNat'' False dividend divisor quot sum = 
    (NAT quot, NAT (dividend - sum))

divModNat :: (((c * b) + d) ~ a, DivModNat a b ~ (NAT c, NAT d)) 
          => NAT a -> NAT b -> (NAT c, NAT d)
divModNat _ _ = (NAT, NAT)

\end{code}
Compiler bug:

\begin{fakecode}
  >divModNat (NAT :: NAT 30) _ :: (NAT 7, NAT 2)
  <interactive>:52:27:
      Found hole `_' with type: NAT 4

  >divMod' (NAT :: NAT 30) _ :: (NAT 7, NAT 2)
  <interactive>:53:25:
      Found hole `_' with type: b0
\end{fakecode}

The definitions are identical - but the class instance cannot infer the type of the 2nd arguement,
even though the functional dependancy demands it, and it is clearly possible.
