\ignore{
\begin{code}
{-# LANGUAGE TemplateHaskell, FlexibleInstances, UndecidableInstances, QuasiQuotes, RankNTypes, DataKinds #-}
{-# OPTIONS_GHC -fno-warn-missing-fields #-}
module TypeComputation.TH.Random where 

import TypeComputation.Class hiding ((+), (-), (*), (^))
import TypeComputation.Numeric.Natural
import TypeComputation.Numeric.Integer
import TypeComputation.Numeric.FloatingPoint

import System.Random
import Language.Haskell.TH
import Numeric 

\end{code}
}

\subsection{Random number generation}\label{sec:Random}

This module generates random type level numbers as Template Haskell splices.

Test cases perform some operation on the type level numbers, then perform the same operation on the 
data level and compare the results. 
\begin{code}

mkTestCase :: ExpQ -> ExpQ -> ExpQ -> ExpQ -> ExpQ -> ExpQ -> ExpQ
mkTestCase mf mg f g mka mkb = fmap return mka >>= \a -> fmap return mkb >>= \b ->  
  [| (show $a, show $b, $mf ($f $a $b), ($g ($mg $a) ($mg $b))) |]

addTest, subTest, multTest, multNZTest, compareTest :: ExpQ -> ExpQ -> ExpQ

\end{code}
Some specific test case functions for addition, subtraction, multiplication, and comparison.
\begin{code}

addTest     = mkTestCase [|numToRational|] [|numToRational|] [|add|]      [|(+)|]
subTest     = mkTestCase [|numToRational|] [|numToRational|] [|sub|]      [|(-)|]
multTest    = mkTestCase [|numToRational|] [|numToRational|] [|mult|]     [|(*)|]
multNZTest  = mkTestCase [|numToRational|] [|numToRational|] [|multNZ|]   [|(*)|]
compareTest = mkTestCase [|id|]            [|numToRational|] [|compareD|] [|compare|]

testRandomNats :: (ExpQ -> ExpQ -> ExpQ) -> Int -> Integer -> Integer -> [Integer] -> ExpQ
testRandomNats = randomTests randomNat

testRandomInts :: (ExpQ -> ExpQ -> ExpQ) -> Int -> Integer -> Integer -> [Integer] -> ExpQ
testRandomInts = randomTests randomInt

testRandomFloats :: (ExpQ -> ExpQ -> ExpQ) -> Int -> Float -> Float -> [Float] -> ExpQ
testRandomFloats = randomTests randomFloat 
 
randomTests :: (num -> num -> [num] -> ExpQ) -> (ExpQ -> ExpQ -> ExpQ) -> Int -> num -> num -> [num] -> ExpQ
randomTests rnd test m a b ex = let xs = replicate (ceiling $ sqrt $ fromIntegral m) (rnd a b ex) in
                                 listE $ take m [ test q r | q <- xs, r <- xs]


\end{code}
Generating random numbers is straighforward. For floating point numbers, a precision is also specified
which limits the domain. 
\begin{code}

sigDigs :: Int -> ([Int], Int) -> ([Int], Int)
sigDigs n (xs, e) = (take n xs, e)

randomNat :: Integer -> Integer -> [Integer] -> Q Exp
randomNat a b xs = do 
  n <- rand a b xs
  [| NAT |] `sigE` (appT [t| NAT |] $ return (LitT (NumTyLit n)))

randomInt :: Integer -> Integer -> [Integer] -> Q Exp
randomInt a b xs = do 
  n <- rand a b xs
  let s  = if n < 0 then [t|Neg|] else [t|Pos|]
      n' = return $ LitT $ NumTyLit $ abs n
  sigE [|INT|] $ foldl1 appT [ [t|INT|], s, n' ]

randomFloat :: Float -> Float -> [Float] -> Q Exp
randomFloat a b xs = do
  (digs, m) <- fmap (sigDigs 5 . floatToDigits 10) (rand a b xs)
  let (n, e) = normalize (fromIntegral $ sum $ zipWith (\x y -> x * 10 ^ y) (reverse digs) [0..], fromIntegral $ m - length digs)
      s  = if n < 0 then [t|Neg|] else [t|Pos|]
      e' =  appT (if e < 0 then [t|E_|] else [t|E|]) (mkN e)
      mkN = return . LitT . NumTyLit . abs
  sigE [|FLOAT|] $ foldl1 appT [ [t|FLOAT|], s, mkN n, e' ]
  
rand :: (Random b, Eq b) => b -> b -> [b] -> Q b
rand x y z = runIO (rand' x y z) where 
  rand' a b xs = randomRIO (a,b) >>= \n -> if n `elem` xs then rand' a b xs else return n 

normalize :: (Integer, Integer) -> (Integer, Integer)
normalize (n,e) = case n `divMod` 10 of 
                    (n', 0) -> normalize (n', e+1)
                    _       -> (n,e)


\end{code}
