\ignore{
\begin{code}
{-# LANGUAGE TemplateHaskell, FlexibleInstances, UndecidableInstances, QuasiQuotes, RankNTypes, TupleSections #-}
{-# OPTIONS_GHC -fno-warn-missing-fields #-}

module TypeComputation.TH.TypeGen where

import Language.Haskell.TH
import Language.Haskell.TH.Quote

import Text.ParserCombinators.ReadP
import Control.Applicative hiding ((<|>))
import Data.List

\end{code}
}

\subsection{\hask{FLOAT} type Quasiquoter}\label{sec:TypeGen}

We define a quasi-quoter for generating float types, as writing them by hand can be tedious and 
it is clearer to see \hask{[float|3.1416|]} than \hask{FLOAT Pos 31416 (E_ 4)}. 
We also write our own parser for floats to prevent loss of precision,
since all existing parsers return \cod{Float} and we would like generated
numbers to appear exactly as written.
\begin{code}
mkFloatT :: String -> Integer -> Integer -> Type 
mkFloatT s int expnt = foldl1 AppT [ConT (mkName "FLOAT"), ConT $ mkName s, LitT $ NumTyLit int, expCons] where 
  expCons =  if expnt < 0 then ConT (mkName "E_" ) `AppT` (LitT $ NumTyLit $ abs expnt)
                          else ConT (mkName "E"  ) `AppT` (LitT $ NumTyLit $     expnt)

digitsToNum :: Num a => [a] -> a
digitsToNum = sum . zipWith (\e n -> n * 10 ^ e) [0..] . reverse 

floatP :: ReadP Type 
floatP = do 
  (s, n0) <- integerP
  n1      <- option [] (char '.' *> natP)
  ex0     <- mbExpn
  let n1' = reverse $ dropWhile (==0) $ reverse n1
      ex1 = genericLength n1'
      num = digitsToNum $ n0' ++ n1'

      ex2 = genericLength n0 - genericLength n0'
      n0' = case n1 of 
              [] -> reverse $ dropWhile (==0) $ reverse n0
              _  -> n0

  return $ mkFloatT s num (ex0 - ex1 + ex2)

oneOf :: [Char] -> ReadP Char 
oneOf xs = satisfy $ \x -> x `elem` xs

digit :: ReadP Char 
digit = oneOf "0123456789"

digits :: ReadP String
digits = munch1 (`elem` "0123456789")

signP :: ReadP String 
signP = option "Pos" (char '-' >> return "Neg")

mbExpn :: ReadP Integer 
mbExpn = option 0 $ do 
  s <- oneOf "eE" *> option 1 (char '-' >> return (-1))
  n <- read <$> digits 
  return (n*s)

integerP :: ReadP (String, [Integer])
integerP = liftA2 (,) signP natP
  
natP :: ReadP [Integer]
natP = map (read . pure) <$> digits

mkFloatType :: String -> Type
mkFloatType s = case lookup "" $ map (uncurry (flip (,))) $ readP_to_S floatP s of 
                 Nothing -> error $ "TypeGen.mkFloatType: could not parse " ++ show s
                 Just x  -> x 

mkFloatExp :: String -> Exp
mkFloatExp s = SigE (ConE $ mkName "FLOAT") (mkFloatType s)

float :: QuasiQuoter
float = QuasiQuoter {quoteExp = return . mkFloatExp, quoteType = return . mkFloatType}
\end{code}
