\ignore{
\begin{code}
{-# LANGUAGE 
    TypeFamilies
  , DataKinds
  , TypeOperators
  , UndecidableInstances
  , PolyKinds
  , ConstraintKinds
  , MultiParamTypeClasses
  , NullaryTypeClasses #-}

module TypeComputation.Boolean 
  ( module TypeComputation.Boolean 
  , module TypeComputation.Error 
  , module TypeComputation.Proxy
  , Constraint 
  ) where 

import TypeComputation.Error 
import TypeComputation.Proxy
import GHC.Prim (Constraint)

\end{code}
}

\section{Boolean functions and equivalence}\label{sec:Boolean}

Basic operations on type level booleans. Also some extra functions on naturals.

\subsection{Binary operators}\label{subsec:BinOps}

\begin{code}

type family (?) a b where 
  True  ? (a, b) = a
  False ? (a, b) = b

type family If (a :: Bool) (b :: k) (c :: k) :: k where 
  If True  a b = a
  If False a b = b

infixr 3 &&, ||
type family (||) (a :: Bool) (b :: Bool) :: Bool where 
  (||) True b = True
  (||) b True = True
  (||) a b    = False 

type family (&&) (a :: Bool) (b :: Bool) :: Bool where 
  (&&) True True = True
  (&&) a    b    = False 

type family Not (a :: Bool) :: Bool where 
  Not True = False
  Not False = True

type Nor a b = Not (a || b)
type Nand a b = Not (a && b)

type family Or (a :: [Bool]) :: Bool where 
  Or '[] = False
  Or (True ': xs) = True
  Or (x    ': xs) = Or xs

type family And (a :: [Bool]) :: Bool where 
   And '[] = True
   And (True ': xs) = And xs
   And (x    ': xs) = False 

\end{code}
\subsection{Equivalence relations}\label{subsec:Equivs}

Constraints are implemented generically in terms of the corresponding relation. However, they
are implementented in such a way as to produce optimal error messages. See Error~\ref{sec:Error}.
\begin{code}

infix 4 ==, /=, ==?, /=?

type family (==?) (a :: k) (b :: k) :: Bool where 
  a ==? a = True
  a ==? b = False

type (/=?) a b = Not (a ==? b)

(==?) :: a -> b -> Proxy (a ==? b)
(/=?) :: a -> b -> Proxy (a /=? b)

((/=?), (==?)) = (\_ _ -> Proxy, \_ _ -> Proxy)

type family (==) (a :: k) (b :: k) :: Constraint where  
  a == a = Always
  a == b = Equal_To a b 

type family (/=) (a :: k) (b :: k) :: Constraint where  
  a /= a = Not_Equal_To a a 
  a /= b = Always 

assertEQ  :: (a == b) => a -> b -> ()
assertNEQ :: (a /= b) => a -> b -> ()

(assertEQ, assertNEQ) = (\_ _ -> (), \_ _ -> ())

\end{code}
The type \cod{Assert} is a useful synonym: if the condition is true, \cod{Assert} is
satisfied. Otherwise it is equal to \cod{err}, which should be a constraint which is 
never satisfied. 
\begin{code}
type Assert bool err = If bool Always err

\end{code}
