
\ignore{
\begin{code}
{-# LANGUAGE 
    MultiParamTypeClasses
  , FunctionalDependencies
  , UndecidableInstances
  , FlexibleInstances
  , RankNTypes
  , FlexibleContexts
  , TypeFamilies
  , GADTs
  , ScopedTypeVariables
  , Rank2Types
  , ConstraintKinds
  , TemplateHaskell
  , TypeOperators
  , QuasiQuotes
  , DefaultSignatures 
  , StandaloneDeriving 
  , PolyKinds
  , DataKinds 
  , OverlappingInstances
  , AllowAmbiguousTypes
  #-}
module TypeComputation.FFT.Frame 
  ( module TypeComputation.FFT.Frame 
  , module TypeComputation.Numeric.FloatingPoint
  , module TypeComputation.UnitDefinitions
  ) where

import TypeComputation.Numeric.FloatingPoint
import TypeComputation.UnitDefinitions
import GHC.Prim (Constraint)

\end{code}
}

\subsection{Frames and dimensions}\label{sec:Frame}

This module contains definitions of frames. Frames are vector spaces with a base unit.
It also defines how to access arbitrary dimensions of a nested container (such as \hask{[]})
indexed by type lists.

The frame datatype \cod{F} is only used to make the writing of frames simpler.
\begin{code}

data F (s :: Symbol) = F 
instance KnownSymbol s => Show (F s) where show = symbolVal

\end{code}
Type level symbols are written as strings. Each unique string represents a unique type. No computation can be 
done with the strings, but they are an idiomatic way of creating new labels without having to 
define dummy datatypes. 

The \cod{Frame} class asserts that a type level string represents a frame. When declaring a frame, a base
unit must be associated with the frame. Optionally, a method can be defined for printing the frame.
\begin{code}

class (KnownUnit (BaseUnit frame)) => Frame frame where
  type BaseUnit frame :: * -> *

  name :: F frame -> String
  default name :: KnownSymbol frame => F frame -> String
  name = symbolVal 

\end{code}
Individual frames are used to build up more complex, multidimensional frames.
In a \cod{CompositeFrame}, each dimension also has associated with it a step size,
a number of samples, and a range unit. 
\begin{code}

type family Frames (x :: [Symbol]) :: Constraint where 
  Frames '[] = ()
  Frames (x ': xs) = (Frame x, Frames xs)

infixr 5 :>
data CompositeFrame xs where 
  F0   :: CompositeFrame '[]
  (:>) :: (F sym, NAT x, FLOAT sn y ex, Proxy (SIUnit m s kg amp mol k)) -> CompositeFrame xs -> 
          CompositeFrame ( '(sym, NAT x, FLOAT sg y ex, SIUnit m s kg amp mol k) ': xs )
 
\end{code}
The type of data stored inside a discretization is dependant on the length of the 
composite frame, the underlying base value, and the vector type used. The \cod{Dimension}
type family computes the resulting type given these things. 

\cod{DimensionPred} is similiar, but it is used to apply its argument one dimension shallower than
\cod{Dimension}.
\begin{code}

type family Dimension (n :: k) (f :: * -> *) (v :: *) :: * where 
  Dimension '[]       f a = a
  Dimension (x ': xs) f a = f (Dimension xs f a)

type family DimensionPred (n :: k) (f :: * -> *) (v :: *) :: * where 
  DimensionPred '[]       f a = a
  DimensionPred (x ': xs) f a = Dimension xs f a
 
\end{code}
The functions \cod{dimension} and \cod{dimensionPred} take a container and a composite frame
as a witness, and apply the given function at the depth specified by frame.
\cod{dimension} applies its function one dimension deeper than \cod{dimensionPred}.
\begin{code}

dimension :: (Dimension x f v ~ fv, Functor f) 
          => Proxy f -> CompositeFrame x -> (v -> v) -> fv -> fv
dimension _ F0         f a = f a
dimension p (_ :> n)   f a = fmap (dimension p n f) a 

dimensionPred :: (DimensionPred x f v ~ fv, Functor f) 
              => Proxy f -> CompositeFrame x -> (v -> v) -> fv -> fv
dimensionPred _ F0       _ a = a
dimensionPred p (_ :> x) f a = dimension p x f a 

\end{code}
A \cod{Discretization} contains a frame and the underlying data, whose type is dependant on the frame.
\begin{code}

data Discretization xs vec val where 
  Discretization :: CompositeFrame xs -> Dimension xs vec val -> Discretization xs vec val

frameOf :: Discretization frame vec val -> CompositeFrame frame
frameOf (Discretization f _) = f

dataOf :: Discretization frame vec val -> Dimension frame vec val 
dataOf (Discretization _ d) = d 

\end{code}
To save the user a lot of typing, we define type synonyms for fixed dimension discretizations up to six dimensions.
\begin{code}

sd = (F, NAT, FLOAT, Proxy)

type Discretization6D dim0 dim1 dim2 dim3 dim4 dim5 = 
     Discretization [dim0, dim1, dim2, dim3, dim4, dim5] 

discretization6D = Discretization (sd :> sd :> sd :> sd :> sd :> sd :> F0)

type Discretization5D dim0 dim1 dim2 dim3 dim4 =
     Discretization [dim0, dim1, dim2, dim3, dim4] 

discretization5D = Discretization (sd :> sd :> sd :> sd :> sd :> F0)

type Discretization4D dim0 dim1 dim2 dim3 =
     Discretization [dim0, dim1, dim2, dim3] 

discretization4D = Discretization (sd :> sd :> sd :> sd :> F0)

type Discretization3D dim0 dim1 dim2 =
     Discretization [dim0, dim1, dim2] 

discretization3D = Discretization (sd :> sd :> sd :> F0)

type Discretization2D dim0 dim1 =
     Discretization [dim0, dim1] 

discretization2D = Discretization (sd :> sd :> F0)

type Discretization1D dim0 =
     Discretization (dim0 ': '[])

discretization1D = Discretization (sd :> F0)

\end{code}
The user is responsible for defining \cod{BaseUnit}s which are consistent; 
they then must create an instance of \cod{AssertDualFrames} to guarantee that the frames are in fact dual to each other.
\cod{AssertDualFrames} will check the frames for consistency and reget invalid assertions. 
\begin{code}

type AssertDualUnits a b = Assert (a :*: b ==? Unitless) (DualUnits a b)

class (Frame a, Frame b,
       AssertDualUnits (BaseUnit a) (BaseUnit b)
      ) => AssertDualFrames a b | a -> b, b -> a 

assertDualFrames :: AssertDualFrames a b => F a -> F b
assertDualFrames = const F

\end{code}
