\ignore{
\begin{code}
{-# LANGUAGE 
    RankNTypes 
  , ScopedTypeVariables 
  , EmptyDataDecls 
  , MultiParamTypeClasses
  , FunctionalDependencies 
  , FlexibleInstances 
  , UndecidableInstances 
  , OverlappingInstances 
  , TypeFamilies  
  , FlexibleContexts 
  , DeriveDataTypeable
  , ConstraintKinds
  , TypeOperators
  , GADTs
  , PolyKinds 
  , ScopedTypeVariables
  , DataKinds
  , KindSignatures
  #-}

module TypeComputation.Numeric.FloatingPoint 
  ( module TypeComputation.Numeric.FloatingPoint
  , module TypeComputation.Numeric.Integer
  , module TypeComputation.TH.TypeGen
  ) where

import TypeComputation.Numeric.Integer
import TypeComputation.List
import TypeComputation.TH.TypeGen

import qualified GHC.TypeLits as TL 
import Data.Ratio ((%))

\end{code}
}

\subsection{Floating point}\label{sec:FloatingPoint}
This module implements arithmetic for decimal numbers in the form of floating point.

Floats are represented exactly in decimal form.
The datatype \cod{E_K} represents an exponent - either positive or negative.
It is lifted to the kind level and never used as data.
\begin{code}
data E_K = E Nat | E_ Nat  

data FLOAT (s :: Sign) (n :: Nat) (exp :: E_K) = FLOAT 

type family NFLOAT (s :: Sign) (n :: Nat) (exp :: E_K) where 
  NFLOAT s n exp = NormalizeFloat (FLOAT s n exp)

class KnownExp (e :: E_K) where expToInt :: Proxy e -> Integer 
instance KnownNat n => KnownExp (E   n) where expToInt _ =  natVal (Proxy :: Proxy n)
instance KnownNat n => KnownExp (E_  n) where expToInt _ = -natVal (Proxy :: Proxy n)

instance (KnownExp ex, KnownNat n, SignVal s) => GenericNumber (FLOAT s n ex) where 
  numToRational _ = if ex < 0 then (n * s) % (10^(-ex)) else (n * s * 10^ex) % 1 
    where n = natVal (Proxy :: Proxy n)
          s = signVal (Proxy :: Proxy s)
          ex = fromInteger (expToInt (Proxy :: Proxy ex))

  numToFloating _ = fromInteger (n * s) * (10**ex)
    where n = natVal (Proxy :: Proxy n)
          s = signVal (Proxy :: Proxy s)
          ex = fromInteger (expToInt (Proxy :: Proxy ex))

instance (KnownExp ex, KnownNat n, SignVal s) => Show (FLOAT s n ex) where 
  show _ = showAsFloat (n * s) ex
    where n = natVal (Proxy :: Proxy n)
          s = signVal (Proxy :: Proxy s)
          ex = fromInteger (expToInt (Proxy :: Proxy ex))

floatVal :: forall proxy s n e . (KnownNat n, KnownExp e, SignVal s) => proxy s n e -> Rational 
floatVal _ = numToRational (FLOAT :: FLOAT s n e)

\end{code}
Helper functions for displaying floats. We don't use the \hask{Show} instance for \hask{Float}
because we would lose precision.
\begin{code}
showAsFloat :: Integer -> Integer -> String
showAsFloat mant expnt = 
  (if abs (fromInteger expnt + numDigs mant - 1) > 5 
    then showAsFloatExp 
    else showAsFloatPoint) mant expnt
     where numDigs = length . takeWhile (>0) . iterate (`div`10)


showAsFloatPoint :: Integer -> Integer -> String 
showAsFloatPoint mant expnt' = 
 (if mant < 0 then "-" else "") ++ whole ++ "." ++ frac
  where  mant' = show $ abs mant 
         expnt = fromInteger expnt'
         (whole,frac) =  if expnt < 0 
                         then splitAt (length mant' + expnt) mant'
                         else (mant' ++ replicate expnt '0', "0")


showAsFloatExp :: Integer -> Integer -> String
showAsFloatExp mant expnt =    (if mant < 0 then "-" else "") 
                            ++ d0:'.':ds ++ "e" 
                            ++ show (expnt + expnt')
  where  expnt' = fromIntegral $ length ds 
         (d0:ds) = show $ abs mant 

\end{code}

\section{Floating point simplification}\label{sec:FloatingPointSimp}


In order to simplify arithmetic with floats, there is a normal form. The numbers 
  \lstinline$FLOAT Pos 1 (E 3)$ and 
  \lstinline$FLOAT Pos 100000 (E_ 2)$
have the same value, but the first is in normal form: it has no trailing zeroes.
Functions on floats will expect the number in normal form but this isn't checked since it would essentially mean
renormalizing each number at every stage of the computation - which would be very expensive.


Count the number of a digits in a number by comparing to successive powers of 10.
\begin{code}
type NumDigits num = NumDigits' 1 0 False (num + 1)
type family NumDigits' pow powCount bool num where 
  NumDigits' pow powCount True num = powCount 
  NumDigits' pow powCount b    num = NumDigits' (pow*10) (powCount+1) (num <=? (pow * 10)) num 

numDigits :: NAT num -> NAT (NumDigits num)
numDigits _ = NAT

\end{code}
Split a number into a list of its digits.
This works by starting at the most significant digit, and adding the largest power of 10 less than the number
until the most significant digit is zero.

For some reason accessing the arithmetic functions indirectly through type synonyms produces strange errors.
So \cod{DigitSplit} uses the qualified type functions directly.
\begin{code}
type DigitSplit num = DigitSplit' num (num TL.<=? 9)

type family DigitSplit' num bool where 
  DigitSplit' num True  = '[num]
  DigitSplit' num False = DigitSplit'' num (NumDigits num)

type family DigitSplit'' num pow where 
  DigitSplit'' num  0    =  '[]
  DigitSplit'' num  pow  =  DigitSplit''' num 0 pow 

type family DigitSplit''' num dig pow where 
  DigitSplit''' num dig pow = 
    DigitSplit'''' num dig pow ((((dig TL.+ 1) TL.* (10 TL.^ (pow TL.- 1))) >? num))

type family DigitSplit'''' num dig pow b where 
  DigitSplit'''' num dig pow True   = 
    dig  ':  DigitSplit''  (num TL.- (dig TL.* (10 TL.^ (pow TL.- 1)))) (pow TL.- 1)

  DigitSplit'''' num dig pow False  =          
             DigitSplit'''  num (dig TL.+ 1) pow 

digitSplit :: (DigitSplit a ~ (x ': xs)) => NAT a -> Proxy (x ': xs)
digitSplit _ = Proxy 

\end{code}
Trim the leading zeros from a natural number and count the amount of zeroes trimmed.
\begin{code}
type NormalizeNat n = NormalizeNat' 0 (Reverse (DigitSplit n))

type family NormalizeNat' c xs where 
  NormalizeNat' c '[]        =  (NAT c, NAT 0) 
  NormalizeNat' c (0 ': xs)  =  NormalizeNat' (c+1) xs 
  NormalizeNat' c (x ': xs)  =  (NAT c, NAT (NormalizeNat'' (x ': xs)))

type family NormalizeNat'' xs where 
  NormalizeNat'' '[] = 0
  NormalizeNat'' (x ': xs) = x + 10 * NormalizeNat'' xs 

normalizeNat :: (NormalizeNat a ~ (NAT b, NAT c)) => NAT a -> (NAT b, NAT c)
normalizeNat _ = (NAT, NAT)

\end{code}
Normalization first converts the mantissa to a list of digits, trims leading zeroes, and then adjusts the exponent
by the amount of digits trimmed.
\begin{code}
type family NormalizeFloat fl where 
  NormalizeFloat (FLOAT sn mn ex) = NormalizeFloat' (NormalizeNat mn) sn (ExpToInt ex)


type family NormalizeFloat' nat sn ex where 
  NormalizeFloat' (NAT 0, NAT man) sn ex = FLOAT sn man (IntToExp ex)
  NormalizeFloat' (NAT m, NAT man) sn ex = FLOAT sn man (IntToExp (AddInt ex (INT Pos m)))

normalizeFloat :: (NormalizeFloat (FLOAT s0 m0 e0) ~ (FLOAT s1 m1 e1)) 
               => FLOAT s0 m0 e0 -> FLOAT s1 m1 e1
normalizeFloat _ = FLOAT 

\end{code}
Is a type a float?
\begin{code}
type family IsFloat n where 
  IsFloat (FLOAT s n e)  = Always

isFloat :: IsFloat n => n -> ()
isFloat _ = ()

\end{code}
Is a float zero?
\begin{code}
instance (n == 0) => IsZero (FLOAT s n e) 
instance (n /= 0) => NonZero (FLOAT s n e) 

\end{code}
Conversion to a float.
\begin{code}
type family ToFloat a where 
  ToFloat (FLOAT s a e)  = FLOAT s a e
  ToFloat (INT s a)      = FLOAT s a (E 0)
  ToFloat (NAT a)        = FLOAT Pos a (E 0)

\end{code}
Negation of floats.
\begin{code}
instance (NegateFloat (FLOAT s0 n0 e0) ~ (FLOAT s1 n1 e1)
         ) => Negate (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) where negate' _ = FLOAT 

type instance Negate' (FLOAT s0 n0 e0) = NegateFloat (FLOAT s0 n0 e0)

type family NegateFloat a where 
  NegateFloat (FLOAT s    0 e) = FLOAT Pos 0 e
  NegateFloat (FLOAT Pos  n e) = FLOAT Neg n e
  NegateFloat (FLOAT Neg  n e) = FLOAT Pos n e

negateFloat :: NegateFloat (FLOAT s0 n0 e0) ~ (FLOAT s1 n1 e1) => FLOAT s0 n0 e0 -> FLOAT s1 n1 e1
negateFloat _ = FLOAT 

\end{code}
Absolute value of floats.
\begin{code}
instance Abs (FLOAT s n e) (FLOAT Pos n e) where abs' _ = FLOAT 

type instance Abs' (FLOAT s n e) = FLOAT Pos n e

\end{code}
Addition of floats.
\begin{code}
instance ( a ~ (FLOAT s0 n0 e0), b ~ (FLOAT s1 n1 e1), c ~ (FLOAT s2 n2 e2)
         , (a + b) ~ c  
         , (c - b) ~ a 
         , (c - a) ~ b
         ) => Add (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) (FLOAT s2 n2 e2) where add _ _ = FLOAT 

type instance (+) (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) = 
  NormalizeFloat (AddFloat (FLOAT s0 n0 e0) (FLOAT s1 n1 e1))

\end{code}
Subtraction of floats.
\begin{code}
instance ( a ~ (FLOAT s0 n0 e0), b ~ (FLOAT s1 n1 e1), c ~ (FLOAT s2 n2 e2)
         , (a - b) ~ c 
         , (c + b) ~ a
         , (a - c) ~ b
         ) => Subtract (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) (FLOAT s2 n2 e2) 
  where sub _ _ = FLOAT 

type instance (-) (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) = 
  SubtractFloat (FLOAT s0 n0 e0) (FLOAT s1 n1 e1)

type family SubtractFloat a b where 
  SubtractFloat (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) = 
    NormalizeFloat (AddFloat (FLOAT s0 n0 e0) (NegateFloat (FLOAT s1 n1 e1))) 

\end{code}
In order to simplify addition, the exponent is converted to an \hask{INT}. This form is not used regularly because we 
consider it too verbose.
\begin{code}
type family PowFloat a where 
  PowFloat (NAT    n) = FLOAT Pos n
  PowFloat (INT s  n) = FLOAT   s n

type family ExpToInt (e :: E_K) where 
  ExpToInt (E   n) = INT Pos n
  ExpToInt (E_  n) = INT Neg n 

type family IntToExp n :: E_K where 
  IntToExp (INT Pos n) = E  n
  IntToExp (INT Neg n) = E_ n

\end{code}
Type family instances for exponents. 
\begin{code}
type instance (+) (a :: E_K) (b :: E_K) = IntToExp (ExpToInt a + ExpToInt b)
type instance (-) (a :: E_K) (b :: E_K) = IntToExp (ExpToInt a - ExpToInt b)
type instance Compare' (a :: E_K) (b :: E_K) = Compare' (ExpToInt a) (ExpToInt b)

\end{code}
The implementation of floating point addition. 
\begin{code}
type family AddFloat a b where 
  AddFloat (FLOAT s0 n0 e ) (FLOAT s1 n1 e ) = PowFloat (AddInt (INT s0 n0) (INT s1 n1)) e 
  AddFloat (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) = 
       AddFloat' (Compare' e0 e1)
                 (ExpToInt (e0 - e1))
                 (INT s0 n0) (ExpToInt e0) 
                 (INT s1 n1) (ExpToInt e1)

type family AddFloat' cmp diff am ae bm be where 
  AddFloat' GT diff am ae bm be = 
    PowFloat (am * (INT Pos 10 ^ diff) + bm) (IntToExp be)
  AddFloat' LT diff am ae bm be = 
    PowFloat (am + bm * (INT Pos 10 ^ Negate' diff)) (IntToExp ae)

\end{code}
Straightforward comparison of floats requires integer division which is very slow. 
This comparison relies on converting the floats to integers and comparing those.
Since a float might have a negative exponent, both operands must be scaled so that they are non-fractional.
\begin{code}
instance (CompareFloat (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) ~ cmp) 
  => Compare (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) cmp 

type instance Compare' (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) = 
  CompareFloat (FLOAT s0 n0 e0) (FLOAT s1 n1 e1)

compareFloat :: CompareFloat (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) ~ cmp 
             => FLOAT s0 n0 e0 -> FLOAT s1 n1 e1 -> Proxy cmp
compareFloat _ _ = Proxy 


type family CompareFloat a b where 
  CompareFloat a                 a                 = EQ 
  CompareFloat (FLOAT s0  n0 e ) (FLOAT s1  n1 e ) = CompareInt (INT s0 n0)   (INT s1 n1)
  CompareFloat (FLOAT Pos n  e0) (FLOAT Pos n  e1) = CompareInt (ExpToInt e0) (ExpToInt e1)
  CompareFloat (FLOAT Neg n  e0) (FLOAT Neg n  e1) = CompareInt (ExpToInt e1) (ExpToInt e0) 

  CompareFloat (FLOAT Pos n0 e0) (FLOAT Pos n1 e1) = 
    CompareFloat' n0 (ExpToInt e0) n1 (ExpToInt e1)
  CompareFloat (FLOAT Neg n0 e0) (FLOAT Neg n1 e1) = 
    CompareFloat' n1 (ExpToInt e1) n0 (ExpToInt e0)
  CompareFloat (FLOAT Pos n0 e0) (FLOAT Neg n1 e1) = GT
  CompareFloat (FLOAT Neg n0 e0) (FLOAT Pos n1 e1) = LT 


type family CompareFloat' n0 e0 n1 e1 where 
  CompareFloat' n0 (INT Pos e0) n1 (INT Pos e1) = 
    CompareNat (NAT (n0 * 10 ^ e0)) (NAT (n1 * 10 ^ e1))

  CompareFloat' n0 (INT Neg e0) n1 (INT Pos e1) = 
    CompareNat (NAT n0) (NAT (n1 * 10 ^ (e1+e0)))

  CompareFloat' n0 (INT Pos e0) n1 (INT Neg e1) = 
    CompareNat (NAT (n0 * 10 ^ (e1+e0))) (NAT n1)

  CompareFloat' n0 (INT Neg e0) n1 (INT Neg e1) =
    If (e0<?e1) (Compare' (NAT (n0 * 10 ^ (e1-e0))) (NAT n1))
                (Compare' (NAT n0) (NAT (n1 * 10 ^ (e0-e1))))

\end{code}
Isomorphism up to normalization - floats which represent the same number but may be denormalized are isomorphic.
Even though floats should always be normalized, a denormalization test is still implemented.
\begin{code}
type IsoFloat a b = CompareFloat a b == EQ 
type IsNormalized a = NormalizeFloat a ==? a 

\end{code}
Multiplication of floats.
\begin{code}
instance (MultFloat (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) ~ (FLOAT s2 n2 e2)
         ) => Mult (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) (FLOAT s2 n2 e2) 
  where mult _ _ = FLOAT 

type instance (*) (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) = 
  MultFloat (FLOAT s0 n0 e0) (FLOAT s1 n1 e1)

type family MultFloat a b where 
  MultFloat (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) = 
    NormalizeFloat (FLOAT (MultSign' s0 s1) (n0*n1) (e0 + e1))

multFloat :: (MultFloat a b ~ FLOAT s2 n2 e2) => a -> b -> FLOAT s2 n2 e2
multFloat _ _ = FLOAT 

\end{code}
Multiplication of non-zero floats. This requires reciprocation.
\begin{code}
instance ( a ~ (FLOAT s0 n0 e0), b ~ (FLOAT s1 n1 e1), c ~ (FLOAT s2 n2 e2)
         , NonZero a, NonZero b, NonZero c
         , (a * b) ~ c
         , (c * (RecipFloat b)) ~ a
         , (c * (RecipFloat a)) ~ b
         ) => MultNZ (FLOAT s0 n0 e0) (FLOAT s1 n1 e1) (FLOAT s2 n2 e2) where multNZ _ _ = FLOAT 

\end{code}
If $c = a/b$ is a terminating decimal then we can compute it. This relies on repeated division.
If the decimal is non-terminating then it produces an error:

\begin{fakecode}
  >:t recipFloat (FLOAT :: FLOAT Pos 3 (E 0))
  
  <interactive>:1:1:
      Occurs check: cannot construct the infinite type: t ~ 3 : t
\end{fakecode}

\begin{code}
type family RecipFloat a where 
  RecipFloat (FLOAT s 1 (E   0)) = FLOAT s 1 (E   0)
  RecipFloat (FLOAT s 1 (E   e)) = FLOAT s 1 (E_  e)
  RecipFloat (FLOAT s 1 (E_  e)) = FLOAT s 1 (E   e)
  RecipFloat (FLOAT s n e) = 
    RecipFloat' s (FracReduce 1 n) e (NumDigits (FracReduce 1 n)) (NumDigits n)


type family RecipFloat' a n e d0 d1 where 
  RecipFloat' s n (E   e) d0 d1 = 
    FLOAT s n (IntToExp (INT Neg e + INT Neg d0 + INT Neg (d1-1)))
  RecipFloat' s n (E_  e) d0 d1 = 
    FLOAT s n (IntToExp (INT Pos e + INT Neg d0 + INT Neg (d1-1)))

recipFloat :: (RecipFloat (FLOAT s0 n0 e0) ~ FLOAT s1 n1 e1) => FLOAT s0 n0 e0 -> FLOAT s1 n1 e1
recipFloat _ = FLOAT 


type family FoldDigits a where 
  FoldDigits '[] = 0
  FoldDigits (x ': xs) = x + 10 * FoldDigits xs 


type family FracReduce a b where 
  FracReduce a b = FoldDigits (Reverse (FracReduce' a b))

type family FracReduce' a b where 
  FracReduce' a b = FracReduce'' (a <? b) a b 

type family FracReduce'' cmp a b where 
  FracReduce'' True  a b = FracReduce' (10*a) b
  FracReduce'' False a b = FracReduce''' (DivModNat a b) b

type family FracReduce''' d b where 
  FracReduce''' (NAT dig, NAT    0) b = '[dig]
  FracReduce''' (NAT dig, NAT rest) b = dig ': FracReduce' rest b

recipNat :: NAT a -> NAT (FracReduce 1 a)
recipNat _ = NAT 

\end{code}
